<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Enums\Log\EmployeeTypeLogType;
use Suiterus\Adg\Models\SM\EmployeeType;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class EmployeeTypeController extends Controller
{
    use HasCustomLogs;
    
    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'title'         => 'required|min:2|unique:adg_db.employee_types,title,NULL,id,deleted_at,NULL',
            'abbreviation'  => 'required|string',
            'status'        => 'required|integer|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $employee_type = EmployeeType::create([
                'title'         => $request->title,
                'abbreviation'  => $request->abbreviation,
                'description'   => $request->description,
                'status'        => $request->status,
                'created_by'    => $request->user()->id,
                'updated_by'    => $request->user()->id
            ]);

            $this->logCustomMessage(
                'create_employee_type',
                $employee_type,
                Auth::user()->name . ' created a new Employee Type: ' . $request->title,
                $employee_type,
                EmployeeTypeLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  $employee_type->title . ' has been created.'
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [ 'There was a problem in creating the employee type record.' ],
                'message'   => $e->getMessage()
            ], 500); 
        }

    }

    public function edit(Request $request) {
        
        $validate = Validator::make($request->all(), [
            'title'         => 'required|min:2|unique:adg_db.employee_types,title,'. $request->id.',id,deleted_at,NULL',
            'status'        => 'required|integer|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $employee_type = EmployeeType::findOrFail($request->id);
            $employee_type->update([
                'title'         => $request->title,
                'abbreviation'  => $request->abbreviation,
                'description'   => $request->description,
                'status'        => $request->status,
                'updated_by'    => $request->user()->id,
            ]);

            $updated = EmployeeType::find($request->id);
            $updated->old = collect($employee_type);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_employee_type',
                $updated,
                Auth::user()->name . ' updated the Employee Type: ' . $request->title,
                $updated,
                EmployeeTypeLogType::UPDATE,
                new Activity()
            );

            $employee_type->save();
            DB::commit();
            return response()->json([
                'text'  => $request->title . ' has been updated.'
            ]);

        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ["Employee Type doesn't exist"],
                'message'   => $me->getMessage()
            ], 500);
        }
        catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in editing the employee type'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function delete(Request $request) {

        DB::beginTransaction();
        try {

            $employee_type = EmployeeType::findOrFail($request->id);
            $title = $employee_type->title;

            $this->logCustomMessage(
                'delete_employee_type',
                $employee_type,
                Auth::user()->name . ' deleted the Employee Type: ' . $title,
                $employee_type,
                EmployeeTypeLogType::DELETE,
                new Activity()
            );

            $employee_type->delete();
            DB::commit();
            return response()->json([
                'text'  => $title. ' has been deleted.'
            ]);
            
        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ["Employee type doesn't exist"],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in deleting the record'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function init_list() {
        return response()->json([
            'data'  => EmployeeType::orderBy('id', 'desc')->paginate(5)
        ]);
    }

    public function search(Request $request) {
        return response()->json([
            'data'  => EmployeeType::where('title', 'LIKE', '%'.$request->keyword.'%')->paginate(5)
        ]);
    }
}
