<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Unit;

use App\Enums\Log\DivisionLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Section;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Services\OrgStructure\Division\DivisionService;

class DivisionController extends Controller
{
    use HasCustomLogs;

    private $orgStructureService;
    private $divisionService;

    public function __construct(DivisionService $divisionService, OrgStructureService $orgStructureService)
    {
        $this->divisionService = $divisionService;
        $this->orgStructureService = $orgStructureService;
    }

    public function create_division(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'name'    =>  'required|string|unique:adg_db.divisions,name,NULL,id,deleted_at,NULL',
            'status' => 'integer|required'
        ], [
            'name.unique'   =>  $req->name . ' is already taken.',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $division = Division::create([
                'name' => $req->name,
                'description' => $req->description,
                'office_id' => $req->office_id,
                'department_id' => $req->department_id,
                'head_employee_id' => $req->head_employee_id,
                'status' => $req->status,
                'created_by' => Auth::id()
            ]);

            if (isset($req->pcfr_configuration)) {
                $this->orgStructureService->configPcr($division, $req->pcfr_configuration);
            }

            if (isset($req->minimum_overtime)) {
                $this->orgStructureService->configMinimumOvertime($division, $req->minimum_overtime);
            }

            if (isset($req->employees)) {
                $this->divisionService->assignEmployeeDivision($req->employees, $division->id);
            }

            if (isset($req->unit_ids)) {
                $this->divisionService->assignUnits($req->unit_ids, $division);
            }

            $this->update_section($req->section_ids, $division->id);

            $this->logCustomMessage(
                'create_division',
                $division,
                Auth::user()->name . ' created a new Division: ' . $req->name,
                $division,
                DivisionLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  $req->name . ' created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function init_list_division(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Division::with('sections', 'department', 'pcrs', 'units', 'overtime')->orderBy('id', 'desc')->paginate($paginate)
        ]);
    }

    public function search_division_name(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Division::where('name', 'LIKE', '%' . $request->keyword . '%')->with('sections', 'department', 'pcrs', 'units', 'overtime')->paginate($paginate)
        ]);
    }

    public function show_employees($divisionId)
    {
        return User::whereHas('actualDesignation', function ($query) use ($divisionId) {
            $query->where('division_id', $divisionId)->whereNull(['section_id']);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get();
    }

    public function edit_division(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'name'      =>  'required|string|min:4|unique:adg_db.divisions,name,' . $req->id . ',id,deleted_at,NULL',
            'status'    => 'integer|required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            try {
                $perms = Division::findOrFail($req->id);
                $perms->name = $req->name;
                $perms->description = $req->description;
                $perms->office_id = $req->office_id;
                $perms->department_id = $req->department_id;
                $perms->head_employee_id = $req->head_employee_id;
                $perms->status = $req->status;
                $perms->updated_by = Auth::id();

                Section::where('division_id', $perms->id)->update(['division_id' => null]);

                $this->update_section($req->section_ids, $perms->id);

                ActualDesignation::where('division_id', $perms->id)->whereNull('section_id')->update([
                    'office_id' => null,
                    'department_id' => null,
                    'division_id' => null,
                    'section_id' => null,
                    'unit_id' => null,
                ]);

                $updated = Division::find($req->id);
                $updated->old = collect($perms);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'update_division',
                    $updated,
                    Auth::user()->name . ' updated the Division: ' . $req->name,
                    $updated,
                    DivisionLogType::UPDATE,
                    new Activity()
                );

                $perms->save();

                if (isset($req->pcfr_configuration)) {
                    $this->orgStructureService->configPcr($perms, $req->pcfr_configuration);
                }

                if (isset($req->minimum_overtime)) {
                    $this->orgStructureService->configMinimumOvertime($perms, $req->minimum_overtime);
                }    

                if (isset($req->employees)) {
                    $this->divisionService->assignEmployeeDivision($req->employees, $perms->id);
                }

                if (isset($req->unit_ids)) {
                    $this->divisionService->assignUnits($req->unit_ids, $perms);
                }

                DB::commit();
                return response()->json([
                    'text'  =>  $req->name . ' has been updated.'
                ]);
            } catch (ME $ee) {
                DB::rollback();
                return response()->json([
                    'errors'    =>  ['Division doesnt exists.'],
                ], 400);
            }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating a Division.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete_division(Request $req)
    {
        DB::beginTransaction();
        try {
            $perm = Division::findOrFail($req->id);

            $has_section = Section::where('division_id', $perm->id)->first();
            if ($has_section) {
                return response()->json([
                    'errors' => ['Cannot delete division with sections.'],
                ], 400);
            }

            $this->logCustomMessage(
                'delete_division',
                $perm,
                Auth::user()->name . ' deleted the Division: ' . $perm->name,
                $perm,
                DivisionLogType::DELETE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Division has been deleted.'
            ]);
        } catch (ME $ee) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Division doesnt exists.'],
            ], 400);
        }
    }

    private function validate_section($section_ids)
    {
        if ($section_ids != null) {
            foreach ($section_ids as $section_id) {
                $section = Section::whereId($section_id)->first();
                if ($section->division_id != null) {
                    return true;
                }
            }
        }
        return false;
    }

    private function update_section($section_ids, $division_id)
    {
        if ($section_ids != null) {
            foreach ($section_ids as $section_id) {
                $section = Section::whereId($section_id)->first();
                $section->update([
                    'division_id' => $division_id,
                ]);
            }
        }
    }

    public function show_sections(Request $request, $division_id)
    {
        $section = Section::where('division_id', $division_id)->without(['units'])->get();

        $sections = $section->map(function ($section) use ($request) {

            $section->evaluation_score = $section->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
            })->avg('overall_rate');
            $section->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($section->evaluation_score);

            $section->units = $section->units()->get()->map(function ($unit) use ($request) {
                $unit->evaluation_score = $unit->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                    $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
                })->avg('overall_rate');
                $unit->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($unit->evaluation_score);

                return $unit;
            });

            return $section;
        });

        return $sections;
    }

    public function show_units(Request $request, $divisionId)
    {
        $units = Unit::where('division_id', $divisionId)->get()->append('organization_type');

        $units = $this->orgStructureService->evaluationScoreMapper($units, $request->start_date, $request->end_date);
        return $units;
    }
}
