<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Unit;

use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Section;
use App\Enums\Log\DepartmentLogType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Suiterus\Adg\Models\SM\Department;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Services\OrgStructure\Department\DepartmentService;

class DepartmentController extends Controller
{
    use HasCustomLogs;

    private $departmentService;
    private $orgStructureService;

    public function __construct(DepartmentService $departmentService, OrgStructureService $orgStructureService)
    {
        $this->departmentService = $departmentService;
        $this->orgStructureService = $orgStructureService;
    }

    public function create_department(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'name'    =>  'required|string|unique:adg_db.departments,name,NULL,id,deleted_at,NULL',
            'status' => 'integer|required'
        ], [
            'name.unique'   =>  $req->name . ' is already taken.',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $department = Department::create([
                'name' => $req->name,
                'office_id' => $req->office_id,
                'description' => $req->description,
                'status' => $req->status,
                'created_by' => Auth::id()
            ]);

            if (isset($req->pcfr_configuration)) {
                $this->orgStructureService->configPcr($department, $req->pcfr_configuration);
            }

            if (isset($req->minimum_overtime)) {
                $this->orgStructureService->configMinimumOvertime($department, $req->minimum_overtime);
            }

            foreach ($req->employees as $userId) {
                $this->departmentService->assignEmployeeDepartment($userId, $department->id);
            }

            $this->departmentService->assignHeadEmployee($req->head_employee_id, $department->id);

            if (isset($req->units)) {
                $this->departmentService->assignUnits($req->units, $department);
            }

            $this->update_division($req->division_ids, $department->id);

            $this->logCustomMessage(
                'create_department',
                $department,
                Auth::user()->name . ' created a new Department: ' . $req->name,
                $department,
                DepartmentLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  $req->name . ' created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function show_employees($departmentId)
    {
        return User::whereHas('actualDesignation', function ($query) use ($departmentId) {
            $query->where('department_id', $departmentId)->whereNull(['division_id']);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get();
    }

    public function init_list_department(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return Department::with(['divisions', 'office', 'pcrs', 'units', 'overtime'])->orderBy('id', 'desc')->paginate($paginate);
    }

    public function search_department_name(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return Department::where('name', 'LIKE', '%' . $request->keyword . '%')->with(['divisions', 'office', 'pcrs', 'units', 'overtime'])->paginate($paginate);
    }

    public function edit_department(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'name'      =>  'required|string|min:4|unique:adg_db.departments,name,' . $req->id . ',id,deleted_at,NULL',
            'status'    => 'integer|required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $perms = Department::findOrFail($req->id);
            $perms->name = $req->name;
            $perms->office_id = $req->office_id;
            $perms->description = $req->description;
            $perms->status = $req->status;
            $perms->updated_by = Auth::id();

            if (isset($req->pcfr_configuration)) {
                $this->orgStructureService->configPcr($perms, $req->pcfr_configuration);
            }

            if (isset($req->minimum_overtime)) {
                $this->orgStructureService->configMinimumOvertime($perms, $req->minimum_overtime);
            }

            Division::where('department_id', $perms->id)->update(['department_id' => null]);

            $this->update_division($req->division_ids, $perms->id);

            ActualDesignation::where('department_id', $perms->id)->whereNull('division_id')->update([
                'office_id' => null,
                'department_id' => null,
                'division_id' => null,
                'section_id' => null,
                'unit_id' => null,
            ]);

            $updated = Department::find($req->id);
            $updated->old = collect($perms);
            $updated->attributes = collect($updated);

            $this->departmentService->assignHeadEmployee($req->head_employee_id, $perms->id);

            foreach ($req->employees as $userId) {
                $this->departmentService->assignEmployeeDepartment($userId, $perms->id);
            }

            $perms->units()->update([
                'department_id' => null,
            ]);

            if (isset($req->units)) {
                $this->departmentService->assignUnits($req->units, $perms);
            }

            $this->logCustomMessage(
                'update_department',
                $updated,
                Auth::user()->name . ' updated the Department: ' . $req->name,
                $updated,
                DepartmentLogType::UPDATE,
                new Activity()
            );

            $perms->save();

            DB::commit();
            return response()->json([
                'text'  =>  $req->name . ' has been updated.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating a Department.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete_department(Request $req)
    {
        DB::beginTransaction();
        try {
            $perm = Department::findOrFail($req->id);

            $has_division = Division::where('department_id', $perm->id)->first();
            if ($has_division) {
                return response()->json([
                    'errors' => ['Cannot delete department with divisions.'],
                ], 400);
            }

            $this->logCustomMessage(
                'delete_department',
                $perm,
                Auth::user()->name . ' deleted the Department: ' . $perm->name,
                $perm,
                DepartmentLogType::DELETE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Department has been deleted.'
            ]);
        } catch (ME $ee) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Department doesnt exists.'],
            ], 400);
        }
    }

    private function validate_division($division_ids)
    {
        if ($division_ids != null) {
            foreach ($division_ids as $division_id) {
                $division = Division::whereId($division_id)->first();
                if ($division->department_id != null) {
                    return true;
                }
            }
        }
        return false;
    }

    private function update_division($division_ids, $department_id)
    {
        if ($division_ids != null) {
            foreach ($division_ids as $division_id) {
                $division = Division::whereId($division_id)->first();
                $division->update([
                    'department_id' => $department_id,
                ]);
            }
        }
    }

    public function show_divisions(Request $request, $department_id)
    {
        $divisions = Division::where('department_id', $department_id)->without(['sections'])->with(['units'])->get();

        return $divisions = $divisions->map(function ($division) use ($request) {
            $division->evaluation_score = $division->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
            })->avg('overall_rate');
            $division->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($division->evaluation_score);

            $division->sections = $division->sections()->without(['units'])->get()->map(function ($section) use ($request) {
                $section->evaluation_score = $section->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                    $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
                })->avg('overall_rate');
                $section->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($section->evaluation_score);
                return $section;
            });

            return $division;
        });
    }

    public function show_units(Request $request, $department_id)
    {
        $units = Unit::where('department_id', $department_id)->get()->append('organization_type');

        $units = $this->orgStructureService->evaluationScoreMapper($units, $request->start_date, $request->end_date);
        return $units;
    }

    public function show_division_section_and_unit($department_id)
    {
        $divisions = Division::where('department_id', $department_id)->without(['sections'])->get()->append('organization_type');

        $sections = Section::whereIn('division_id', $divisions->pluck('id')->toArray())->without(['units'])->get()->append('organization_type');

        $units = Unit::whereIn('section_id', $sections->pluck('id')->toArray())->orWhere('department_id', $department_id)->get()->append('organization_type');

        return [...$divisions, ...$sections, ...$units];
    }
}
