<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Enums\ContributionType;
use App\Enums\Status;
use App\Enums\Log\ContributionLogType;
use App\Traits\Logs\HasCustomLogs;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SM\Contribution;
use Suiterus\Adg\Models\SM\EmployeeContribution;
use App\Http\Controllers\Controller;
use App\Models\User;
use Suiterus\Adg\Models\Payroll\PayrollEmployee;


class ContributionController extends Controller
{
    use HasCustomLogs;

    private $db;
    function __construct(){
        $this->db = DB::connection('adg_db');
    }

    public function index()
    {
        try {

            $philhealth = Contribution::where([['status', Status::ACTIVE],['type', ContributionType::PHILHEALTH]])->without(['employeeContribution'])->first();
            $gsis = Contribution::where([['status', Status::ACTIVE],['type', ContributionType::GSIS]])->without(['employeeContribution'])->first();
            $pagibig = Contribution::where([['status', Status::ACTIVE],['type', ContributionType::PAGIBIG]])->without(['employeeContribution'])->first();

            $data = [
                'philheatlh' => $philhealth,
                'gsis' => $gsis,
                'pagibig' => $pagibig
            ];

            return response()->json([
                'data' => $data
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function show(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data' => Contribution::where('type', $request->type)->orderBy('status', 'asc')->paginate($paginate)
        ]);
    }

    public function store(Request $request)
    {
       $validate = Validator::make($request->all(), [
            'title'  =>  'required',
            'type'  =>  'required|in:1,2,3',
            'employee_share'  => 'required',
            'company_share' =>  'required',
            'effectivity_date'  =>  'required',
            'status'    =>  'required:numeric'
       ]);

       if ($validate->fails()) {
            return response()->json([
                'errors'    =>  $validate->errors()
            ], 400);
       }

       $this->db->beginTransaction();
       try {

            $contribution = Contribution::create([
                'name'  =>  $request->title,
                'type'  =>  $request->type,
                'employee_share'    =>  $request->employee_share,
                'company_share' => $request->company_share,
                'savings_rate'  =>  $request->savings_rate,
                'effectivity_date'  =>  $request->effectivity_date,
                'status'    =>  $request->status,
                'description'   =>  $request->description,
                'created_by'    =>  Auth::id()
            ]);

            foreach($request->users as $user){
                if ($contribution->type == ContributionType::GSIS){
                    EmployeeContribution::updateOrCreate(['user_id' => $user['id'] ,'contribution_id' => $contribution->id],[
                        'user_id'   =>  $user['id'],
                        'ecip'      =>  $user['ecip']
                    ]);
                }else if ($contribution->type == ContributionType::PAGIBIG){
                    EmployeeContribution::updateOrCreate(['user_id' => $user['id'], 'contribution_id' => $contribution->id],[
                        'user_id'   =>  $user['id'],
                        'amount'    =>  $user['amount']
                    ]);
                }
            }

            $this->logCustomMessage(
                'create_contribution',
                $contribution,
                Auth::user()->name . ' created a new contribution: ' . $contribution->name,
                $contribution,
                ContributionLogType::CREATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);

       } catch(Exception $e) {
        $this->db->rollBack();
        return response()->json([
            'errors'    => [__('responses.exception')],
            'message'   => $e->getMessage(),
            'line'      =>  $e->getLine()
        ], 500);
    }
    }

    public function update(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.contributions,id',
            'title' => 'required',
            'type' => 'required|in:1,2,3',
            'employee_share' => 'required',
            'company_share' => 'required',
            'effectivity_date' => 'required',
            'status' => 'required:numeric',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $data = Contribution::find($request->id);
            $oldData = $data;
            $data->update([
                'name' => $request->title,
                'type' => $request->type,
                'employee_share' => $request->employee_share,
                'company_share' => $request->company_share,
                'savings_rate'  =>  $request->savings_rate,
                'effectivity_date' => $request->effectivity_date,
                'status' => $request->status,
                'description' => $request->description,
            ]);

            $data->employeeContribution()->delete();

            foreach($request->users as $user){
                if ($data->type == ContributionType::GSIS){
                    EmployeeContribution::updateOrCreate(['user_id' => $user['id'] ,'contribution_id' => $data->id],[
                        'user_id'   =>  $user['id'],
                        'ecip'      =>  $user['ecip']
                    ]);
                }else if ($data->type == ContributionType::PAGIBIG){
                    EmployeeContribution::updateOrCreate(['user_id' => $user['id'], 'contribution_id' => $data->id],[
                        'user_id'   =>  $user['id'],
                        'amount'    =>  $user['amount']
                    ]);
                }
            }

            $data->save();

            $this->logCustomMessage(
                'update_contribution',
                $data,
                Auth::user()->name . ' updated the contribution: ' . $data->name,
                $oldData,
                ContributionLogType::UPDATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.update')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function destroy(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:' . env('ADG_DB_CONNECTION') .'.employee_contributions,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $contribution = EmployeeContribution::findOrFail($request->id);

            $this->logCustomMessage(
                'remove_employee_contribution',
                $contribution,
                Auth::user()->name . ' Removed employee contribution',
                $contribution,
                COntributionLogType::REMOVE,
                new Activity()
            );

            $contribution->delete();

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.delete')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
}
