<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Enums\Status;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Payroll\Allowance;
use Suiterus\Adg\Models\Payroll\SpecialPay;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\Salary\EmployeeSpecialPay;
use Suiterus\Adg\Models\Activity\Activity;
use App\Traits\Logs\HasCustomLogs;
use App\Enums\Log\PayrollLogType;
use Illuminate\Support\Facades\Auth;

/**
 * Setup Controller to set the allowances for the payroll
 */

class AllowanceController extends Controller
{
    use HasCustomLogs;

    private $db;

    function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request){

        $paginate = isset($request->paginate) && $request->paginate !== null ? $request->paginate : env('DEFAULT_PAGECOUNT');
        try {

            $data = Allowance::when(isset($request->search) && $request->search !== null, function($query) use ($request) {
                $keyword = '%' . $request->search . '%';
                $query->where('title', 'LIKE', $keyword);
            })->paginate($paginate);

            return response()->json([
                'data' => $data
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch all allowances list for display at forms
     */
    public function all() {
        return response()->json([
            'data'  => Allowance::where('status', Status::ACTIVE)->get()
        ]); 
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'title'     => 'required',
            'status'    => 'required|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $allowance = Allowance::create([
                'title' => $request->title,
                'description' => $request->description,
                'status'      => $request->status  
            ]);

            $this->logCustomMessage(
                'create_allowance',
                $allowance,
                Auth::user()->name . ' created an allowance: ' . $allowance->title,
                $allowance,
                PayrollLogType::CREATE_ALLOWANCE,
                new Activity()
            );

            $this->db->commit();

            return response()->json([
                'text'  => __('responses.success.create')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        return response()->json([
            'data' => Allowance::find($request->id)
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.allowances,id',
            'title'     => 'required',
            'status'    => 'required|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $allowance = Allowance::find($request->id);
            $allowance->update([
                'title' => $request->title,
                'description' => $request->description,
                'status'      => $request->status  
            ]);
            $allowance->save();

            $updatedAllowance = Allowance::find($request->id);
            $updatedAllowance->old = collect($allowance);
            $updatedAllowance->attributes = collect($updatedAllowance);
                
            $this->logCustomMessage(
                'update_allowance',
                $updatedAllowance,
                Auth::user()->name . ' updated the allowance: ' . $updatedAllowance->title,
                $updatedAllowance,
                PayrollLogType::UPDATE_ALLOWANCE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.update')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.allowances,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $allowance = Allowance::find($request->id);

            $this->logCustomMessage(
                'delete_allowance',
                $allowance,
                Auth::user()->name . ' deleted the allowance: ' . $allowance->title,
                $allowance,
                PayrollLogType::DELETE_ALLOWANCE,
                new Activity()
            );

            $allowance->delete();

            $this->db->commit();
            return response()->json([
                'text' => __('responses.success.delete')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
}
