<?php 

namespace Suiterus\Adg\Controllers\Reports;

use App\Traits\Logs\HasCustomLogs;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Enums\Log\ReportLogType;

use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Position_has_salary;
use Suiterus\Adg\Reports\TimeToHireRateReport;
use Illuminate\Pagination\LengthAwarePaginator;
use Suiterus\Adg\Reports\NewHireTurnoverRateReport;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Suiterus\Adg\Reports\CostPerHireExcel;
use Suiterus\Adg\Reports\NewHireTurnOverReport;
use Suiterus\Adg\Reports\TimeSinceLastPromotionExcelReport;
use Suiterus\Adg\Resources\NewHireTurnoverRateResponse;
use Suiterus\Adg\Services\TalentAcquisition\TalentAcquisitionService;
use Suiterus\Hrjp\Models\Position;


class TalentAcquisitionReportsController
{
    use HasCustomLogs;
    
    private $talentAcquisitionService;

    public function __construct(TalentAcquisitionService $talentAcquisitionService)
    {
        $this->talentAcquisitionService = $talentAcquisitionService;
    }

    public function newHireTurnoverRatePaginate(Request $request)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        $data = $this->talentAcquisitionService->fetchNewHireTurnoverRate($request->start_date, $request->end_date, $request->keyword);

        if ($data->isEmpty()) {
            return response()->json(['message' => 'No records found.'], 404);
        }

        $flattenedData = [];
        foreach ($data as $department) {
            foreach ($department as $division) {
                $flattenedData[] = $division;
            }
        }

        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $pagedData = array_slice($flattenedData, ($currentPage - 1) * $paginate, $paginate);
        $flattenedData = new LengthAwarePaginator($pagedData, count($flattenedData), $paginate, $currentPage);
        $flattenedData->setPath(\Request::url());

        return response()->json([
            'data' => $flattenedData
        ]);
    }

    public function newHireTurnoverRateShow(Request $request)
    {
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        $activeUsers = $this->talentAcquisitionService->countNewHireTurnoverRateActiveUsers($startDate, $endDate);
    
        $inactiveUsers = $this->talentAcquisitionService->countNewHireTurnoverRateInactiveUsers($startDate, $endDate);

        $rate = ($activeUsers == 0) ? '0' : number_format(($inactiveUsers / $activeUsers) * 100, 2) . '%';

        $dateRange = date('F', strtotime($startDate)) . ' - ' . date('F Y', strtotime($endDate));

        $data[] = [
            'dateRange' => $dateRange,
            'activeUsers' => $activeUsers,
            'inactiveUsers' => $inactiveUsers,
            'rate' => $rate
        ];

        return $data;
    } 


    public function newHireTurnoverRateDownload(Request $request) {
        $data = $this->talentAcquisitionService->fetchNewHireTurnoverRate($request->start_date, $request->end_date, $request->keyword);

        if ($data->isEmpty()) {
            return response()->json(['message' => 'No records found.'], 404);
        }

        $index = [];
        $number = 1;
        foreach ($data as $divisions) {
            foreach ($divisions as $attributes) {
                $index[] = [
                    'index' => $number++,
                    'department' => $attributes['department'],
                    'division' => $attributes['division'],
                    'inactive_users' => $attributes['inactive_users'],
                    'active_users' => $attributes['active_users'],
                    'total' => $attributes['total']
                ];
            }
        }

        $this->logCustomMessage(
            'download_report',
            null,
            Auth::user()->name . ' downloaded the New Hire Turnover Rate Report',
            null,
            ReportLogType::DOWNLOAD,
            new Activity()
        );

        return (new NewHireTurnoverRateReport($index))
            ->download(Carbon::now()->format('F-Y') . '-New-Hire-Turnover-Report.xlsx');
    }

    public function timeToHireRatePaginate(Request $request) 
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        $data = $this->talentAcquisitionService->fetchTimeToHireRate($request->start_date, $request->end_date, $request->keyword);

        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $pagedData = array_slice($data, ($currentPage - 1) * $paginate, $paginate);
        $data = new LengthAwarePaginator($pagedData, count($data), $paginate, $currentPage);
        $data->setPath(\Request::url());

        return response()->json([
            'data' => $data
        ]);
    }

    public function timeToHireRateShow(Request $request) 
    {
        $startDate = $request->start_date;
        $endDate = $request->end_date;
    
        $jobs = Position_has_salary::whereBetween('post_start', [$startDate, $endDate])->get();
        
        $totalSlots = 0;
        $totalTimeToHire = 0;
        $numIndividuals = 0;
    
        foreach ($jobs as $job) {
            $employee = $this->talentAcquisitionService->fetchTimeToHireEmployee($job->position_id, $job->item_code_id);

            if ($employee) {
                $remark = $employee->user->serviceRecord->first()->remark;
                $startDate = $employee->user->serviceRecord->first();
                $dateHired = $remark->id == 1 ? Carbon::parse($employee->date_hired)->format('F j, Y') : Carbon::parse($startDate->start_date)->format('F j, Y');
                $dateJobOpened = Carbon::parse($job->post_start);
                $slots = $job->slots;
                $timeToHire = $dateJobOpened->diffInDays($dateHired);
    
                $totalSlots += $slots;
                $totalTimeToHire += $timeToHire;
                $numIndividuals++;
            }
        }
    
        $averageTimeToHire = $numIndividuals > 0 ? $totalTimeToHire / $numIndividuals : 0;
        $dateRange = date('F', strtotime($startDate)) . ' - ' . date('F Y', strtotime($endDate));
    
        return response()->json([
            'date_range' => $dateRange,
            'total_number_of_slots' => $totalSlots,
            'total_number_of_time_to_hire_individual' => $totalTimeToHire,
            'time_to_hire_average_rate' => number_format($averageTimeToHire, 1)
        ]);
    }

    public function timeToHireRateDownload(Request $request) 
    {
        $data = $this->talentAcquisitionService->fetchTimeToHireRate($request->start_date, $request->end_date, $request->keyword);

        $index = [];

        foreach ($data as $key => $item) {
            $index[] = [
                'index' => $key + 1,
                'job_post_id' => $item['job_post_id'],
                'slots' => $item['slots'],
                'position' => $item['position'],
                'item_code' => $item['item_code'],
                'date_job_opened' => $item['date_job_opened'],
                'date_hired' => $item['date_hired'],
                'time_to_hire' => $item['time_to_hire'],
            ];
        }

        $this->logCustomMessage(
            'download_report',
            null,
            Auth::user()->name . ' downloaded the Time to Hire Rate Report',
            null,
            ReportLogType::DOWNLOAD,
            new Activity()
        );

        return (new TimeToHireRateReport($index))
            ->download(Carbon::now()->format('F-Y') . '-Time-To-Hire-Report.xlsx');
    }

    public function fetchCostPerHireReport(Request $request)
    {
        return response()->json([
            'data' => $this->talentAcquisitionService->fetchCostPerHireTable($request),
        ]);
    }

    public function createCostPerHireReport(Request $request)
    {
        $data = $this->talentAcquisitionService->createCostPerHireReport($request->position_id);
        $this->logCustomMessage('create_cost_per_hire_report', $data, Auth::user()->name . ' has created cost per hire report', $data, 'cost per hire report created', new Activity());

        return response()->json([
            'data' => $data,
        ]);
    }

    public function exportCostPerHireReport(Request $request)
    {   
        $position = Position::whereId($request->position_id)->first();
        $phs = Position_has_salary::where('position_id', $request->position_id)
            ->with('itemCodes','expenses', 'position')
            ->get();

        $this->logCustomMessage('download_cost_per_hire_report', null, Auth::user()->name . ' has downloaed cost per hire report', null, 'cost per hire report downloaded', new Activity());
        
        return Excel::download(
            new CostPerHireExcel($phs),
            'cost_per_hire_'.$position->title.'_export.xlsx'
        );
    }

    public function createTimeSinceLastPromotionReport(Request $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $data = $this->talentAcquisitionService->createTimeSinceLastPromotionReportData($request->department_id, $request->from, $request->to);
            $this->logCustomMessage('create_time_since_last_promotion_report', $data, Auth::user()->name . ' has created time since last promotion report', $data, 'time since last promotion report created', new Activity());

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'data' => $data,
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return $e->getMessage();
        }
    }

    public function fetchTimeSinceLastPromotionReport(Request $request)
    {
        return response()->json([
            'data' => $this->talentAcquisitionService->fetchTimeSinceLastPromotionTable($request),
        ]);
    }

    public function viewTimeSinceLastPromotionData(Request $request)
    {
        return response()->json([
            'data' => $this->talentAcquisitionService->fetchTimeSinceLastPromotionData($request),
        ]);
    }

    public function exportTimeSinceLastPromtionReport(Request $request)
    {   
        $data = $this->talentAcquisitionService->fetchTimeSinceLastPromotionData($request, true);

        $this->logCustomMessage('download_time_since_last_promotion_report', null, Auth::user()->name . ' has downloaed time since last promotion report', null, 'time since last promotion downloaded', new Activity());
        
        return Excel::download(
            new TimeSinceLastPromotionExcelReport($data),
            'time_since_last_promotion_export.xlsx'
        );
    }
}
