<?php

namespace Suiterus\Adg\Controllers\Reports;

use Exception;
use Validator;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use App\Enums\ReportType;
use App\Enums\ReportStatus;
use Illuminate\Http\Request;
use App\Enums\Log\ReportLogType;
use App\Enums\Payroll\PayrollType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Department;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Models\Reports\Reports;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Payroll\PayrollEmployee;
use Suiterus\Adg\Models\Payroll\PayrollEarningDeduction;

class ReportsController extends Controller
{

    use HasCustomLogs;

    //This function creates the request to generate a report.
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            if ($request->entity_type == 'department') {
                $department = Department::find($request->entity_id);
                $report = $department->reportable()->save(new Reports([
                    'report_creator'        => Auth::id(),
                    'report_type'           => $request->report_type,
                    'snapshot'              => null, // null until approved
                    'status'                => ReportStatus::PENDING,
                    'start_date'            => $request->start_date ? Carbon::createFromFormat('Y-m-d', $request->start_date)->toDateString() : null,
                    'end_date'              => $request->end_date ? Carbon::createFromFormat('Y-m-d', $request->end_date)->toDateString() : null
                ]));
            }
            elseif ($request->entity_type == 'user') {
                $user = User::find($request->entity_id);
                $report = $user->reportable()->save(new Reports([
                    'report_creator'        => Auth::id(),
                    'report_type'           => $request->report_type,
                    'snapshot'              => null, // null until approved
                    'status'                => ReportStatus::PENDING,
                    'start_date'            => $request->start_date ? Carbon::createFromFormat('Y-m-d', $request->start_date)->toDateString() : null,
                    'end_date'              => $request->end_date ? Carbon::createFromFormat('Y-m-d', $request->end_date)->toDateString() : null
                ]));
            } else {
                $report = Reports::create([
                    'report_creator'        => Auth::id(),
                    'entity_id'             => null, // null until application of polymorph
                    'entity_type'           => null,
                    'report_type'           => $request->report_type,
                    'snapshot'              => null, // null until approved
                    'status'                => ReportStatus::PENDING,
                    'start_date'            => $request->start_date ? Carbon::createFromFormat('Y-m-d', $request->start_date)->toDateString() : null,
                    'end_date'              => $request->end_date ? Carbon::createFromFormat('Y-m-d', $request->end_date)->toDateString() : null
                ]);
            }

            $logDescriptions = [
                // Add here for other types of report
                ReportType::GSIS => 'GSIS Report',
                ReportType::PHILHEALTH => 'PhilHealth Report',
                ReportType::PAGIBIG_LOAN => 'Pag-IBIG Loan Remittance Report',
                ReportType::PAGIBIG => 'Pag-IBIG Report',
                ReportType::WITHOLDING_TAX => 'Withholding Tax Report',
                ReportType::SALN => 'SALN Report',
                ReportType::HIRED_EMPLOYEE => 'Hired Employees',
                ReportType::RESIGNED_EMPLOYEE => 'Resigned Employees',
                ReportType::THIRTEENTH_MONTH => '13th and 14th Month Pay Report',
                ReportType::DAILY_RATE => 'Rate Per Day Report',
                ReportType::RATA => 'RATA Report',
               ReportType::OTHER_INCOME_DEDUCTIONS => 'Monthly Report of Other Income / Deductions',
                ReportType::SPECIAL_PAYROLL => 'Special Payroll Report'
            ];

            $customLogs = $logDescriptions[$request->report_type] ?? null;

            if ($customLogs) {
                $this->logCustomMessage(
                    'create_report',
                    $report,
                    Auth::user()->name . ' created the report ID no.'. $report->id. ' of ' . $customLogs,
                    $report,
                    ReportLogType::CREATE,
                    new Activity()
                );
            }

            $db->commit();
            return response()->json([
                'text'      => 'The report has been generated',
            ]);
        } catch (Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in generating the report'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    /**
     * The function "paginate" takes a request object and returns a paginated list of leave types based
     * on the search query and page count provided in the request.
     *
     * @param Request request The  parameter is an instance of the Request class, which
     * represents an HTTP request made to the server. It contains information about the request, such
     * as the request method, URL, headers, and any data sent with the request.
     *
     * @return a paginated collection of LeaveType models.
     */


    //This function fetches reports based on report type
    public function show(Request $request, $report_type)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        return Reports::where('report_type', $report_type)
            ->with(['creator' => function ($query) {
                $query->select('id', 'name')->without([
                    'roles',
                    'permissions',
                    'storage',
                    'employeeMetaInfo',
                    'training_latest',
                    'training_filter'
                ]);
            }])->when(isset($request->user_id) && $request->user_id != 'null', function ($query) use ($request) {
                $query->where('entity_id',  $request->user_id);
            })->when(isset($request->generated_date) && $request->generated_date != 'null', function ($query) use ($request) {
                $query->where('created_at', 'LIKE', '%' . $request->generated_date . '%');
            })->when(isset($request->report_id) && $request->report_id != 'null', function ($query) use ($request) {
                $query->where('id',  $request->report_id);
            })->when(isset($request->status) && $request->status != 'null', function ($query) use ($request) {
                $query->where('status', $request->status);
            })->when(isset($request->generated_by) && $request->generated_by != 'null', function ($query) use ($request) {
                $query->whereHas('creator', function ($query) use ($request) {
                    $query->where('name', 'LIKE', '%' . $request->generated_by . '%');
                });
            })->orderBy('created_at', 'desc')->paginate($paginate);
    }

    /**
     * Handle the request and perform the necessary actions.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Support\Facades\Storage generated report file that is stored in storage
     */
    public function download(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'reports,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        try {
            $report = Reports::where('id', $request->id)
                ->whereStatus(ReportStatus::COMPLETED)
                ->first();

            $logDescriptions = [
                // Add here for other types of report
                ReportType::GSIS => 'GSIS',
                ReportType::PHILHEALTH => 'PhilHealth',
                ReportType::PAGIBIG_LOAN => 'Pag-IBIG Loan Remittance',
                ReportType::PAGIBIG => 'Pag-IBIG',
                ReportType::WITHOLDING_TAX => 'Withholding Tax',
                ReportType::SALN => 'SALN',
                ReportType::HIRED_EMPLOYEE => 'Hired Employees',
                ReportType::RESIGNED_EMPLOYEE => 'Resigned Employees',
                ReportType::THIRTEENTH_MONTH => '13th and 14th Month Pay',
                ReportType::DAILY_RATE => 'Rate Per Day',
                ReportType::RATA => 'RATA',
                ReportType::OTHER_INCOME_DEDUCTIONS => 'Monthly Report of Other Income / Deductions',
                ReportType::SPECIAL_PAYROLL => 'Special Payroll'
            ];

            $customLogs = $logDescriptions[$report->report_type] ?? null;

            if ($customLogs) {
                $this->logCustomMessage(
                    'download_report',
                    $report,
                    Auth::user()->name . ' downloaded the report '. $report->snapshot,
                    $report,
                    ReportLogType::DOWNLOAD,
                    new Activity()
                );
            }

            return Storage::disk('generated_reports')->download($report->snapshot);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage(),
                'line'  => $e->getLine()
            ], 500);
        }
    }
}
