<?php

namespace Suiterus\Adg\Controllers\Reports;

use Exception;
use Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreRequest;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\Reports\SalnReports;
use Suiterus\Adg\Services\Reports\SalnReportService;

class EmployeeSalnReportsController extends Controller
{
    public function generate()
    {
        try {
            $sourceFilePath = public_path('saln_form/saln.doc');

            if (!file_exists($sourceFilePath)) {
                return response()->json(
                    [
                        'errors' => ['The requested file does not exist.'],
                    ],
                    404
                );
            }

            return response()->download($sourceFilePath, 'saln.doc');
        } catch (Exception $e) {
            return response()->json(
                [
                    'errors' => ['Cannot process your request. Please contact the developer. Error Code: AM-comp-0x05'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function store(StoreRequest $request)
    {

        DB::beginTransaction();

        try {
            $salnReport = $request->has('id') ? SalnReports::findOrFail($request->id) : new SalnReports();
            $attachments = $request->file('attachment');
            $salnReportService = new SalnReportService();

            $salnReportService->updateCreateSalnReports(
                $request->purpose,
                $request->id,
                $salnReport,
                $request->user_id,
                $request->net_worth,
                $request->justification,
                $attachments,
                $request->status
            );

            DB::commit();

            return response()->json([
                'text' => 'Saln report has been ' . ($request->has('id') ? 'updated' : 'created') . '.',
            ]);
        } catch (Exception $e) {
            return response()->json(
                [
                    'errors' => ['Cannot process your request. Please contact the developer. Error Code: AM-comp-0x05'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function show(Request $request, $user_id)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            return SalnReports::where('user_id', $user_id)
                ->when(isset($request->year) && $request->year != 'null', function ($query) use ($request) {
                    $query->where('created_at', 'LIKE', '%' . $request->year . '%');
                })
                ->orderBy('created_at', 'desc')
                ->paginate($paginate);
        } catch (Exception $e) {
            return response()->json(
                [
                    'errors' => ['Cannot process your request. Please contact the developer. Error Code: AM-comp-0x06'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function download(Request $request)
    {
        $file = SalnReports::where('id', $request->id)
            ->where('user_id', $request->user_id)
            ->first();

        if (!$file) {
            return response()->json(
                [
                    'message' => 'File not found',
                ],
                404
            );
        }

        $filePath = $file->attachment . '.enc';

        return response()->streamDownload(function () use ($filePath) {
            FileVault::disk('saln_attachment')->streamDecrypt($filePath);
        }, '');
    }
}
