<?php

namespace Suiterus\Adg\Controllers\Reports;

use Exception;
use Carbon\Carbon;
use App\Enums\ReportStatus;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Storage;
use Illuminate\Queue\Events\JobProcessed;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use App\Jobs\Absenteeism\EmployeeAbsenteeism;
use Suiterus\Adg\Models\Reports\AbsenceReports;
use App\Jobs\Absenteeism\SaveEmployeeAbsenteeism;
use LdapRecord\Models\Events\Saved;
use Suiterus\Adg\Services\Reports\AbsenceRateReportService;

class AbsenceRerportController extends Controller
{
    use HasCustomLogs;

    private $absenceRateReportService;

    public function __construct()
    {
        $this->absenceRateReportService = new AbsenceRateReportService;
    }

    public function index(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data' => AbsenceReports::when(isset($request->dates), function ($query) use ($request) {
                $query->whereDate('start_date', '>=', $request->dates[0])
                    ->whereDate('end_date', '<=', $request->dates[1]);
            })->paginate($paginate),
        ]);
    }

    public function fetchCardDetails(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'from' => 'required',
            'to' => 'required',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }
        return response()->json([
            'data' => $this->absenceRateReportService->getAbsentRate($request->from, $request->to),
        ]);
    }

    public function createAbsenceReport(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'from' => 'required',
            'to' => 'required',
        ]);
    
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
    
        try {
            $startDate = Carbon::parse($request->from);
            $endDate = Carbon::parse($request->to);
            
            if ($endDate->diffInDays($startDate) + 1 > 31) {
                return response()->json([
                    'errors' => ['The date range provided exceeds the allowed limit of 31 days.']
                ], 400);
            }

            $users = $this->absenceRateReportService->getUsers($request->from, $request->to);
    
            if ($users->count() == 0) {
                return response()->json([
                    'errors' => ['No users/attendance found for the specified date range.']
                ], 400);
            }
    
            $totalEmployees = $users->count();
    
            $data = AbsenceReports::create([
                'start_date' => $request->from,
                'end_date' => $request->to,
                'total_employees' => $totalEmployees,
                'working_days' => 0,
                'average_working_days' => 0,
                'days_lost' => 0,
                'absent_rate' => 0,
                'status' => ReportStatus::PENDING,
                'created_by' => Auth::id(),
            ]);
    
            foreach ($users as $user) {
                $this->absenceRateReportService->employeeAbsenteeism($data, $user);
            }

            $this->logCustomMessage(
                'create_absenteeism_report',
                $data,
                Auth::user()->name . ' downloaded an absenteeism report ',
                $data,
                'Create absenteeism report',
                new Activity()
            );

            DB::commit();
    
            return response()->json([
                'message' => 'Absenteeism Report has been created.'
            ], 200);
    
        } catch (Exception $e) {
            Log::error('Failed to create absence report: ', ['error' => $e->getMessage()]);
            DB::rollBack();
            return response()->json([
                'errors' => ['An error occurred while creating the absence report.'],
            ], 500);
        }
    }

    public function export(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.absence_rate_reports,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $data = AbsenceReports::whereId($request->id)->first();

        $this->logCustomMessage(
            'download_absenteeism_report',
            $data,
            Auth::user()->name . ' downloaded an absenteeism report ',
            $data,
            'absenteeism report downloaded',
            new Activity()
        );

        return Storage::disk('absenteeism_reports')->download($data->file_path);
    }
}
