<?php

namespace Suiterus\Adg\Controllers\Promotion;

use App\Enums\Promotion\PromotionApplicationStatus;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Exception;
use Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Promotion\PromotionApplication;
use Suiterus\Adg\Services\Promotion\PromotionService;

class PromotionController extends Controller
{
    use HasCustomLogs;

    private $promotionService;

    public function __construct(){
         $this->promotionService = new PromotionService();
    }
    public function list_applications(Request $req)
    {
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        $records = PromotionApplication::where('status', $req->status)
            ->when($req->position_id != null, function ($query) use ($req) {
                $query->where('position_id', $req->position_id);
            })
            ->when($req->item_code_id != null, function ($query) use ($req) {
                $query->whereHas('phs', function ($query) use ($req) {
                    $query->where('item_code_id', $req->item_code_id);
                });
            })
            ->when($req->step != null, function ($query) use ($req) {
                $query->whereHas('process', function ($query) use ($req) {
                    $query->where('step', $req->step);
                });
            })
            ->with('user', 'process', 'phs');

        if (is_array($req->user_id) && count($req->user_id) > 0) {
            $records->whereIn('user_id', $req->user_id);
        }

        return response()->json([
            'data' => $records->paginate($paginate)
        ]);

    }
    public function fetch_applications(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'step' => 'required',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        return PromotionApplication::whereHas('process', function ($query) use ($req) {
            $query->where('step', $req->step);
        })
            ->whereNotIn('status', [PromotionApplicationStatus::COMPLETED, PromotionApplicationStatus::FAILED])
            ->with(['user', 'process', 'phs', 'position'])
            ->paginate($paginate);
    }

    public function download_file_attachment(Request $request)
    {
        $file = $this->promotionService->downloadFileAttachment(
            $request->entityName,
            $request->id,
            $request->applicant_req_id
        );

        if (!$file) {
            return response()->json([
                'message' => 'Data not found'
            ], 404);

        }

        $filePath = $file->attachment . '.enc';

        $this->logCustomMessage('download_employee_requirement', $file, Auth::user()->name . ' downloaded an employee requirement', $file, 'Downloaded employee requirement', new Activity());

        return response()->streamDownload(function () use ($filePath) {
            FileVault::disk('promotion_requirements')->streamDecrypt($filePath);
        }, '');

    }

    public function approve_disapprove_requirement(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|integer',
            'applicant_req_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
            'status' => 'required|integer',
            'justification' => 'required|string'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $this->promotionService->updateRequirementStatus(
                $request->id,
                $request->applicant_req_id,
                $request->status,
                $request->justification,
                $request->entityName
            );

            $statusMessage = [
                1 => 'pending',
                2 => 'approved',
                3 => 'declined'
            ];

            DB::commit();
            return response()->json([
                'text' => "Attachment has been " . $statusMessage[$request->status]
            ]);

        } catch (Exception $e) {
            return response()->json([
                'errors' => ['There was a problem in disapproving the Attachment.'],
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function process_application(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'application_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
            'step' => 'required',
            'status' => 'required',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $this->promotionService->processApplication(
                $req->application_id,
                $req->step,
                $req->status,
                $req->schedule,
                $req->interviewer,
                $req->link,
                $req->description,
            );

            DB::commit();
            return response()->json([
                'text' => "Succesfully updated the promotion."
            ]);

        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'error' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    public function process_reschedule(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'application_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
            'schedule' => 'required',
            'status' => 'required',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $this->promotionService->processReschedule(
                $req->application_id, 
                $req->schedule, 
                $req->status, 
                $req->reason);

            DB::commit();
            return response()->json([
                'text' => "Succesfully process the reschedule."
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'error' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    public function generate_application_form(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'application_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }
        try {
            $filePath = $this->promotionService->generateForm($req->application_id);
            return response()->download($filePath)->deleteFileAfterSend(true);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['There was a problem in generating form.'],
                'message' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    public function upload_application_form(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'application_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
            'upload_appointment_form.*' => 'required|max:50000|mimes:pdf',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $this->promotionService->uploadForm($req->application_id, $req->file('upload_appointment_form'));

            DB::commit();
            return response()->json([
                'text' => 'Form/s has been uploaded successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'msg' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    public function download_application_form(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'application_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        try {
            $filePath = $this->promotionService->downloadForm($req->application_id);
            return response()->download($filePath)->deleteFileAfterSend(true);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['File not found'],
                'message' => $e->getMessage(),
            ], 404);
        }
    }

    public function promote_employee(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'application_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
            'item_code_id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.item_codes,id',
            'phs_id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.position_has_salaries,id',
            'corp_id' => 'nullable|exists:' . env('ADG_DB_CONNECTION') . '.corporations,id',
            'branch_id' => 'nullable|exists:' . env('ADG_DB_CONNECTION') . '.branches,id',
            'division_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.divisions,id',
            'department_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.departments,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }
        try {
            $this->promotionService->promoteEmployee(
                $req->application_id,
                $req->item_code_id,
                $req->phs_id,
                $req->corp_id,
                $req->branch_id,
                $req->division_id,
                $req->department_id
            );
            return response()->json([
                'text' => "Succesfully process the reschedule."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['There was a problem in promoting'],
                'message' => $e->getMessage(),
            ], 404);
        }
    }

}
