<?php

namespace Suiterus\Adg\Controllers\Promotion\Employee;

use App\Http\Classes\NotificationMessage;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Exception;
use Validator;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Promotion\PromotionApplication;
use Suiterus\Adg\Services\Promotion\PromotionService;
use Suiterus\Hrjp\Models\Position_has_salary as PHS;

class PromotionController extends Controller
{
    use HasCustomLogs;

    private $promotionService;

    public function __construct(){
         $this->promotionService = new PromotionService();
    }

    public function fetch_jobs(Request $req)
    {
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        $promotionService = new PromotionService();

        $data = $promotionService->fetch_all_position();

        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $pagedData = array_slice($data, ($currentPage - 1) * $paginate, $paginate);
        $data = new LengthAwarePaginator($pagedData, count($data), $paginate, $currentPage);
        $data->setPath(\Request::url());

        return response()->json([
            'data' => $data
        ]);
    }

    public function fetch_applications(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return PromotionApplication::where('user_id', Auth::id())
            ->when($req->step != null, function ($query) use ($req) {
                $query->whereHas('process', function ($query) use ($req) {
                    $query->where('step', $req->step);
                });
            })
            ->when($req->phs_id != null, function ($query) use ($req) {
                $query->where('phs_id', $req->phs_id);
            })
            ->with(['process', 'phs', 'position'])
            ->paginate($paginate);
    }

    public function view_application(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'phs_id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.position_has_salaries,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $data = PromotionApplication::where('applicant_id', Auth::id())->where('phs_id', $req->phs_id)->first();

        return response()->json([
            'data' => $data
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'position_id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.positions,id',
            'phs_id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.position_has_salaries,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {
            if (PromotionApplication::where('phs_id', $req->phs_id)->where('user_id', Auth::id())->first()) {
                return response()->json([
                    'message' => 'Application already created',
                ], 409);
            }

            $application = $this->promotionService->create_application(
                Auth::id(),
                $req->position_id,
                $req->phs_id,
                $req->position_option
            );

            $position_has_salaries = PHS::where([
                ['id', $req->phs_id],
                ['position_id', $req->position_id]
            ])->with('itemCodes')->get();

            foreach ($position_has_salaries as $position) {
                $data[] = [
                    'item_codes' => $position->itemCodes->item_code,
                    'description' => $position->description,
                    'salary' => $position->salary
                ];
            }

            DB::commit();

            $this->logCustomMessage(
                'create_promotion_application',
                $application,
                Auth::user()->name .
                ' created an applicantion for promotion',
                $application,
                'Created application promotion',
                new Activity()
            );

            return response()->json([
                'text' => 'Application created successfully!.',
                'application' => $application,
                'position_has_salary' => $data
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your Application as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function upload_file_attachment(Request $req)
    {
        $attachments = $req->file('attachment');

        foreach ($attachments as $key => $attachment) {

            $filePath = Storage::disk('promotion_requirements')->put(Auth::id(), $attachment);
            FileVault::disk('promotion_requirements')->encrypt($filePath);

            $this->promotionService->updateFileAttachment(
                $req->entityName,
                $req->id[$key],
                $req->applicant_req_id[$key],
                $filePath,
                $req->status,
            );

        }

        $application = PromotionApplication::whereId($req->applicant_req_id[0])->with('phs')->first();
        $totalIterations = count($attachments);
        if ($application) {
            NotificationMessage::notifySenderAndReceiver('Upload Requirement Files', $application->phs->hiring_manager->id, Auth::id(), null, null, null, null, $totalIterations);
        }

        $this->logCustomMessage('upload_employee_requirement', null, Auth::user()->name . ' uploaded an employee requirement', null, 'Uploaded applicant requirement', new Activity());

        return response()->json([
            'message' => 'upload successful'
        ]);
    }

    public function request_reschedule(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'application_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.promotion_application,id',
            'reason' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $this->promotionService->requestReschedule($req->application_id, $req->reason);
            DB::commit();
            return response()->json([
                'text' => "Succesfully requested a reschedule."
            ]);

        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'error' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }
}
