<?php

namespace Suiterus\Adg\Controllers\Payslip;

use App\Enums\LoansStatus;
use App\Enums\OvertimePurpose;
use App\Enums\OvertimeStatus;
use App\Enums\Payroll\PayrollStatus;
use App\Enums\Payroll\PayrollType;
use App\Enums\Status;
use App\Enums\UserHasLoansStatus;
use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Sector;
use Illuminate\Support\Facades\App;
use App\Http\Controllers\Controller;
use App\Models\UserLoan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\Payroll\PayrollEmployee;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Suiterus\Adg\Controllers\Payroll\Generator\PayrollGenerator;
use Suiterus\Adg\Models\Approvals\UserOvertime;
use Suiterus\Adg\Models\Payroll\Payroll;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Suiterus\Adg\Models\LeaveManagement\LeaveType;
use Suiterus\Adg\Models\LongevityPay;
use Suiterus\Adg\Models\Payroll\Allowance;
use Suiterus\Adg\Models\Payroll\SpecialPay;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\Salary\EmployeeSpecialPay;
use Suiterus\Adg\Models\SM\Loan;

class PayslipController extends Controller
{

    public function fetch_filter_period()
    {
        return Payroll::distinct()->where('status', PayrollStatus::DISBURSED)->get(['period_start', 'period_end']);
    }

    public function paginate_payslip(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return PayrollEmployee::where('user_id', Auth::id())->whereHas('payroll', function ($query) use ($request) {
            $query->where([
                'status' => PayrollStatus::DISBURSED
            ]);
            $query->when($request->filter_period, function ($query) use ($request) {
                $query->where([
                    'period_start' => $request['filter_period']['period_start'],
                    'period_end' => $request['filter_period']['period_end'],
                ]);
            });
        })->with(['payroll'])->paginate($paginate);
    }

    public function get_pdf(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.payroll_employees,id',
            'user_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        $payslip = [];
        $employee_breakdown = $this->fetch_employee_payroll_breakdown($request->id, $request->user_id);

        //basic info
        $employee_id = $employee_breakdown['employee_payroll']->user->employeeMetaInfo->employee_id;
        $employee_name = $employee_breakdown['employee_payroll']->user->name;
        $employee_position = $employee_breakdown['employee_payroll']->user->employeeMetaInfo->position->title;

        // allowance
        $allowances = $this->get_allowances($employee_breakdown['employee_payroll']->user->id);

        //rate monthly
        $rate = $employee_breakdown['salary'];
        // work load
        $basic_pay = $employee_breakdown['deductions']['basic_pay'];
        $overtime_pay = $this->get_overtime($employee_breakdown['employee_payroll']['user'], $rate, $employee_breakdown['payroll_period_start'], $employee_breakdown['payroll_period_end']);

        //earnings 
        $earnings = collect($employee_breakdown['earnings']);
        $pera_pay = $earnings->firstWhere('title', 'PERA Pay');
        $hazard_pay = $earnings->firstWhere('title', 'Hazard Pay');
        $holiday_pay = $earnings->firstWhere('title', 'Holiday Pay');
        $night_diff_pay = $earnings->firstWhere('title', 'Night Differential Pay');
        $refund_pay = $earnings->firstWhere('title', 'Refund Pay');

        //deductions
        $gsis = (float)str_replace(',', '', $employee_breakdown['deductions']['gsis']);
        $pagibig = (float)str_replace(',', '', $employee_breakdown['deductions']['pagibig']);
        $philhealth = (float)str_replace(',', '', $employee_breakdown['deductions']['philhealth']);
        $withholding_tax = (float)str_replace(',', '', $employee_breakdown['deductions']['withholding_tax']);
        $union_dues = (float)str_replace(',', '', $employee_breakdown['deductions']['union_dues']);
        $lwop = (float)str_replace(',', '', $employee_breakdown['lwop']['total_lwop_deduction']);

        //loans
        $loans = $this->get_loans($request->user_id, $employee_breakdown['payroll_period_start'], $employee_breakdown['payroll_period_end']);

        //totals
        $total_deductions = number_format($gsis + $pagibig + $philhealth + $withholding_tax + $union_dues + $lwop, 2);
        $gross_pay = $employee_breakdown['employee_payroll']['gross_pay'];
        $net_pay = $employee_breakdown['employee_payroll']['net_pay'];

        //longevity pay
        $longevity_pay = $this->get_longevity_pay($employee_breakdown['employee_payroll']['user'], $employee_breakdown['payroll_period_start']);

        array_push(
            $payslip,
            [
                "period_start" => $employee_breakdown['payroll_period_start'],
                "period_end" => $employee_breakdown['payroll_period_end'],
                "employee_id" => $employee_id,
                "employee_name" => $employee_name,
                "position" => $employee_position,
                "allowances" => $allowances,
                "rate_monthly" => $rate,
                "longevity_pay" => $longevity_pay,
                "basic_pay" => $basic_pay,
                "overtime_pay" => number_format($overtime_pay, 2),
                "night_diff_pay" => number_format($night_diff_pay['amount'], 2),
                "deductions" => [
                    "gsis" => number_format($gsis, 2),
                    "pagibig" => number_format($pagibig, 2),
                    "philhealth" => number_format($philhealth, 2),
                    "withholding_tax" => number_format($withholding_tax, 2),
                    "union_dues" => number_format($union_dues, 2),
                    "lwop" => number_format($lwop, 2)
                ],
                "loans" => $loans,
                "gross_pay" => $gross_pay,
                "total_deduction" => $total_deductions,
                "net_pay" => $net_pay,
                "pera_pay" => $pera_pay['amount'],
                "hazard_pay" => $hazard_pay['amount'],
                "holiday_pay" => $holiday_pay['amount'],
                "refund_pay" => $refund_pay['amount']
            ]
        );

        return $payslip;
    }

    public function fetch_employee_payroll_breakdown($id, $user_id)
    {

        try {

            $sector = Sector::where('status', 'active')->first();

            $employee_payroll = PayrollEmployee::findOrFail($id);
            $user = User::find($user_id);
            $generator = new PayrollGenerator();

            // Payroll Period Formatted
            $payroll = $employee_payroll->payroll;
            $period_start = $payroll->period_start;
            $period_end = $payroll->period_end;
            $month = date('F', strtotime($period_start));
            $start_day = date('j', strtotime($period_start));
            $end_day = date('j', strtotime($period_end));
            $year = date('Y', strtotime($period_end));
            $payroll_period = $month . ' ' . $start_day . ' - ' . $end_day . ', ' . $year;

            // User Salary
            $salary = 0;
            if ($sector->name == 'public') {
                $user_salary = $user->salary()->without('user')->with(['publicSalary'])->first();
                $salary = $user_salary->publicSalary->salaryGrade->value;
                $salary = str_replace(',', '', $salary);
            } else {
                $user_salary = $user->salary()->without('user')->with(['privateSalary'])->first();
                $salary = $user_salary->privateSalary->salary;
            }

            // Fetch LWOP details
            $lwop = $generator->fetch_lwop_details($employee_payroll, $user);

            // Fetch Deductions
            $deductions = $generator->fetch_deductions($employee_payroll, $user);

            // Fetch Pera Pay
            $earnings = $generator->fetch_allowances($employee_payroll);

            if ($payroll->type == PayrollType::SEMIMONTHLY) {
                $first_half = $employee_payroll->getRawOriginal('net_pay') / 2;
                $second_half = $employee_payroll->getRawOriginal('net_pay') - $first_half;
                $employee_payroll['first_half'] = number_format($first_half, 2);
                $employee_payroll['second_half'] = number_format($second_half, 2);
            }

            $data = array(
                'payroll_period' => $payroll_period,
                'salary'    => number_format($salary, 2),
                'employee_payroll'  => $employee_payroll,
                'lwop'  => $lwop,
                'deductions'    => $deductions,
                'earnings'  => $earnings,
                'payroll_period_start' => $period_start,
                "payroll_period_end" =>  $period_end
            );

            return $data;
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'errors'    => ['The selected employee payroll could not be found.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['The selected employee payroll could not be found.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    private function get_allowances($user_id)
    {

        $allowances = Allowance::where('status', Status::ACTIVE)->get();
        $special_pays = SpecialPay::where('status', Status::ACTIVE)->get();
        $total_earnings = 0;
        foreach ($allowances as $allowance) {
            $employee_allowance = EmployeeAllowance::where(
                [
                    ['user_id', $user_id],
                    ['allowance_id', $allowance->id]
                ]
            )->value('amount');
            $total_earnings += $employee_allowance;
            $name = $allowance['title'];
            $data[$name] = number_format($employee_allowance, 2);
        }

        foreach ($special_pays as $specialPay) {
            $employee_special_pay = EmployeeSpecialPay::where([
                ['user_id', $user_id],
                ['special_pay_id', $specialPay->id]
            ])->value('amount');
            $total_earnings += $employee_special_pay;
            $name = $specialPay['title'];
            $data[$name] = number_format($employee_special_pay, 2);
        }

        $data['total_earnings'] = $total_earnings;
        return $data;
    }

    // Compute Longevity Pay
    protected function get_longevity_pay($user, $date)
    {

        /**
         *  1. Get date hired of employee - month only
         *  2. Get month of payroll
         *  3. Check if month matches month of date hired and payroll month
         *  4. Subtract current year and date hired year
         */

        $date_hired = $user->employeeMetaInfo->date_hired;
        $current_date = date('Y-m-d', strtotime($date));

        $dh_month = substr($date_hired, 5, 2);
        $dh_year = substr($date_hired, 0, 4);

        $cd_month = substr($current_date, 5, 2);
        $cd_year = substr($current_date, 0, 4);

        $years_difference = $cd_year - $dh_year;
        if (($years_difference >= 5 && $years_difference % 5 == 0) && $cd_month == $dh_month) {

            $total = 0;
            $longevity_pays = LongevityPay::where('user_id', $user->id)->get();

            foreach ($longevity_pays as $pay) {
                $total += $pay->latest_pay;
            }

            return number_format($total, 2);
        }

        return null;
    }

    protected function get_overtime($user, $salary, $start, $end)
    {
        /**
         *  1. Get all overtime records of user within the dates
         *  2. Get the days within the records and compute for the overtime set in hours or minutes
         */

        $overtimes = UserOvertime::where('user_id', $user->id)
            ->where(function ($query) use ($start, $end) {
                $query->whereDate('start_date', '>=', $start)
                    ->whereDate('end_date', '<=', $end);
            })
            ->where('purpose', OvertimePurpose::PAY)
            ->where('status', OvertimeStatus::APPROVED)
            ->get();

        $total = 0;
        foreach ($overtimes as $overtime) {
            $start = strtotime($overtime->start_date);
            $end = strtotime($overtime->end_date);
            $diff = $end - $start;


            $start_time = strtotime($overtime->start_time);
            $end_time = strtotime($overtime->end_time);
            $time = ($end_time - $start_time) / 60;

            $days = round($diff / (60 * 60 * 24)) + 1;

            /**
             * Overtime formula:
             * ( ( Salary Basic / 22 / 8 ) / 60 ) * ot minutes total
             */

            $salary = (float)str_replace(',', '', $salary);
            $ot = (($salary / 22 / 8) / 60) * ($days * $time);
            $total += $ot;
        }

        return $total;
    }

    protected function get_loans($user_id, $start_date, $end_date)
    {   
        $loans = Loan::where('status', Status::ACTIVE)->get();
        foreach ($loans as $loan) {
            $user_loans = UserLoan::where('user_id', $user_id)
                ->where('status', UserHasLoansStatus::APPROVED)
                ->where('loan_id', $loan->id)
                ->whereBetween('from',[$start_date, $end_date])
                ->without('user')
                ->value('amount');
            $name = $loan['name'];
            $data[$name] = number_format($user_loans, 2);
        }
        return $data;
    }
}
