<?php

namespace Suiterus\Adg\Controllers\Payroll;

use App\Enums\PDS_Status;
use App\Enums\Payroll\EmployeePayrollVoidStatus;
use App\Enums\Payroll\PayrollStatus;
use App\Enums\Payroll\PayrollEntityType;
use App\Enums\Payroll\PayrollType;
use App\Enums\ScheduleBasis;
use App\Enums\Status;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use DateInterval;
use DatePeriod;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Models\Payroll\Payroll;
use Suiterus\Adg\Models\Payroll\PayrollBankSetup;
use Suiterus\Adg\Models\Payroll\PayrollEmployee;
use Suiterus\Adg\Controllers\Payroll\Generator\PayrollGenerator;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator as Paginator;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\SM\Branch;
use Suiterus\Adg\Models\SM\Department;
use Suiterus\Adg\Models\SM\Sector;
use Suiterus\Adg\Reports\PayrollJournalReport;
use Suiterus\Hrjp\Models\Position;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use App\Enums\AttendanceLegend;
use App\Models\LoanPaymentHistory;
use Suiterus\Adg\Models\SM\Suspension;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Suiterus\Adg\Models\LeaveManagement\LeaveCreditHistory;
use Suiterus\Adg\Models\SM\Holiday;
use Suiterus\Adg\Services\Payroll\PayrollService;
use App\Enums\Log\PayrollLogType;
use App\Enums\LeaveWithoutPay;

/**
 * Controller to control and generate the payroll of employees
 */

class PayrollController extends Controller
{
    use HasCustomLogs;
    private $payroll_service;

    public function __construct(PayrollService $payroll_service)
    {
        $this->payroll_service = $payroll_service;
    }


    /** Generate new payroll */
    public function generate_payroll(Request $request)
    {

        $validate = Validator::make($request->all(), []);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $officer = User::findOrFail($request->user()->id);

            $payroll_generator = new PayrollGenerator();
            $payslips = $payroll_generator->generate($officer->id, $request->type, $request->entity_type, $request->entity, $request->period_start, $request->period_end);

            $db->commit();
            return response()->json([
                'text'      => 'The payroll has generated',
                'payroll'   => $payslips
            ]);
        } catch (ModelNotFoundException $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['There payroll officer could not be found.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in generating the payroll'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function get_payroll(Request $request){
        return Payroll::without(['user'])->find($request->payroll_id);
    }

    /**
     * Approve whole payroll
     */
    public function approve_payroll(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->fails()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $payroll = Payroll::findOrFail($request->id);
            $payroll->update([
                'status'            =>  PayrollStatus::DISBURSED,
                'completion_date'   => date('Y-m-d')
            ]);

            // Get all employees with individual payrolls not voided
            $payslips = $payroll->payroll_employee()->where('void_status', EmployeePayrollVoidStatus::NOT_VOIDED)->with([
                'user' => function ($query) {
                    $query->without(['storage', 'roles', 'permissions', 'employeeMetaInfo', 'supervisor', 'exitInterview'])
                        ->with(['salary' => function ($query) {
                            $query->with(['publicSalary', 'privateSalary']);
                        }]);
                }
            ])->get();

            // Write to file and save file name
            $file = $this->write_to_file($payslips);
            $payroll->file = $file;
            $payroll->save();

            foreach ($payslips as $payroll_employee){
                $get_earning_deduction = $payroll_employee->earning_deduction;
                $status = 1; // Approved
                foreach($get_earning_deduction as $deduction){
                    LoanPaymentHistory::where('payroll_earning_deduction_id', $deduction->id)->update(['status' => $status]);
                }
            }

            $this->payroll_service->refund_leave_balance_v2($payroll->payroll_employee);

            // Deduct Leave Balance
            if (config('payroll.deduct_vl_when_generating_payroll')) {
                $this->payroll_service->deduct_leave_balance($payroll->payroll_employee, $payroll->period_start, $payroll->period_end);
            }

            $this->logCustomMessage(
                'payroll_approved',
                $payroll,
                Auth::user()->name . ' approved payroll ID no.'. $payroll->id,
                $payroll,
                PayrollLogType::APPROVE_PAYROLL,
                new Activity()
            );

            $db->commit();
            return response()->json([
                'text'  => 'The payroll has been approved'
            ]);
        } catch (ModelNotFoundException $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['The selected payroll could not be found'],
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in voiding the payroll'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Void whole payroll
     */
    public function void_payroll(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->fails()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $payroll = Payroll::findOrFail($request->id);
            $payroll->update([
                'status'    =>  PayrollStatus::VOIDED,
                'remarks'   => $request->remarks,
                'completion_date'   => date('Y-m-d')
            ]);
            $payroll->save();

            foreach ($payroll->payroll_employee as $payroll_employee){
                $get_earning_deduction = $payroll_employee->earning_deduction;
                $status = 2; // Voided
                foreach($get_earning_deduction as $deduction){
                    LoanPaymentHistory::where('payroll_earning_deduction_id', $deduction->id)->update(['status' => $status]);
                }
            }

            $this->payroll_service->deduct_leave_balance_v2($payroll->payroll_employee);

            // Refund Leave Balance
            if (config('payroll.deduct_vl_when_generating_payroll')) {
                $this->payroll_service->refund_leave_balance($payroll->payroll_employee);
            }

            $this->logCustomMessage(
                'payroll_voided',
                $payroll,
                Auth::user()->name . ' voided payroll ID no.'. $payroll->id,
                $payroll,
                PayrollLogType::VOID_PAYROLL,
                new Activity()
            );

            $db->commit();
            return response()->json([
                'text'  => 'The payroll has been voided'
            ]);
        } catch (ModelNotFoundException $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['The selected payroll could not be found'],
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in voiding the payroll'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Void specific employee payroll
     */
    public function void_employee_payroll(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.payroll_employees,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $db->commit();
            $employee_payroll = PayrollEmployee::find($request->id);
            $employee_payroll->void_status = EmployeePayrollVoidStatus::VOIDED;
            $employee_payroll->save();

            $get_earning_deduction = $employee_payroll->earning_deduction;
            $status = 2; // Voided
            foreach($get_earning_deduction as $deduction){
                LoanPaymentHistory::where('payroll_earning_deduction_id', $deduction->id)->update(['status' => $status]);
            }

            $this->logCustomMessage(
                'employee_payroll_voided',
                $employee_payroll,
                Auth::user()->name . ' voided  employee`s in payroll ID no.'. $employee_payroll->id,
                $employee_payroll,
                PayrollLogType::VOID_EMPLOYEE_PAYROLL,
                new Activity()
            );

            $db->commit();
            return response()->json([
                'text'  => "Employee's payroll has been voided"
            ]);
        } catch (Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ["There was a problem in voiding the employee's payroll."],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download payroll text file
     */
    public function download_file(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'filename'  => 'required|string'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $file = storage_path('app/payroll') . '/' . $request->filename;

            $this->logCustomMessage(
                'download_payroll',
                null,
                Auth::user()->name .' downloaded the file '. $request->filename,
                null,
                PayrollLogType::DOWNLOAD_PAYROLL,
                new Activity()
            );

            return response()->download($file);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem with downloading the file.'],
                'message'   => $e->getMessage()
            ]);
        }
    }

    /**
     * List all payrolls generated
     */
    public function list_payrolls(Request $request)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        $payrolls = Payroll::when($request->id, function ($query) use ($request) {
            $query->where('id', $request->id);
        })->when($request->date, function ($query) use ($request) {
            $query->whereDate('completion_date', $request->date);
        })->when($request->payroll_type, function ($query) use ($request) {
            $query->where('type', $request->payroll_type);
        })->when($request->month, function ($query) use ($request) {
            $nmonth = date("m", strtotime($request->month));
            $query->whereMonth('completion_date', $nmonth);
        })->when($request->year, function ($query) use ($request) {
            $query->whereYear('completion_date', $request->year);
        })->when($request->status, function ($query) use ($request) {
            $query->where('status', $request->status);
        })->when($request->payroll_officer, function ($query) use ($request) {
            $query->where('payroll_officer', $request->payroll_officer);
        })->with(['user' => function ($query) {
            $query->select('id', 'name')->without([
                'roles',
                'permissions',
                'storage',
                'employeeMetaInfo',
                'training_latest',
                'training_filter'
            ]);
        }])->orderBy('updated_at', 'desc')->paginate($paginate);

        return response()->json([
            'data'  => $payrolls
        ]);
    }

    //Fetch employee with record
    public function fetch_employee_with_record(Request $request)
    {
        return response()->json([
            'data' => User::where('status', Status::ACTIVE)->orderBy('id', 'desc')
            ->whereHas('employee_payroll_record', function ($query) use ($request) {
                $query->where('payroll_id', $request->payroll_id);
            })->with(['employeeMetaInfo' => function ($query) {
                $query->without(['corporation', 'branch', 'employee_type']);
            }])->without('storage', 'roles', 'permissions', 'supervisor')->get()
        ]);
    }

    /**
     * Fetch payroll details - list of employees with their basic pay, deductions etc.
     */
    public function fetch_payroll_details(Request $request)
    {

        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $payroll = Payroll::findOrFail($request->id);
            $records = $payroll->payroll_employee()->select('id', 'void_status', 'payroll_id', 'user_id', 'days_worked', 'stored_basic_salary', 'basic_pay', 'absences', 'lates', 'undertimes', 'gross_pay', 'deductions', 'net_pay', 'salary_grade', 'salary_step')
                    ->with(['user' => function($query) {
                        $query->select('id', 'name')->without(['roles', 'permissions', 'storage', 'training_latest', 'training_filter'])
                            ->with(['employeeMetaInfo' => function($query) {
                                $query->select('id', 'user_id', 'employee_id')
                                    ->without(['corporation', 'branch', 'division', 'department', 'employeeType']);
                            }]);
                        }])->when(isset($request->employee_name) && $request->employee_name != null, function($query) use ($request){
                            $query->whereHas('user', function($query) use($request){
                                $query->where('name' , 'LIKE', '%' . $request->employee_name . '%');
                            });
                        })->when($request->position_id != null, function($query) use ($request) {
                            $query->whereHas('user.employeeMetaInfo', function($query) use ($request) {
                                $query->where('position_id', $request->position_id);
                            });
                        })->when($request->department_id != null, function($query) use ($request) {
                            $query->whereHas('user.employeeMetaInfo', function($query) use ($request) {
                                $query->where('department_id', $request->department_id);
                            });
                        })->when($request->division_id != null, function($query) use ($request) {
                            $query->whereHas('user.employeeMetaInfo', function($query) use ($request) {
                                $query->where('division_id', $request->division_id);
                            });
                        });

                        if (is_array($request->employee_id) && count($request->employee_id) > 0) {
                            $records->whereIn('user_id', $request->employee_id);
                        }

            return response()->json([
                'data'  => $records->orderBy('user_id', 'asc')->paginate($paginate)
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'errors'    => ['The selected payroll record could not be found.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in getting the payroll.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch LWOP of a specific employee's payroll.
     * This includes both from the absences and the points deducted from the leaves when
     * the balance falls below 0
     */
    public function fetch_timekeeping_deductions(Request $request)
    {

        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        try {

            $employee_payroll = PayrollEmployee::findOrFail($request->id);

            $timekeeping =  $employee_payroll->timekeeping_deductions;
            $timekeeping_lwops = [];
            
            $generator = new PayrollGenerator();
            $user = User::whereId($employee_payroll->user_id)->first();
            foreach ($timekeeping as $attendance) {
                array_push($timekeeping_lwops, [
                    'type'      => $attendance->formatted_type,
                    'duration'  => $attendance->formatted_minutes,
                    'date'      => $attendance->formatted_date,
                    'basic_pay_deduction'    => $attendance->basic_pay_deduction,
                    'pera_deduction'    => $attendance->pera_deduction,
                    'vl_balance_remaining'  =>  $attendance->leave_balance_remaining,
                    'vl_balance_deduct' =>  $attendance->leave_balance_deduction,
                    'vl_balance_result' =>  $attendance->leave_balance_result
                ]);
            }

            $lwops = $employee_payroll->lwop_deductions;

            $leave_lwops = [];
            foreach ($lwops as $lwop) {
                $leave = $lwop->leave_without_pay->leave;
                $last_day = date('Y-m-d', strtotime("+1 day", strtotime($leave->end_date)));
                $dates = new DatePeriod(
                    new DateTime($leave->start_date),
                    new DateInterval('P1D'),
                    new DateTime($last_day)
                );
                foreach ($dates as $date) {
                    array_push($leave_lwops, [
                        'type'              => $leave->title . ' - Absent',
                        'duration'      => '8 hours, 0 minutes',
                        'date'          => $date->format('F d, Y'),
                    ]);
                }
            }

            /**
             * The data here has a manual pagination since they are from different tables
             * Tables include timekeeping_deductions and leave_without_pays, which are data/deductions
             * from attendance and leaves
             */

            $items = array_merge($leave_lwops, $timekeeping_lwops);

            $total = count($items);
            $per_page = $paginate;
            $current_page = $request->input("page") ?? 1;
            $starting_point = ($current_page * $per_page) - $per_page;

            usort($items, function ($object1, $object2) {
                return $object1['date'] > $object2['date'];
            });
            $collection = new Collection($items);
            $array = $collection->slice($starting_point, $per_page)->values();

            $array = new Paginator($array, $total, $per_page, $current_page, [
                'path'  => $request->url(),
                'query' => $request->query()
            ]);

            return response()->json([
                'data'  => $array
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'errors'    => ['The selected employee payroll record could not be found.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in getting the records.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    // Fetch Breakdown of each employee's payroll in generated payroll record
    public function fetch_employee_payroll_breakdown(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $sector = Sector::where('status', 'active')->first();

            $employee_payroll = PayrollEmployee::findOrFail($request->id);
            $user = User::find($employee_payroll->user_id);
            $generator = new PayrollGenerator();

            // Payroll Period Formatted
            $payroll = $employee_payroll->payroll;
            $period_start = $payroll->period_start;
            $period_end = $payroll->period_end;

            $start_month = date('M', strtotime($period_start));
            $end_month = date('M', strtotime($period_end));
            $start_day = date('j', strtotime($period_start));
            $end_day = date('j', strtotime($period_end));
            $year = date('Y', strtotime($period_end));


            if ($start_month !== $end_month) {
                // If the start and end months are different, display both months
                $payroll_period = $start_month . ' ' . $start_day . ' - ' . $end_month . ' ' . $end_day . ', ' . $year;
            } else {
                // If the start and end months are the same, display only one month
                $payroll_period = $start_month . ' ' . $start_day . ' - ' . $end_day . ', ' . $year;
            }

            // User Salary
            $salary = 0;
            if ($sector->name == 'public') {
                $user_salary = $user->salary()->without('user')->with(['publicSalary'])->first();
                $salary = $user_salary->publicSalary->salaryGrade->value;
                $salary = str_replace(',', '', $salary);
            } else {
                $user_salary = $user->salary()->without('user')->with(['privateSalary'])->first();
                $salary = $user_salary->privateSalary->salary;
            }

            // Fetch LWOP details
            $lwop = $generator->fetch_lwop_details($employee_payroll, $user);

            // Fetch Deductions
            $deductions = $generator->fetch_deductions($employee_payroll, $user);

            // If payroll is semi-monthly
            if ($payroll->type == PayrollType::SEMIMONTHLY || $payroll->type == PayrollType::MONTHLY) {
                $first_half = $employee_payroll->getRawOriginal('net_pay') / 2;
                $second_half = $employee_payroll->getRawOriginal('net_pay') - $first_half;
                $employee_payroll['first_half'] = number_format($first_half, 2);
                $employee_payroll['second_half'] = number_format($second_half, 2);
            }

            return response()->json([
                'data'  => array(
                    'payroll_period' => $payroll_period,
                    'salary'    => number_format($salary, 2),
                    'employee_payroll'  => $employee_payroll,
                    'lwop'  => $lwop,
                    'earnings' => $generator->fetch_allowances($employee_payroll),
                    'deductions'    => $deductions
                )
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'errors'    => ['The selected employee payroll could not be found.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['The selected employee payroll could not be found.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    /**
     * Fetch entities for payroll
     * This includes entity types position, department, all, and individual
     */
    public function fetch_entities(Request $request) {

        $validate = Validator::make($request->all(), [
            'entity_type'  => 'required|numeric',
            'payroll_type'  => 'required|numeric',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $entities = null;

            if ($request->entity_type == PayrollEntityType::BRANCH) {
                $entities = Branch::all();
            } else if ($request->entity_type == PayrollEntityType::DEPARTMENT) {
                $entities = Department::all();
            } else if ($request->entity_type == PayrollEntityType::POSITION) {
                $entities = Position::all();
            } else {
                $entities = User::select('id', 'name')->whereHas('employeeMetaInfo')->with(['employeeMetaInfo' => function ($query) {
                    $query->select('id', 'user_id', 'employee_id')->without([
                        'branch', 'corporation', 'division', 'department', 'employeeType'
                    ]);
                }])->whereDoesntHave('employeeExtraField.employeeExtraFieldColumn.updatePDS', function($query) {
                    $query->where('status', PDS_Status::ONHOLD);
                })
                ->whereHas('salary')
                ->without(['roles', 'permissions', 'storage', 'training_latest', 'training_filter'])
                ->get();
            }

            return response()->json([
                'data'  => $entities
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the entities.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }


    // Bank Setup Functions
    public function fetch_setup(){
        return response()->json([
            'data'  => PayrollBankSetup::first()
        ]);
    }

    /**
     * Create new payroll bank setup record
     */
    public function create_setup(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'bank'          => 'required',
            'delimiter'     => 'required',
            'file_format'   => 'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            PayrollBankSetup::create([
                'bank'          => $request->bank,
                'delimiter'     => $request->delimiter,
                'file_format'   => $request->file_format
            ]);

            $db->commit();

            return response()->json([]);
        } catch (Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the bank setup.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // Change Bank Setup
    public function update_setup(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'bank'          => 'required',
            'delimiter'     => 'required',
            'file_format'   => 'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $setup = PayrollBankSetup::first();
            $setup->delete();
            $setup = PayrollBankSetup::create([
                'bank'          => $request->bank,
                'delimiter'     => $request->delimiter,
                'file_format'   => $request->file_format
            ]);

            $db->commit();
            return response()->json([
                'text'  => 'Bank Setup has been updated.'
            ]);
        } catch (Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in changing the bank setup'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // Function to write payroll data of employees to file
    private function write_to_file($payslips)
    {
        $format = PayrollBankSetup::first();
        $sector = Sector::where('status', 'active')->first();
        $delimiter = $format->delimiter;
        $date = date('YmdHis');
        $file = "";
        $total_netpay = 0;
        $index = 1;
        foreach ($payslips as $payslip) {
            $account_number = $payslip->user->salary->account_number;
            $salary = $payslip->user->salary;
            $salary = $sector->name === 'public' ? $salary->publicSalary->salaryGrade->getRawOriginal('value') : $salary->privateSalary->getRawOriginal('salary');
            $net_pay = $payslip->getRawOriginal('net_pay');
            $file = $file . $account_number . $index . "\t" . number_format($salary, 2, ".", "") . "\t" . number_format($net_pay, 2, ".", "") . $delimiter . "\n";
            $index++;
            $total_netpay += $net_pay;
        }
        $file = $file . round($total_netpay, 2);
        $filename = 'sal-payroll-' . $date . '.txt';
        Storage::disk('payroll')->put($filename, $file);
        return $filename;
    }

    public function generate_journal(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->logCustomMessage(
            'download_report',
            null,
            Auth::user()->name . ' downloaded a report for payroll journal',
            null,
            'Download report',
            new Activity()
        );

        $journal = new PayrollJournalReport();
        return $journal->generate($request->id);


    }
}
