<?php

namespace Suiterus\Adg\Controllers\Payroll;

use App\Enums\Log\LongevityPayLogType;
use App\Enums\Status;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\LongevityPay;
use Suiterus\Adg\Models\Payroll\Longevity;
use Validator;


class LongevityPayController extends Controller
{
    use HasCustomLogs;

    public function index(Request $request){
        try {
            $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
            
            $total_pay = 0;

            $longevity_pays = LongevityPay::where('user_id', $request->user_id)->orderBy('date', 'desc')->withTrashed()->paginate($paginate);
            $active_pays = LongevityPay::where('user_id', $request->user_id)->where('status', Status::ACTIVE)->get();
            
            foreach($active_pays as $active_pay) {
                $total_pay += $active_pay->latest_pay;
            }
            
            return response()->json([
                'data' => $longevity_pays,
                'total_amount' => $total_pay
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request){
        $validate = Validator::make($request->all(), [
            'user_id'       =>  'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'effectivity_date'          =>  'required',
            'percentage'    =>  'required|integer',
            'status'        =>  'required:numeric'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $user = User::whereId($request->user_id)->first();
            $user_salary = $user->salary()->without('user')->with(['publicSalary'])->first();
            $salary = $user_salary->publicSalary->salaryGrade->value;
            $salary = str_replace(',', '', $salary);
        
            $longevity_pays = LongevityPay::where('user_id', $request->user_id)->where('status', Status::ACTIVE)->get();

            $total = 0;
            $total_pay = 0; 

            $rate = $request->percentage / 100;
            $latest_pay = $salary * $rate;

            foreach($longevity_pays as $pay){
                $total += $pay->latest_pay;
            }

            $total_pay += $total += $latest_pay;

            $pay = LongevityPay::create([
                'user_id'   =>  $request->user_id,
                'date'      =>  $request->effectivity_date,
                'number'    =>  count($longevity_pays) + 1,
                'salary_rate'    =>  $request->salary_rate,
                'percentage'    =>  $request->percentage,
                'latest_pay'    =>  $latest_pay,
                'total_pay'     =>  $total_pay,
                'status'     =>  $request->status
            ]);

            $this->logCustomMessage(
                'create_longevity_pay',
                $pay,
                Auth::user()->name . ' created a new longevity pay',
                $pay,
                LongevityPayLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function update(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.longevity_pays,id',
            'percentage'    =>  'required|integer',
            'effectivity_date' => 'required',
            'status' => 'required:numeric',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $data = LongevityPay::find($request->id);
            $oldData = $data;

            $longevity_pays = LongevityPay::where('user_id', $request->user_id)->where('status', Status::ACTIVE)->get();

            $total = 0;
            $total_pay = 0;

            $rate = $request->percentage / 100;
            $latest_pay = $data->salary_rate * $rate;

            foreach($longevity_pays as $pay){
                $total += $pay->latest_pay;
            }

            $total_pay += $total;

            $data->update([
                'date'      =>  $request->effectivity_date,
                'salary_rate'    =>  $request->salary_rate,
                'percentage'    =>  $request->percentage,
                'latest_pay'    =>  $latest_pay,
                'total_pay'     =>  $total_pay,
                'status'     =>  $request->status
            ]);
                
            $this->logCustomMessage(
                'update_longevity_pay',
                $data,
                Auth::user()->name . ' updated a longevity pay ',
                $oldData,
                LongevityPayLogType::UPDATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => __('responses.success.update')
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function getEmployeeLongevityPay(Request $request)
    {
        $year = Carbon::parse($request->date)->format('Y');
        $month = Carbon::parse($request->date)->format('m');

        return LongevityPay::where('user_id', $request->user_id)->whereYear('date', '<=', $year)->whereMonth('date', '<=', $month)->get();
    }
}
