<?php

namespace Suiterus\Adg\Controllers\Payroll;

use App\Http\Controllers\Controller;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Payroll\LeaveRefund;

class LeaveRefundController extends Controller
{
    public function index(Request $request) {
        try {
            $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

            return response()->json([
                'data' => LeaveRefund::when(isset($request->employee_id) && !empty($request->employee_id), function ($query) use ($request) {
                    $query->whereHas('user', function ($query) use ($request) {
                        $query->whereIn('id', $request->employee_id);
                    });
                })->when(isset($request->status) && $request->status == !null, function ($query) use ($request) {
                    $query->where('status', $request->status);
                })->when(isset($request->name), function ($query) use ($request) {
                    $query->whereHas('user', function ($query) use ($request) {
                        $query->where('name', 'LIKE', '%' . $request->name . '%');
                    });
                })->when(isset($request->dates) && $request->dates == !null, function ($query) use ($request) {
                    $query->whereHas('leaveRefundDetails', function ($query) use ($request) {
                        $query->whereDate('date', $request->dates);
                    });
                })->with([
                            'user' => function ($query) {
                                $query->select('id', 'name', 'email', 'status')
                                    ->with([
                                        'employeeMetaInfo' => function ($query) {
                                            $query->select('id', 'user_id', 'employee_id', 'division_id', 'department_id', 'position_id', 'item_code_id', 'date_hired');
                                        }
                                    ])
                                    ->without([
                                        'currentRole',
                                        'roles',
                                        'permissions',
                                        'storage',
                                        'supervisor',
                                        'user_supervisor',
                                        'exitInterview',
                                        'userProfilePicture',
                                        'profileBasicInfo',
                                    ]);
                            }, 
                            'leaveRefundDetails'
                        ])
                    ->paginate($paginate)
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }     
    }

    public function fetchLeaveRefund(Request $request) {
        $validate = Validator::make($request->all(), [
            'leave_id'       =>  'required|exists:' . env('ADG_DB_CONNECTION') . '.leaves,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        try {

            return response()->json([
                'data'  => LeaveRefund::where('leave_id', $request->leave_id)
                    ->where('user_id', $request->user_id)
                    ->with('leaveRefundDetails')
                    ->first()
            ]);

        } catch (Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine(),
                'file'      => $e->getFile()
            ], 500);
        }

    }
}
