<?php

namespace Suiterus\Adg\Controllers\Payroll\Generator;

use Carbon\Carbon;
use App\Enums\Status;
use Carbon\CarbonPeriod;
use App\Enums\LeaveAllowance;
use App\Enums\YearlyBonusDates;
use App\Enums\ContributoryTypes;
use App\Enums\ExitInterviewStatus;
use Illuminate\Support\Facades\Log;
use Suiterus\Adg\Models\LongevityPay;
use Suiterus\Adg\Models\Payroll\Allowance;
use Suiterus\Adg\Models\Payroll\Longevity;
use Suiterus\Adg\Models\Payroll\SpecialPay;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Services\Payroll\PayrollService;
use Suiterus\Adg\Models\Salary\EmployeeSpecialPay;
use Suiterus\Adg\Services\Payroll\SpecialPayService;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Services\Attendance\AttendanceService;
use Suiterus\Adg\Services\LeaveManagement\LeaveService;
use Suiterus\Adg\Models\Payroll\PayrollEarningDeduction;
use Suiterus\Adg\Models\Payroll\PayrollAllowanceBreakdown;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Suiterus\Adg\Controllers\Payroll\Generator\PayrollGenerator;
use Suiterus\Adg\Factories\Contribution\ContributionCalculatorFactory;
use Suiterus\Adg\Models\Payroll\LeaveRefund;
use Suiterus\Adg\Models\Payroll\LeaveRefundDetails;

class PublicPayrollGenerator extends PayrollGenerator
{

    /*
        Public Payroll includes PERA in the computations
        Selected Salary differs from private payroll - based on Salary Grades of current Tranche of government
    */

    private $pera;
    protected $start_date;
    protected $end_date;
    protected $work_days;
    protected $expected_weeks;
    protected $payroll;

    public function __construct($payroll)
    {
        $this->pera = 2000;
        $this->payroll = $payroll;
        $this->start_date = $payroll->period_start;
        $this->end_date = $payroll->period_end;
        $this->work_days = $this->get_work_days();
    }

    // Weekly payroll
    public function weekly($user)
    {
        $payslips = array();

            /**
             * Get the salary, attendance, leaves first for computations
             */
            $user_salary = $this->get_salary($user);
            $date_hired = $user->employeeMetaInfo->date_hired;
            $leaves = $this->get_leave_deductions($user, $this->start_date, $this->end_date);
            $attendances = $user->attendance()->whereBetween('date_in', [$this->start_date, $this->end_date]);
            $schedule = $user->employeeSchedules()->first();

            $salary = $user_salary['salary'];

            /**
             * After getting the needed data to process, call the functions to compute the
             * LWOP, deductions, and fetching the loans
             */
            $lwop = [];
            $deductions = $this->compute_deductions($date_hired, $user_salary, $user);
            $loans = $this->get_computed_loans($user);

            $rates = $this->get_hourly_rates($salary, $schedule);
            $basic_pay = $this->compute_hours_worked($rates, $attendances->sum('total'), $schedule);
            $pera_pay = $this->pera / 4;
            $holiday_pay = $this->compute_holiday($user, $salary);
            $overtime_pay = $this->compute_overtime($user, $salary);
            $allowances = $this->get_allowances($user_salary['user_salary']);

            $earnings = [
                'basic_pay'     => $basic_pay,
                'holiday_pay'   => $holiday_pay,
                'overtime_pay'  => $overtime_pay,
                'pera_pay'      => $pera_pay
            ];

            $gross_pay = $basic_pay + $pera_pay;

            $deductions['withholding_tax'] /= 4;
            $deductions['mandatory_contributions']['gsis_sss'] /= 4;
            $deductions['mandatory_contributions']['ecip'] /= 4;
            $deductions['mandatory_contributions']['philhealth'] /= 4;
            $deductions['mandatory_contributions']['pagibig'] /= 4;
            $loans['total_amount'] /= 4;

            $total_deductions = $deductions['withholding_tax'] +
                $deductions['mandatory_contributions']['gsis_sss'] +
                $deductions['mandatory_contributions']['ecip'] +
                $deductions['mandatory_contributions']['philhealth'] +
                $deductions['mandatory_contributions']['pagibig'] +
                $loans['total_amount'];

            $net_pay = $gross_pay + $allowances['total_earnings'] - $total_deductions;

            $payslip = [
                'account_number'    => $user_salary['user_salary']->account_number,
                'salary'            => $user_salary['salary'],
                'net_pay'           => $net_pay,
            ];
            array_push($payslips, $payslip);

            // Record Individual Employee Payroll
            $employee_payroll = $this->payroll->payroll_employee()->create([
                'user_id'       => $user->id,
                'days_worked'   => $attendances->count(),
                'absences'      => 0,
                'lates'         => 0,
                'undertimes'    => 0,
                'basic_pay'     => $basic_pay,
                'gross_pay'     => $gross_pay,
                'deductions'    => $total_deductions,
                'net_pay'       => $net_pay,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            /**
             * Record all of the computed earnings and deductions, together with the
             * individual payroll records of the employees
             */
            $earnings_deductions = $this->earnings_deductions($employee_payroll, $earnings, $allowances, $deductions, $loans, $lwop);
            $timekeeping_deductions = $this->timekeeping_deductions($employee_payroll, $lwop);
            $leave_deductions = $this->leave_deductions($leaves['leaves'], $employee_payroll->id);
            $this->record_employee_payroll($employee_payroll, $earnings_deductions, $timekeeping_deductions, $leave_deductions, $deductions, $user_salary);

        return $payslips;
    }

    // Monthly payroll
    public function monthly($user, $period_start, $period_end) {
        $payslips = array();

            /**
             * Get the salary, attendance, leaves first for computations
             */
            $employee_payroll = $this->payroll->payroll_employee()->create([
                'user_id'       => $user->id,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            $user_salary = $this->get_salary($user);
            $leaves = $this->get_leave_deductions($user, $this->start_date, $this->end_date);
            $date_hired = $user->employeeMetaInfo->date_hired;
            $salary = $user_salary['salary'];

            $attendance_service = new AttendanceService($user);
            $date_range = CarbonPeriod::create($this->start_date, $this->end_date);
            ($date_range->toArray());
            $count_dayOff = 0;
            foreach ($date_range as $day) {
                if ($attendance_service->isRestDay($day)) {
                    $count_dayOff += 1;
                }
            }

            $calendar_days = count($date_range);
            $working_days = $calendar_days - $count_dayOff;
            $daily_rate = $salary/$working_days;
            $attendance = $this->get_timekeeping_scheduled($employee_payroll, $user, $calendar_days, $working_days, $salary);

            //For period date range
            $period_dateRange =  CarbonPeriod::create($period_start, $period_end);
            $count_dayOffs = 0;
            foreach ($period_dateRange as $day) {
                if ($attendance_service->isRestDay($day)) {
                    $count_dayOffs += 1;
                }
            }
            $period_workingDays = count($period_dateRange) - $count_dayOffs;

            $salaryPeriod = $daily_rate * $period_workingDays;

            /**
             * After getting the needed data to process, call the functions to compute the
             * LWOP, deductions, and fetching the loans
             */
            $lwop = $this->compute_lwop($user->id, $attendance, $leaves, $salary, count($period_dateRange), $period_workingDays);
            $deductions = $this->compute_deductions($date_hired, $user_salary, $user);
            $loans = $this->get_computed_loans($user);

            $basic_pay = $salary - $lwop['basic_lwop_deduction'];
            $pera_pay = $this->pera - $lwop['pera_lwop_deduction'];
            $holiday_pay = $this->compute_holiday($employee_payroll, $user, $salary);
            $overtime_pay = $this->compute_overtime($employee_payroll, $user, $salary);
            $night_diff_pay = 0;

            // Get user attendances
            $user_attendances = $user->attendance()->where(function($query) {
                $query->whereDate('time_in', '>=', $this->start_date)
                    ->whereDate('time_out', '<=', $this->end_date);
            })->get();


            $night_diff_pay = $this->get_night_diff($employee_payroll, $user_attendances, $salary, $period_workingDays);


            // Other pays and allowances
            $hazard_pay = $this->compute_hazard_pay($user_salary);
            $longevity_pay = $this->compute_longevity_pay($user, $user_salary);
            $refund = $this->fetch_refund($user, $employee_payroll);

            $allowances = $this->get_allowances($employee_payroll, $user_salary['user_salary'], false, $attendance['absences']);

            $earnings = [
                'basic_pay'     => $basic_pay,
                'pera_pay'      => $pera_pay,
                'hazard_pay'    => $hazard_pay,
                'longevity_pay' => $longevity_pay,
            ];
            $earnings = array_merge($earnings);

            $other_pays = $night_diff_pay + $pera_pay + $hazard_pay + $holiday_pay + $overtime_pay + $longevity_pay;
            $gross_pay = $basic_pay + $other_pays + $allowances['total_earnings'];

            $total_deductions = $deductions['withholding_tax'] +
                $deductions['mandatory_contributions']['gsis_sss'] +
                $deductions['mandatory_contributions']['ecip'] +
                $deductions['mandatory_contributions']['philhealth'] +
                $deductions['mandatory_contributions']['pagibig'] +
                $loans['total_amount'];

            $has_attendance = $attendance['total_days'] > 0;

            if ($has_attendance) {
                $net_pay = $gross_pay - $total_deductions;
                $net_pay = $net_pay > 0 ? $net_pay : 0;
            }
            else {
                $net_pay = 0;
            }

            $payslip = [
                'account_number'    => $user_salary['user_salary']->account_number,
                'salary'            => $user_salary['salary'],
                'net_pay'           => $net_pay
            ];
            array_push($payslips, $payslip);

            // Record Individual Employee Payroll

            $this->payroll->payroll_employee()->where('user_id', $user->id)->update([
                'days_worked'   => $attendance['total_days'],
                'absences'      => count($lwop['absences']),
                'lates'         => count($lwop['late']),
                'undertimes'    => count($lwop['undertime']),
                'stored_basic_salary'  => $salary,
                'basic_pay'     => $basic_pay,
                'gross_pay'     => $gross_pay,
                'deductions'    => $total_deductions,
                'net_pay'       => $net_pay,
                'salary_grade' => $user_salary['user_salary']->publicSalary->salaryGrade->salary_grade,
                'salary_step'  => $user_salary['user_salary']->publicSalary->salaryGrade->step,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            /**
             * Record all of the computed earnings and deductions, together with the
             * individual payroll records of the employees
             */
            $earnings_deductions = $this->earnings_deductions($employee_payroll, $earnings, $allowances, $deductions, $loans, $lwop);
            $timekeeping_deductions = $this->timekeeping_deductions($employee_payroll, $lwop);
            $leave_deductions = $this->leave_deductions($leaves['leaves'], $employee_payroll->id);
            $this->record_employee_payroll($employee_payroll, $earnings_deductions, $timekeeping_deductions, $leave_deductions, $deductions, $user_salary);

        return $payslips;
    }

    // Semi-monthly payroll
    public function semi_monthly($user, $period_start, $period_end) {
        $payslips = array();
            $this->pera = $this->pera/2;

            /**
             * Get the salary, attendance, leaves first for computations
             */
            $employee_payroll = $this->payroll->payroll_employee()->create([
                'user_id'       => $user->id,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            $user_salary = $this->get_salary($user);
            $leaves = $this->get_leave_deductions($user, $this->start_date, $this->end_date);
            $date_hired = $user->employeeMetaInfo->date_hired;
            $salary = $user_salary['salary']/2;

            $attendance_service = new AttendanceService($user);
            $date_range = CarbonPeriod::create($this->start_date, $this->end_date);
            ($date_range->toArray());
            $count_dayOff = 0;
            foreach ($date_range as $day) {
                if ($attendance_service->isRestDay($day)) {
                    $count_dayOff += 1;
                }
            }
            $calendar_days = count($date_range);
            $working_days = $calendar_days - $count_dayOff;
            $daily_rate = $salary/$working_days;
            $attendance = $this->get_timekeeping_scheduled($employee_payroll, $user, $calendar_days, $working_days, $salary);
            //For period date range
            $period_dateRange =  CarbonPeriod::create($period_start, $period_end);
            $count_dayOffs = 0;
            foreach ($period_dateRange as $day) {
                if ($attendance_service->isRestDay($day)) {
                    $count_dayOffs += 1;
                }
            }
            $period_workingDays = count($period_dateRange) - $count_dayOffs;

            $salaryPeriod = $daily_rate * $period_workingDays;

            /**
             * After getting the needed data to process, call the functions to compute the
             * LWOP, deductions, and fetching the loans
             */
            $lwop = $this->compute_lwop($user->id, $attendance, $leaves, $salary, count($period_dateRange), $period_workingDays);
            $deductions = $this->compute_deductions($date_hired, $user_salary, $user);
            $loans = $this->get_computed_loans($user, true);

            $basic_pay = $salary - $lwop['basic_lwop_deduction'];
            $pera_pay = ($this->pera - $lwop['pera_lwop_deduction']);
            $holiday_pay = $this->compute_holiday($employee_payroll, $user, $user_salary['salary']);
            $overtime_pay = $this->compute_overtime($employee_payroll, $user, $salary);
            $allowances = $this->get_allowances($employee_payroll, $user_salary['user_salary'], true, $attendance['absences']);
            $night_diff_pay = 0;

            // Get user attendances
            $user_attendances = $user->attendance()->where(function($query) {
                $query->whereDate('time_in', '>=', $this->start_date)
                    ->whereDate('time_out', '<=', $this->end_date);
            })->get();


            $night_diff_pay = $this->get_night_diff($employee_payroll, $user_attendances, $salary, $period_workingDays);

            // Other pays and allowances
            $hazard_pay = $this->compute_hazard_pay($user_salary) /2;
            $longevity_pay = $this->compute_longevity_pay($user, $user_salary);
            $refund = $this->fetch_refund($user, $employee_payroll);

            $earnings = [
                'basic_pay'     => $basic_pay,
                'pera_pay'      => $pera_pay,
                'hazard_pay'    => $hazard_pay,
                'longevity_pay' => $longevity_pay,
            ];
            $earnings = array_merge($earnings, $allowances);

            $other_pays = $night_diff_pay + $pera_pay + $hazard_pay + $holiday_pay + $overtime_pay + $longevity_pay;
            $gross_pay = $basic_pay + $other_pays + $allowances['total_earnings'];

            $deductions['withholding_tax'] /= 2;
            $deductions['mandatory_contributions']['gsis_sss'] /= 2;
            $deductions['mandatory_contributions']['ecip'] /= 2;
            $deductions['mandatory_contributions']['philhealth'] /= 2;
            $deductions['mandatory_contributions']['pagibig'] /= 2;

            $total_deductions = $deductions['withholding_tax'] +
                $deductions['mandatory_contributions']['gsis_sss'] +
                $deductions['mandatory_contributions']['ecip'] +
                $deductions['mandatory_contributions']['philhealth'] +
                $deductions['mandatory_contributions']['pagibig'] +
                $loans['total_amount'];

            $has_attendance = $attendance['total_days'] > 0;

            if ($has_attendance) {
                $net_pay = $gross_pay - $total_deductions;
                $net_pay = $net_pay > 0 ? $net_pay : 0;
            }
            else {
                $net_pay = 0;
            }

            $payslip = [
                'user'  => $user,
                'account_number'    => $user_salary['user_salary']->account_number,
                'salary'            => $user_salary['salary'],
                'net_pay'           => $net_pay,
            ];
            array_push($payslips, $payslip);

            // Record Individual Employee Payroll

            $this->payroll->payroll_employee()->where('user_id', $user->id)->update([
                'days_worked'   => $attendance['total_days'],
                'absences'      => count($lwop['absences']),
                'lates'         => count($lwop['late']),
                'undertimes'    => count($lwop['undertime']),
                'stored_basic_salary'  => $salary,
                'basic_pay'     => $basic_pay,
                'gross_pay'     => $gross_pay,
                'deductions'    => $total_deductions,
                'net_pay'       => $net_pay,
                'salary_grade' => $user_salary['user_salary']->publicSalary->salaryGrade->salary_grade,
                'salary_step'  => $user_salary['user_salary']->publicSalary->salaryGrade->step,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            /**
             * Record all of the computed earnings and deductions, together with the
             * individual payroll records of the employees
             */
            $earnings_deductions = $this->earnings_deductions($employee_payroll, $earnings, $allowances, $deductions, $loans, $lwop, true);
            $timekeeping_deductions = $this->timekeeping_deductions($employee_payroll, $lwop);
            $leave_deductions = $this->leave_deductions($leaves['leaves'], $employee_payroll->id);
            $this->record_employee_payroll($employee_payroll, $earnings_deductions, $timekeeping_deductions, $leave_deductions, $deductions, $user_salary);

        return $payslips;
    }


    // --------------------------- PRIVATE FUNCTIONS - ONLY FOR PUBLIC PAYROLL --------------------------- //

    /**
     * Compute the LWOP
     */
    private function compute_lwop($user_id, $attendance, $leaves, $salary, $basic_lwop_days = 31, $pera_days = 22)
    {
        // Constants

        $total_late_minutes = $attendance['total_late_minutes'];
        $total_undertime_minutes = $attendance['total_undertime_minutes'];
        $total_halfday_minutes = $attendance['total_halfday_minutes'];
        $total_absents = count($attendance['absences']);
        $total_leave_points = $leaves['total_points'];

        // Vairables
        $total_lwop = 0;

        // Deductions
        $basic_pay_deduction = 0;
        $total_lwop_deduction = 0;
        $pera_pay_deduction = 0;

        // Remaining Balance for Sick and Vacation Leave
        $sl_store = 0;
        $vl_store = 0;
        $sl_remaining_balance = 0;
        $vl_remaining_balance = 0;
        $is_deducted = false;

        /**
         * Compute the LWOP
         * This will only compute if there is value or points in either the
         * lates, undertimes, absents, halfdays, and leaves
         */
        if ($total_late_minutes > 0 || $total_undertime_minutes > 0 || $total_absents > 0 || $total_halfday_minutes > 0 || $total_leave_points > 0) {

            $total_lwop += $leaves['total_points'] + $attendance['total_lwop_points'];

            $vacation_leave_balance = LeaveBalance::where('user_id', $user_id)->where('leave_type_id', '1')->first();

            if ($vacation_leave_balance) {
                $vl_remaining_balance = $vacation_leave_balance->balance - $total_lwop;

                if ($vl_remaining_balance < 0) {
                    $vl_store = abs($vl_remaining_balance);
                    $vl_remaining_balance = 0;
                    $is_deducted = true;
                }
            }

            $payroll_config = config('payroll.deduct_vl_when_generating_payroll');

            if ($is_deducted || $payroll_config) {
                $total_lwop = $payroll_config ? $vl_store : $total_lwop;
            }

            $basic_pay_deduction = ($salary / $basic_lwop_days) * $total_lwop;
            $pera_pay_deduction = ($this->pera / $pera_days) * $total_lwop;
            $total_lwop_deduction = $basic_pay_deduction + $pera_pay_deduction;
        }

        return array(
            'late'     => [
                'records'      => $attendance['lates'],
                'total_minutes'     => $total_late_minutes,
            ],
            'undertime' => [
                'records'      => $attendance['undertimes'],
                'total_minutes'     => $total_undertime_minutes
            ],
            'half_days' => [
                'records'      => $attendance['half_days'],
                'total_minutes'     => $total_halfday_minutes
            ],
            'absences'  => [
                'records'       => $attendance['absences'],
                'total_minutes' => $attendance['total_absent_minutes']
            ],
            'leaves'    => [
                'records'                => $leaves['leaves'],
            ],
            'total_lwop_minutes'    => $total_late_minutes + $total_undertime_minutes + $attendance['total_absent_minutes'],
            'total_lwop_points'     => $total_lwop,
            'basic_lwop_deduction'  => $basic_pay_deduction,
            'pera_lwop_deduction'   => $pera_pay_deduction,
            'total_deduction'       => $total_lwop_deduction,
        );
    }

    private function getMandatedDeductions($user_id, $salary)
    {
        $calculator_GSIS = ContributionCalculatorFactory::create(ContributoryTypes::GSIS);
        $calculator_PHILHEALTH = ContributionCalculatorFactory::create(ContributoryTypes::PHILHEALTH);
        $calculator_PAGIBIG = ContributionCalculatorFactory::create(ContributoryTypes::PAGIBIG);

        $gsis_contribution = $calculator_GSIS->compute($user_id, $salary, $this->start_date);
        $philhealth_contribution = $calculator_PHILHEALTH->compute($user_id, $salary, $this->start_date);
        $pagibig_contribution = $calculator_PAGIBIG->compute($user_id, $salary, $this->start_date);

        return [
            'gsis_share' => $gsis_contribution['personal_share'],
            'ecip_share' => $gsis_contribution['employee_contribution'],
            'philhealth_share' => $philhealth_contribution['personal_share'],
            'pagibig_share' => $pagibig_contribution['personal_share']
        ];
    }
    /**
     * Compute the deductions - mandatory
     */
    private function compute_deductions($date_hired, $salary, $user)
    {
        $tax_option = $salary['user_salary']->withholding_tax_option;
        $salary = $salary['salary'];

        $mandated_deductions = $this->getMandatedDeductions($user->id, $salary);
        $gsis = $mandated_deductions['gsis_share'];
        $gsis_ecip = $mandated_deductions['ecip_share'];
        $philhealth = $mandated_deductions['philhealth_share'];
        $pagibig = $mandated_deductions['pagibig_share'];

        // TAXABLE BASIC SALARY COMPUTATION
        $annual_salary = $salary * 12;
        $annual_gsis = $gsis * 12;
        $annual_ecip = $gsis_ecip * 12;
        $annual_philhealth = $philhealth * 12;
        $annual_pagibig = $pagibig * 12;

        $employee_union_dues = 400.00;

        $taxable_basic_salary = $annual_salary - ($annual_gsis + $annual_ecip + $annual_philhealth + $annual_pagibig + $employee_union_dues);

        // 13th MONTH PAY & OTHER BENEFITS COMPUTATION

        /**
         * Computations can be found on the provided excel sheet
         */
        $mid_year_bonus = $this->getYearlyBonus($date_hired, YearlyBonusDates::getMidYearDate(), $salary, $user);
        $year_end_bonus = $this->getYearlyBonus($date_hired, YearlyBonusDates::getYearEndDate(), $salary, $user);

        $pei = 5000.00;
        $taxable_13th_month = $mid_year_bonus + $year_end_bonus + $pei;

        if ($taxable_13th_month > 90000) {
            $taxable_13th_month -= 90000;
        }

        // TOTAL TAXABLE COMPENSATION
        $total_taxable_compensation = $taxable_basic_salary + $taxable_13th_month;
        $annual_tax = 0;
        // TAX BRACKET
        $annual_tax = $this->get_annual_tax($total_taxable_compensation);

        $tax_exemption = $annual_tax > 0 ? 2 : 1;

        // Withholding Tax Options
        // Option A or 1 - Not Included in Mid-year and Year-end bonus
        // Option B or 2 - Included in Mid-year and Year-end bonus - Divisor 14

        $withholding_tax = 0;
        if ($total_taxable_compensation > 250000) {
            $divisor = $tax_option == 1 ? 12 : 14;
            $withholding_tax = $annual_tax / $divisor;
        }

        return array(
            'mandatory_contributions' => [
                'gsis_sss'          => $gsis,
                'ecip'              => $gsis_ecip,
                'philhealth'        => $philhealth,
                'pagibig'           => $pagibig,
                'union_dues'        => $employee_union_dues,
            ],
            'bonuses'   => [
                'mid_year_bonus' => $mid_year_bonus,
                'year_end_bonus' => $year_end_bonus,
            ],
            'annual_basic_pay'           => $annual_salary,
            'annual_gsis_sss'               => $annual_gsis,
            'annual_ecip'                   => $annual_ecip,
            'annual_philhealth'             => $annual_philhealth,
            'annual_pagibig'                => $annual_pagibig,
            'annual_union_dues'             => $employee_union_dues,
            'tax_option'                    => $tax_option,
            'pei'                           => $pei,
            'tax_exemption'                 => $tax_exemption,
            'taxable_basic_salary'          => $taxable_basic_salary,
            'taxable_13th_month'            => $taxable_13th_month,
            'total_taxable_compensation'    => $total_taxable_compensation,
            'annual_tax_due'                => $annual_tax,
            'withholding_tax'               => $withholding_tax,
        );
    }

    // Earnings/Deductions of employee
    protected function earnings_deductions($employee_payroll, $earnings, $allowances, $deductions, $loans, $lwop, $is_semi_monthly = false)
    {
        $employee_deductions = array(
            'Withholding Tax'   => $deductions['withholding_tax'],
            'GSIS'              => $deductions['mandatory_contributions']['gsis_sss'],
            'ECIP'              => $deductions['mandatory_contributions']['ecip'],
            'Pag-Ibig'          => $deductions['mandatory_contributions']['pagibig'],
            'Philhealth'        => $deductions['mandatory_contributions']['philhealth'],
            'Union Dues'        => $deductions['mandatory_contributions']['union_dues'],
            'Basic Pay'         => $lwop['basic_lwop_deduction'],
            'PERA Pay'          => $lwop['pera_lwop_deduction']
        );
        unset($allowances['total_earnings']);
        $employee_earnings = array(
            'Basic Pay'     => $earnings['basic_pay'],
            'PERA Pay'      => $earnings['pera_pay'],
            'Hazard Pay'    => $earnings['hazard_pay'],
            'Longevity Pay' => $earnings['longevity_pay'],
        );

        $employee_earnings = array_merge($employee_earnings);
        $employee_loans = [];
        foreach ($loans['loans'] as $loan) {
            $employee_loans[$loan->loanType->name] = [
                'amount' => $is_semi_monthly ? $loan->amount/2 : $loan->amount,
                'id' => $loan->id,
                'number_of_payments' => $loan->number_of_payments
            ];
        }
        $earnings_deductions = [];
        $earnings_deductions = $this->record_earnings_deductions($employee_earnings, $employee_deductions, $employee_loans, $employee_payroll->id);
        return $earnings_deductions;
    }

    // Compute Hazard Pay
    protected function compute_hazard_pay($salary)
    {

        /**
         *  1. Get the salary grade currently assigned to the employee
         *  2. Compute Salary Grade with conditions:
         *      - SG 19 below: 25%
         *      - 20         - 15%
         *      - 21         - 13%
         *      - 22         - 12%
         *      - 23         - 11%
         *      - 24-25      - 10%
         *      - 26         - 9%
         *      - 27         - 8%
         *      - 28         - 7%
         *      - 29-30      - 6%
         *      - 31         - 5%
         *  3. return computed hazard pay
         */

        $salary_grade = $salary['user_salary']->publicSalary->salaryGrade->salary_grade;

        $percentage = 0;

        if ($salary_grade <= 19) {
            $percentage = 0.25;
        } else if ($salary_grade == 20) {
            $percentage = 0.15;
        } else if ($salary_grade == 21) {
            $percentage = 0.13;
        } else if ($salary_grade == 22) {
            $percentage = 0.12;
        } else if ($salary_grade == 23) {
            $percentage = 0.11;
        } else if ($salary_grade == 24 || $salary_grade == 25) {
            $percentage = 0.10;
        } else if ($salary_grade == 26) {
            $percentage = 0.09;
        } else if ($salary_grade == 27) {
            $percentage = 0.08;
        } else if ($salary_grade == 28) {
            $percentage = 0.07;
        } else if ($salary_grade == 29 || $salary_grade == 30) {
            $percentage = 0.06;
        } else if ($salary_grade == 31) {
            $percentage = 0.05;
        }

        return $salary['salary'] * $percentage;
    }

    // Compute Longevity Pay
    protected function compute_longevity_pay($user, $salary)
    {

        /**
         *  1. Get date hired of employee - month only
         *  2. Get month of payroll
         *  3. Check if month matches month of date hired and payroll month
         *  4. Subtract current year and date hired year
         */

        $date_hired = $user->employeeMetaInfo->date_hired;
        $current_date = date('Y-m-d', strtotime($this->start_date));
        $longevity_pay = LongevityPay::where('user_id', $user->id)->orderBy('date', 'desc')->first();

        $cd_month = substr($current_date, 5, 2);
        $cd_year = substr($current_date, 0, 4);

        if($longevity_pay){
            $lpd_month = substr($longevity_pay->date, 5, 2);
            $lpd_year = substr($longevity_pay->date, 0, 4);
        }else{
            $lpd_month = substr($date_hired, 5, 2);
            $lpd_year = substr($date_hired, 0, 4);
        }

        $years_difference = $cd_year - $lpd_year;
        if (($years_difference >= 5 && $years_difference % 5 == 0) && $cd_month == $lpd_month) {

            $total = 0;
            $longevity_pays = LongevityPay::where('user_id', $user->id)->where('status', Status::ACTIVE)->get();

            foreach ($longevity_pays as $pay) {
                $total += $pay->latest_pay;
            }

            $existingLongevityPay = LongevityPay::where('user_id', $user->id)
                ->whereMonth('date', $cd_month)
                ->whereYear('date', $cd_year)
                ->first();

            if (!$existingLongevityPay) {
                $active_longevity = Longevity::where('status', Status::ACTIVE)->first();
                $rate = $active_longevity->value('percentage');
                $percentage = $rate / 100;

                $current_longevity_pay = $salary['salary'] * $percentage;
                $total += $current_longevity_pay;

                LongevityPay::create([
                    'user_id'       => $user->id,
                    'date'          => $this->start_date,
                    'number'        => count($longevity_pays) + 1,
                    'salary_rate'   =>  $salary['salary'],
                    'percentage'    =>  $rate,
                    'latest_pay'    => $current_longevity_pay,
                    'total_pay'     => $total,
                ]);
            }

            return $total;
        }

        return null;
    }

    /**
     * Compute for public payroll allowances
     */
    private function get_allowances($employee_payroll, $user_salary, $is_semi_monthly = false, $absences)
    {

        $allowances = Allowance::where('status', Status::ACTIVE)->get();
        $special_pays = SpecialPay::where('status', Status::ACTIVE)->get();

        $total_earnings = 0;

        foreach ($allowances as $allowance) {

            $leave_allowances = (new LeaveService())->getLeaveAllowanceByPeriod($allowance->id, $user_salary->user_id, LeaveAllowance::NOT_PAID, $this->start_date, $this->end_date);

            $leave_days = 0;

            foreach ($leave_allowances as $leave_allowance) {
                $leave = $leave_allowance->leave()->first();
                $leave_days = $leave->number_of_days;
            }

            $total_absents = count($absences) + $leave_days;

            $allowance_amount = 0;

            $employee_allowance = EmployeeAllowance::where(
                [
                    ['user_id', $user_salary->user_id],
                    ['allowance_id', $allowance->id]
                ]
            )->where(function($query){
                $query->whereDate('effectivity_date' , '<=', $this->start_date)->orWhereDate('effectivity_date', '<=', $this->end_date);
            })->orderBy('effectivity_date', 'desc')->first();

            if($employee_allowance){
                $allowance_amount = $employee_allowance->amount;
            }

            $allowance_amount_based_on_payroll_type = $is_semi_monthly ? $allowance_amount / 2 : $allowance_amount;

            if ($employee_allowance) {
                $allowance_title = $employee_allowance->allowance()->first()->title;

                $working_days = strtolower($allowance_title) == 'subsistence allowance' ? 30 : 22;

                $allowance_amount_per_day = $allowance_amount / $working_days;

                $deducted_amount = $allowance_amount_per_day * $total_absents;

                $allowance_amount = $allowance_amount_based_on_payroll_type - abs($deducted_amount);
            }

            $allowance_amount = $allowance_amount < 0 ? 0 : $allowance_amount;
            $total_earnings += $allowance_amount;
            $name = $allowance['title'];
            $data[$name] = $allowance_amount;

            $payroll_earning_deduction = PayrollEarningDeduction::create([
                'payroll_employee_id'   => $employee_payroll->id,
                'name'          => $allowance['title'],
                'amount'        => $allowance_amount ?? 0,
                'type'          => 1,
                'category'      => 1,
                'created_at'    => now(),
                'updated_at'    => now(),
            ]);

            if ($employee_allowance) {
                $resulting_allowance = 0;

                foreach ($absences as $absent) {
                    $resulting_allowance = $allowance_amount_based_on_payroll_type - $allowance_amount_per_day;

                    PayrollAllowanceBreakdown::create([
                        'earning_deduction_id' => $payroll_earning_deduction->id,
                        'lwop_type' => 1,
                        'date' => $absent['date'],
                        'minutes' => $absent['minutes'],
                        'gross_amount' => $allowance_amount_based_on_payroll_type,
                        'deducted_amount' => $allowance_amount_per_day,
                        'resulting_amount' => $resulting_allowance
                    ]);
                    $allowance_amount_based_on_payroll_type = $resulting_allowance;
                }
            }
        }

        foreach ($special_pays as $special_pay) {

            $employee_special_pay = EmployeeSpecialPay::where([
                ['user_id', $user_salary->user_id],
                ['special_pay_id', $special_pay->id]
            ])->value('amount');

            $get_special_pay = (new SpecialPayService())->getSpecialPayByFrequency($special_pay, $employee_special_pay, $this->start_date, $this->end_date, $is_semi_monthly);

            $total_earnings += $get_special_pay;
            $name = $special_pay['title'];
            $data[$name] = $get_special_pay;

            $payroll_earning_deduction = PayrollEarningDeduction::create([
                'payroll_employee_id'   => $employee_payroll->id,
                'name'          => $special_pay['title'],
                'amount'        => $get_special_pay ?? 0,
                'type'          => 1,
                'category'      => 1,
                'created_at'    => now(),
                'updated_at'    => now(),
            ]);
        }

        $data['total_earnings'] = $total_earnings;
        return $data;
    }

    /**
     * Get the salary of the employee - based on salary grades of tranches
     */
    protected function get_salary($user)
    {
        $user_salary = $user->salary()->without('user')->with(['publicSalary'])->first();
        $salary = $user_salary->publicSalary->salaryGrade->value;
        $salary = str_replace(',', '', $salary);

        return array(
            'user_salary'   => $user_salary,  // Salary Object
            'salary'        => $salary        // Salary Value
        );
    }

    /**
     * The function calculates the yearly bonus based on the date of hire, bonus date, and salary, and user
     * date of exit effectivity. returning the bonus amount if the employee has been hired for at least four months,
     * otherwise returning 0.
     *
     * @param \DateTimeInterface The date when the user was hired.
     * @param Carbon The `bonus_date` parameter represents the date on which the bonus is being calculated.
     * @param float The salary parameter is the employee's monthly salary.
     * @param object The "user" parameter is an object that represents a user or employee
     *
     * @return float a float value, which represents the yearly bonus amount.
     */
    private function getYearlyBonus($date_hired, $bonus_date, $salary, $user): float
    {
        $hired_date = Carbon::parse($date_hired);
        $four_months_ago = $bonus_date->copy()->subMonths(4);

        if (isset($user->exitInterview)) {
            if ($user->exitInterview->status == ExitInterviewStatus::DONE) {
                //NOTE: For now date of exit is used because the system has no date of exit effectivity
                $date_of_exit = Carbon::parse($user->exitInterview->date);
                if ($hired_date->lte($four_months_ago) && $date_of_exit->gte($bonus_date)) {
                    return (float)$salary;
                }
            }
        } elseif ($hired_date->lte($four_months_ago)) {
            return (float) $salary;
        }
        return 0;
    }

    private function fetch_refund ($user, $employee_payroll) {
        $approved = 1;
        $refunds = LeaveRefund::where([
            ['status', $approved], ['user_id', $user->id]
        ])->get();

        $basic_pay_refund = 0;

        foreach ($refunds as $refund) {
            $basic_pay_refund += $refund->total_amount;
        }

        $payroll_earning_deduction = PayrollEarningDeduction::create([
            'payroll_employee_id'   => $employee_payroll->id,
            'name'          => 'Refund Pay',
            'amount'        => $basic_pay_refund ?? 0,
            'type'          => 1,
            'category'      => 1,
            'created_at'    => now(),
            'updated_at'    => now(),
        ]);

        LeaveRefund::where([
            ['status', $approved], ['user_id', $user->id]
        ])->update([
            'earning_deduction_id' => $payroll_earning_deduction->id
        ]);
    }
}
