<?php

namespace Suiterus\Adg\Controllers\Payroll\Generator;

use App\Enums\ExitInterviewStatus;
use App\Enums\Status;
use App\Enums\YearlyBonusDates;
use Carbon\Carbon;
use Suiterus\Adg\Controllers\Payroll\Generator\PayrollGenerator;
use Suiterus\Adg\Models\Payroll\Allowance;
use Suiterus\Adg\Models\Payroll\SpecialPay;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\Salary\EmployeeSpecialPay;

class PrivatePayrollGenerator extends PayrollGenerator
{

    /*
        Private Payroll does not include PERA in the computations
        Selected Salary also differs from public payroll (User-specific salary, not based on Salary Grades)
    */

    protected $start_date;
    protected $end_date;
    protected $work_days;
    protected $payroll;
    protected $type;

    public function __construct($payroll)
    {
        $this->payroll = $payroll;
        $this->type = $payroll->type;
        $this->start_date = $payroll->period_start;
        $this->end_date = $payroll->period_end;
        $this->work_days = $this->get_work_days();
    }

    public function weekly($users)
    {
        $payslips = array();
        foreach ($users as $user) {
            /**
             * Get the salary, attendance, leaves first for computations
             */
            $user_salary = $this->get_salary($user);
            $date_hired = $user->employeeMetaInfo->date_hired;
            $leaves = $this->get_leave_deductions($user, $this->start_date, $this->end_date);
            $attendances = $user->attendance()->whereBetween('date_in', [$this->start_date, $this->end_date]);

            $salary = $user_salary['salary'];

            /**
             * After getting the needed data to process, call the functions to compute the
             * LWOP, deductions, and fetching the loans
             */
            $lwop = [];
            $deductions = $this->compute_deductions($date_hired, $user_salary, $user);
            $loans = $this->get_computed_loans($user);
            $schedule = $user->employeeSchedules()->first();

            $rates = $this->get_hourly_rates($salary, $schedule);

            $basic_pay = $this->compute_hours_worked($rates, $attendances->sum('total'), $schedule);
            $holiday_pay = 0;
            $overtime_pay = 0;

            $earnings = [
                'basic_pay'     => $basic_pay,
                'holiday_pay'   => $holiday_pay,
                'overtime_pay'  => $overtime_pay,
                'official_business_pay' => $user_salary['official_business_pay'],
                'e_cola'    =>  $user_salary['e_cola'],
                'special_pay'   =>  $user_salary['special_pay'],
                'load_allowance'    =>  $user_salary['load_allowance'],
                'transportation_allowance'  =>  $user_salary['transportation_allowance'],
                'branch_allowance'  =>  $user_salary['branch_allowance'],
                'gate_allowance'    =>  $user_salary['gate_allowance'],
                'rice_allowance'    =>  $user_salary['rice_allowance'],
                'substinence_allowance' =>  $user_salary['substinence_allowance'],
                'laundry_allowance' =>  $user_salary['laundry_allowance']
            ];

            $gross_pay = $basic_pay;

            $total_deductions = ($deductions['withholding_tax'] / 4) +
                ($deductions['mandatory_contributions']['gsis_sss'] / 4) +
                ($deductions['mandatory_contributions']['philhealth'] / 4) +
                ($deductions['mandatory_contributions']['pagibig']/ 4) +
                ($loans['total_amount'] / 4);

            $net_pay = $gross_pay - $total_deductions;

            $payslip = [
                'user'  => $user,
                'account_number'    => $user_salary['user_salary']->account_number,
                'salary'            => $user_salary['salary'],
                'net_pay'           => $net_pay,
            ];
            array_push($payslips, $payslip);

            // Record Individual Employee Payroll
            $employee_payroll = $this->payroll->payroll_employee()->create([
                'user_id'       => $user->id,
                'days_worked'   => $attendances->count(),
                'absences'      => 0,
                'lates'         => 0,
                'undertimes'    => 0,
                'basic_pay'     => $basic_pay,
                'gross_pay'     => $gross_pay,
                'deductions'    => $total_deductions,
                'net_pay'       => $net_pay,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            /**
             * Record all of the computed earnings and deductions, together with the
             * individual payroll records of the employees
             */
            $earnings_deductions = $this->earnings_deductions($allowances = [], $employee_payroll, $earnings, $deductions, $loans, $user_salary);
            $timekeeping_deductions = $this->timekeeping_deductions($employee_payroll, $lwop);
            $leave_deductions = $this->leave_deductions($leaves['leaves'], $employee_payroll->id);
            $this->record_employee_payroll($employee_payroll, $earnings_deductions, $timekeeping_deductions, $leave_deductions, $deductions, $user_salary);
        }
        return $payslips;
    }

    public function monthly($users)
    {
        $payslips = array();
        foreach ($users as $user) {
            /**
             * Get the salary, attendance, leaves first for computations
             */
            $user_salary = $this->get_salary($user);
            $leaves = $this->get_leave_deductions($user, $this->start_date, $this->end_date);
            $attendance = $this->get_timekeeping_scheduled($user);
            $salary = $user_salary['salary'];

            /**
             * After getting the needed data to process, call the functions to compute the
             * LWOP, deductions, and fetching the loans
             */
            $lwop = $this->compute_lwop($attendance, $leaves, $salary);
            $deductions = $this->compute_deductions($attendance, $user_salary, $user);
            $loans = $this->get_computed_loans($user);

            $basic_pay = $salary - $lwop['basic_lwop_deduction'];
            $holiday_pay = $this->compute_holiday($user, $salary);
            $overtime_pay = $this->compute_overtime($user, $salary);
            $allowances = $this->get_allowances($user_salary['user_salary']);

            $earnings = [
                'basic_pay'     => $basic_pay,
                'holiday_pay'   => $holiday_pay,
                'overtime_pay'  => $overtime_pay
            ];
            $earnings = array_merge($earnings, $allowances);

            $gross_pay = $basic_pay;

            $total_deductions = $deductions['withholding_tax'] +
                $deductions['mandatory_contributions']['gsis_sss'] +
                $deductions['mandatory_contributions']['philhealth'] +
                $deductions['mandatory_contributions']['pagibig'] +
                $loans['total_amount'];
            $net_pay = $gross_pay + $allowances['total_earnings'] - $total_deductions;

            $payslip = [
                'user'  => $user,
                'account_number'    => $user_salary['user_salary']->account_number,
                'salary'            => $user_salary['salary'],
                'net_pay'           => $net_pay,
            ];
            array_push($payslips, $payslip);

            // Record Individual Employee Payroll

            $employee_payroll = $this->payroll->payroll_employee()->create([
                'user_id'       => $user->id,
                'days_worked'   => $attendance['total_days'],
                'absences'      => count($lwop['absences']),
                'lates'         => count($lwop['late']),
                'undertimes'    => count($lwop['undertime']),
                'basic_pay'     => $basic_pay,
                'gross_pay'     => $gross_pay,
                'deductions'    => $total_deductions,
                'net_pay'       => $net_pay,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            /**
             * Record all of the computed earnings and deductions, together with the
             * individual payroll records of the employees
             */
            $earnings_deductions = $this->earnings_deductions($allowances, $employee_payroll, $earnings, $deductions, $loans, $user_salary);
            $timekeeping_deductions = $this->timekeeping_deductions($employee_payroll, $lwop);
            $leave_deductions = $this->leave_deductions($leaves['leaves'], $employee_payroll->id);
            $this->record_employee_payroll($employee_payroll, $earnings_deductions, $timekeeping_deductions, $leave_deductions, $deductions, $user_salary);
        }
        return $payslips;
    }

    //semi-monthly
    public function semi_monthly($users, $period_start, $period_end)
    {
        $payslips = array();
        foreach ($users as $user) {
            /**
             * Get the salary, attendance, leaves first for computations
             */

            $user_salary = $this->get_salary($user);
            $leaves = $this->get_leave_deductions($user, $this->start_date, $this->end_date);
            $attendance = $this->get_timekeeping_scheduled($user);

            $salary = $user_salary['salary'];

            $start_date = Carbon::parse($period_end)->firstOfMonth();
            $end_date = Carbon::parse($period_end)->lastOfMonth();
            $date_range = CarbonPeriod::create($start_date, $end_date);
            ($date_range->toArray());
            $count_dayOff = 0;
            foreach ($date_range as $day) {
                if ($this->is_day_off($user['id'], $day)) {
                    $count_dayOff += 1;
                }
            }
            $calendar_days = Carbon::parse($period_end)->daysInMonth;
            $working_days = $calendar_days - $count_dayOff;
            $daily_rate = $salary / $working_days;

            //For period date range
            $period_dateRange =  CarbonPeriod::create($period_start, $period_end);
            $count_dayOffs = 0;
            foreach ($period_dateRange as $day) {
                if ($this->is_day_off($user['id'], $day)) {
                    $count_dayOffs += 1;
                }
            }
            $period_workingDays = count($period_dateRange) - $count_dayOffs;

            $salary = $daily_rate * $period_workingDays;

            /**
             * After getting the needed data to process, call the functions to compute the
             * LWOP, deductions, and fetching the loans
             */
            $lwop = $this->compute_lwop($attendance, $leaves, $salary);
            $deductions = $this->compute_deductions($attendance, $user_salary, $user);
            $loans = $this->get_computed_loans($user);

            $basic_pay = $salary - $lwop['basic_lwop_deduction'];
            $holiday_pay = $this->compute_holiday($user, $salary);
            $overtime_pay = $this->compute_overtime($user, $salary);
            $allowances = $this->get_allowances($user, $user_salary['user_salary']);

            $earnings = [
                'basic_pay'     => $basic_pay,
                'holiday_pay'   => $holiday_pay,
                'overtime_pay'  => $overtime_pay,
                'official_business_pay' => $user_salary['official_business_pay'],
                'e_cola'    =>  $user_salary['e_cola'],
                'special_pay'   =>  $user_salary['special_pay'],
                'load_allowance'    =>  $user_salary['load_allowance'],
                'transportation_allowance'  =>  $user_salary['transportation_allowance'],
                'branch_allowance'  =>  $user_salary['branch_allowance'],
                'gate_allowance'    =>  $user_salary['gate_allowance'],
                'rice_allowance'    =>  $user_salary['rice_allowance'],
                'substinence_allowance' =>  $user_salary['substinence_allowance'],
                'laundry_allowance' =>  $user_salary['laundry_allowance']
            ];

            $gross_pay = $basic_pay;

            $total_deductions = $deductions['withholding_tax'] +
                $deductions['mandatory_contributions']['gsis_sss'] +
                $deductions['mandatory_contributions']['philhealth'] +
                $deductions['mandatory_contributions']['pagibig'] +
                $loans['total_amount'];

            $net_pay = $gross_pay - $total_deductions;

            $payslip = [
                'user'  => $user,
                'account_number'    => $user_salary['user_salary']->account_number,
                'salary'            => $user_salary['salary'],
                'net_pay'           => $net_pay,
            ];
            array_push($payslips, $payslip);

            // Record Individual Employee Payroll
            $employee_payroll = $this->payroll->payroll_employee()->create([
                'user_id'       => $user->id,
                'days_worked'   => $attendance['total_days'],
                'absences'      => count($lwop['absences']),
                'lates'         => count($lwop['late']),
                'undertimes'    => count($lwop['undertime']),
                'basic_pay'     => $basic_pay,
                'gross_pay'     => $gross_pay,
                'deductions'    => $total_deductions,
                'net_pay'       => $net_pay,
                'created_by'    => $this->payroll->created_by,
                'updated_by'    => $this->payroll->created_by,
            ]);

            /**
             * Record all of the computed earnings and deductions, together with the
             * individual payroll records of the employees
             */
            $earnings_deductions = $this->earnings_deductions($allowances, $employee_payroll, $earnings, $deductions, $loans, $user_salary);
            $timekeeping_deductions = $this->timekeeping_deductions($employee_payroll, $lwop);
            $leave_deductions = $this->leave_deductions($leaves['leaves'], $employee_payroll->id);
            $this->record_employee_payroll($employee_payroll, $earnings_deductions, $timekeeping_deductions, $leave_deductions, $deductions, $user_salary);
        }
        return $payslips;
    }

    // --------------------------- PRIVATE FUNCTIONS - ONLY FOR PRIVATE PAYROLL --------------------------- //

    /**
     * Compute the LWOP
     */
    private function compute_lwop($attendance, $leaves, $salary)
    {
        // Constants
        $basic_lwop_days = 31;

        $total_late_minutes = $attendance['total_late_minutes'];
        $total_undertime_minutes = $attendance['total_undertime_minutes'];
        $total_halfday_minutes = $attendance['total_halfday_minutes'];
        $total_absents = count($attendance['absences']);
        $total_halfday_minutes = $attendance['total_halfday_minutes'];
        $total_leave_points = $leaves['total_points'];

        // Vairables
        $total_lwop = 0;

        // Deductions
        $basic_pay_deduction = 0;
        $total_lwop_deduction = 0;

        /**
         * Compute the LWOP
         * This will only compute if there is value or points in either the
         * lates, undertimes, absents, halfdays, and leaves
         */
        if ($total_late_minutes > 0 || $total_undertime_minutes > 0 || $total_absents > 0 || $total_halfday_minutes > 0 || $total_leave_points > 0) {

            $total_lwop = $this->get_lwop_fraction($total_late_minutes, $total_undertime_minutes, $total_halfday_minutes, $total_absents);
            $total_lwop += $total_leave_points;
            $basic_pay_deduction = ($salary / $basic_lwop_days) * $total_lwop;
            $total_lwop_deduction = $basic_pay_deduction;
        }

        return array(
            'late'     => [
                'records'      => $attendance['lates'],
                'total_minutes'     => $total_late_minutes,
            ],
            'undertime' => [
                'records'      => $attendance['lates'],
                'total_minutes'     => $total_undertime_minutes
            ],
            'half_days' => [
                'records'      => $attendance['half_days'],
                'total_minutes'     => $total_halfday_minutes
            ],
            'absences'  => [
                'records'       => $attendance['absences'],
                'total_minutes' => $attendance['total_absent_minutes']
            ],
            'leaves'    => [
                'records'   => $leaves['leaves']
            ],
            'total_lwop_minutes'    => $total_late_minutes + $total_undertime_minutes + ($total_absents * 8),
            'total_lwop_points'     => $total_lwop,
            'basic_lwop_deduction'  => $basic_pay_deduction,
            'total_deduction'       => $total_lwop_deduction
        );
    }

    /**
     * Compute the deductions - mandatory
     */
    private function compute_deductions($date_hired, $salary, $user)
    {
        $tax_option = $salary['user_salary']->withholding_tax_option;
        $salary = $salary['salary'];
        $sss = $salary * 0.09;
        $philhealth = ($salary * 0.03) / 2;
        $philhealth = $philhealth > 900 ? 900 : $philhealth; // ceiling of additional - 900 php
        $pagibig = 100;

        // TAXABLE BASIC SALARY COMPUTATION
        $annual_salary = $salary * 12;
        $annual_sss = $sss * 12;
        $annual_philhealth = $philhealth * 12;
        $annual_pagibig = $pagibig * 12;

        $employee_union_dues = 400.00;

        $taxable_basic_salary = $annual_salary - ($annual_sss + $annual_philhealth + $annual_pagibig);

        // 13th MONTH PAY & OTHER BENEFITS COMPUTATION
        $year_end_bonus = $this->getYearlyBonus($date_hired, YearlyBonusDates::getYearEndDate(), $salary, $user);

        $pei = 0;
        $taxable_13th_month = $year_end_bonus + $pei;
        if ($taxable_13th_month > 90000) {
            $taxable_13th_month -= 90000;
        }

        // TOTAL TAXABLE COMPENSATION
        $total_taxable_compensation = $taxable_basic_salary + $taxable_13th_month;

        // TAX BRACKET
        $annual_tax = $this->get_annual_tax($total_taxable_compensation);
        $tax_exemption = $annual_tax > 0 ? 2 : 1;

        // Withholding Tax Options
        // Option A - Not Included in Mid-year and Year-end bonus
        // Option B - Included in Mid-year and Year-end bonus - Divisor 14

        $withholding_tax = 0;
        if ($total_taxable_compensation > 250000) {
            $divisor = $tax_option == 1 ? 12 : 14;
            $withholding_tax = $annual_tax / $divisor;
        }

        return array(
            'mandatory_contributions' => [
                'gsis_sss'              => $sss,
                'philhealth'        => $philhealth,
                'pagibig'           => $pagibig,
                'union_dues'        => $employee_union_dues,
            ],
            'bonuses'   => [
                'year_end_bonus' => $year_end_bonus,
            ],
            'annual_basic_pay'           => $annual_salary,
            'annual_gsis_sss'               => $annual_sss,
            'annual_philhealth'             => $annual_philhealth,
            'annual_pagibig'                => $annual_pagibig,
            'annual_union_dues'             => $employee_union_dues,
            'tax_option'                    => $tax_option,
            'pei'                           => $pei,
            'tax_exemption'                 => $tax_exemption,
            'taxable_basic_salary'          => $taxable_basic_salary,
            'taxable_13th_month'            => $taxable_13th_month,
            'total_taxable_compensation'    => $total_taxable_compensation,
            'annual_tax_due'                => $annual_tax,
            'withholding_tax'               => $withholding_tax,
        );
    }

    // Earnings/Deductions of employee
    protected function earnings_deductions($allowances, $employee_payroll, $earnings, $deductions, $loans, $user_salary)
    {
        $employee_deductions = array(
            'Withholding Tax'   => $deductions['withholding_tax'],
            'SSS'              => $deductions['mandatory_contributions']['gsis_sss'],
            'Pag-Ibig'          => $deductions['mandatory_contributions']['pagibig'],
            'Philhealth'        => $deductions['mandatory_contributions']['philhealth'],
        );
        unset($allowances['total_earnings']);
        $employee_earnings = array(
            'Basic Pay'     => $earnings['basic_pay'],
            'Holiday Pay'   => $earnings['holiday_pay'],
            'Overtime Pay'  => $earnings['overtime_pay']
        );
        $employee_earnings = array_merge($employee_earnings, $allowances);
        $employee_loans = [];
        foreach ($loans['loans'] as $loan) {
            $employee_loans[$loan->loanType->name] = $loan->amount;
        }
        $earnings_deductions = $this->record_earnings_deductions($employee_earnings, $employee_deductions, $employee_loans, $employee_payroll->id);
        return $earnings_deductions;
    }

    /**
     * Compute for private payroll allowances
     */
    private function get_allowances($user_salary)
    {

        $allowances = Allowance::where('status', Status::ACTIVE)->get();
        $special_pays = SpecialPay::where('status', Status::ACTIVE)->get();
        $total_earnings = 0;
        foreach ($allowances as $allowance) {
            $employee_allowance = EmployeeAllowance::where(
                [
                    ['user_id', $user_salary->user_id],
                    ['allowance_id', $allowance->id]
                ]
            )->value('amount');
            $total_earnings += $employee_allowance;
            $name = $allowance['title'];
            $data[$name] = $employee_allowance;
        }

        foreach ($special_pays as $specialPay) {
            $employee_special_pay = EmployeeSpecialPay::where([
                ['user_id', $user_salary->user_id],
                ['special_pay_id', $specialPay->id]
            ])->value('amount');
            $total_earnings += $employee_special_pay;
            $name = $specialPay['title'];
            $data[$name] = $employee_special_pay;
        }

        $data['total_earnings'] = $total_earnings;
        return $data;
    }

    /**
     * Get the salary of the employee - based on private salary
     */
    protected function get_salary($user)
    {
        $user_salary = $user->salary()->without('user')->with(['privateSalary'])->first();
        $salary = $user_salary->privateSalary->salary;

        return array(
            'user_salary'   => $user_salary,
            'salary'        => $salary
        );
    }

    /**
     * The function calculates the yearly bonus based on the date of hire, bonus date, and salary, and user
     * date of exit effectivity. returning the bonus amount if the employee has been hired for at least four months,
     * otherwise returning 0.
     *
     * @param \DateTimeInterface The date when the user was hired.
     * @param Carbon The `bonus_date` parameter represents the date on which the bonus is being calculated.
     * @param float The salary parameter is the employee's monthly salary.
     * @param object The "user" parameter is an object that represents a user or employee
     *
     * @return float a float value, which represents the yearly bonus amount.
     */
    private function getYearlyBonus($date_hired, $bonus_date, $salary, $user): float
    {
        $hired_date = Carbon::parse($date_hired);
        $four_months_ago = $bonus_date->copy()->subMonths(4);

        if (isset($user->exitInterview)) {
            if ($user->exitInterview->status == ExitInterviewStatus::DONE) {
                //NOTE: For now date of exit is used because the system has no date of exit effectivity
                $date_of_exit = Carbon::parse($user->exitInterview->date);
                if ($hired_date->lte($four_months_ago) && $date_of_exit->gte($bonus_date)) {
                    return (float)$salary;
                }
            }
        } elseif ($hired_date->lte($four_months_ago)) {
            return (float) $salary;
        }
        return 0;
    }


}
