<?php

namespace Suiterus\Adg\Controllers\Payroll\Generator;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use App\Enums\LeaveType;
use Carbon\CarbonPeriod;
use App\Enums\ScheduleBasis;
use App\Enums\HolidayPurpose;
use App\Enums\OvertimeStatus;
use App\Jobs\GeneratePayroll;
use App\Enums\LeaveWithoutPay;
use App\Enums\OvertimePurpose;
use App\Enums\AttendanceLegend;
use App\Enums\ContributionType;
use App\Enums\Log\PayrollLogType;
use App\Enums\Payroll\PayrollType;
use App\Models\LoanPaymentHistory;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Sector;
use Suiterus\Adg\Models\SM\Holiday;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Suspension;
use App\Enums\Payroll\PayrollEntityType;
use Suiterus\Adg\Models\Payroll\Payroll;
use Suiterus\Adg\Models\SM\Contribution;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Services\Time\TimeService;
use Suiterus\Adg\Models\Payroll\LeaveRefund;
use Suiterus\Adg\Models\SM\ScheduleTemplate;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\Approvals\UserOvertime;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Suiterus\Adg\Models\Payroll\PayrollPayBreakdown;
use Suiterus\Adg\Models\Payroll\NightDifferentialPay;
use Suiterus\Adg\Models\Payroll\TimekeepingDeduction;
use Suiterus\Adg\Services\Attendance\AttendanceService;
use Suiterus\Adg\Models\Payroll\PayrollEarningBreakdown;
use Suiterus\Adg\Models\Payroll\PayrollEarningDeduction;
use Suiterus\Adg\Models\Payroll\PayrollHolidayBreakdown;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;

/**
 * Generator for payroll computations - parent class
 * Class has child classes - Public and Private for separation of computations for the two sectors
 * Reason for separation is computation speed. If we put the functions of both private and public
 * Reason for separation is computation speed. If we put the functions of both private and public
 * in the same class, we would need to put conditions to the part of the computations where there is a difference
 * in public and private, which would slow down the generating of payroll when reaching high numbers of employees
 * and attendances to compute the payroll for.
 *
 *
 * IMPORTANT NOTES:
 *  December, 2022
 *  1. The weekly payroll was not maintained during development since NKTI did not have a weekly payroll,
 *     and the main focus of the development was monthly payroll, so the results may not be accurate.
 *  2. Checking of half-day has issues for schedules where employee needs to time-out half day,
 *     might need to add condition where if the employee did not time-out and continued working
 *     at the second half of the day, it would not consider as half-day. BUT it still depends on the client's needs.
 *
 */

class PayrollGenerator
{
    use HasCustomLogs;
    protected $sector;

    public function __construct()
    {
        $sector = Sector::where('status', 'active')->first();
        $this->sector = $sector->name;
    }

    // ---------------------------------------------------------------- //
    //                         PUBLIC FUNCTIONS                         //
    // ---------------------------------------------------------------- //

    // Generate Employee Payroll
    public function generate($payroll_officer, $type, $entity_type, $entity, $period_start, $period_end)
    {

        $users = $this->get_users($type, $entity_type, $entity);

        $pending = 5;

        $pagibig = Contribution::where('type', ContributionType::PAGIBIG)->whereDate('effectivity_date', '<=', $period_start)->first();
        $gsis = Contribution::where('type', ContributionType::GSIS)->whereDate('effectivity_date' , '<=', $period_start)->first();
        $philhealth = Contribution::where('type', ContributionType::PHILHEALTH)->whereDate('effectivity_date' , '<=', '2024-01-01')->first();

        $metadata = [
            'philhealth_personal_rate' => $philhealth->employee_share / 100,
            'philhealth_company_rate' => $philhealth->company_share / 100,
            'pagibig_personal_rate' => $pagibig->employee_share / 100,
            'gsis_personal_rate' => $gsis->employee_share / 100,
            'gsis_company_rate' => $gsis->employee_share / 100
        ];

        $payroll = Payroll::create([
            'payroll_officer'   => $payroll_officer,
            'type'              => $type,
            'entity_type'       => $entity_type,
            'entity'            => $entity,
            'period_start'      => $period_start,
            'period_end'        => $period_end,
            'file'              => null, // null until approved
            'status'            => $pending,
            'completion_date'   => null, // null until approved
            'payroll_employee_generated_count' => 0,
            'payroll_employee_total_count' => count($users),
            'metadata'          => $metadata,
            'created_by'        => Auth::id(),
            'updated_by'        => Auth::id(),
        ]);

        foreach ($users as $user) {
            dispatch(new GeneratePayroll($payroll, $this->sector, $user));
        }

        $this->logCustomMessage(
            'payroll_generated',
            $payroll,
            Auth::user()->name . ' generated payroll ID no.' . $payroll->id,
            $payroll,
            PayrollLogType::GENERATE_PAYROLL,
            new Activity()
        );
    }

    /*
        Fetch LWOP details of Employee - Paginate/Table
        Gets the LWOP details of the employee from the employee payroll record created when generating a payroll
    */

    // Fetch LWOP details of Employee - Paginate/Table
    public function fetch_lwop_details($employee_payroll, $user)
    {

        $total_late_minutes = 0;
        $total_undertime_minutes = 0;

        $total_lwop = 0;
        $basic_pay_deduction = 0;
        $total_lwop_deduction = 0;
        $total_lwop_time = 0;
        $pera_pay_deduction = 0;

        $deduction = 2;

        // Salary
        $salary = 0;
        $pera = 0;
        if ($employee_payroll->payroll->type != PayrollType::WEEKLY) {
            if ($this->sector == 'public') {
                $user_salary = $user->salary()->without('user')->with(['publicSalary'])->first();
                $salary = $user_salary->publicSalary->salaryGrade->value;
                $salary = str_replace(',', '', $salary);
                $pera = 2000;
            } else {
                $user_salary = $user->salary()->without('user')->with(['privateSalary'])->first();
                $salary = $user_salary->privateSalary->salary;
            }

            $payroll = $employee_payroll->payroll;
            $period_start = $payroll->period_start;
            $period_end = $payroll->period_end;

            $semi_monthly = 2;

            $pera = $payroll->type == $semi_monthly ? $pera / 2 : $pera; //otherwise consider this as monthly
            $salary = $payroll->type == $semi_monthly ? $salary / 2 : $salary; //otherwise consider this as monthly

            $late_minutes = $employee_payroll->timekeeping_deductions()->where('type', LeaveWithoutPay::LATE)->get();
            $undertime_minutes = $employee_payroll->timekeeping_deductions()->where('type', LeaveWithoutPay::UNDERTIME)->get();
            $halfday_minutes = $employee_payroll->timekeeping_deductions()->where('type', LeaveWithoutPay::HALFDAY)->get();
            $absences = $employee_payroll->timekeeping_deductions()->where('type', LeaveWithoutPay::ABSENT)->count();

            $total_lwop = 0;
            $total_lwop += $absences;

            foreach ($late_minutes as $late_minute) {
                $total_lwop += $this->get_fraction($late_minute->minutes);
            }

            foreach ($undertime_minutes as $undertime_minute) {
                $total_lwop += $this->get_fraction($undertime_minute->minutes);
            }

            foreach ($halfday_minutes as $halfday_minute) {
                $total_lwop += $this->get_fraction($halfday_minute->minutes);
            }

            $attendance_service = new AttendanceService($user);
            $date_range = CarbonPeriod::create($period_start, $period_end);
            ($date_range->toArray());
            $count_dayOff = 0;
            foreach ($date_range as $day) {
                if ($attendance_service->isRestDay($day)) {
                    $count_dayOff += 1;
                }
            }
            $calendar_days = count($date_range);
            $working_days = $calendar_days - $count_dayOff;
            $daily_rate = $salary / $working_days;

            //For period date range
            $period_dateRange =  CarbonPeriod::create($period_start, $period_end);
            $count_dayOffs = 0;
            foreach ($period_dateRange as $day) {
                if ($attendance_service->isRestDay($day)) {
                    $count_dayOffs += 1;
                }
            }

            $period_workingDays = count($period_dateRange) - $count_dayOffs;

            $basic_lwop_days = count($period_dateRange);

            $basic_pay = $employee_payroll->earning_deduction()->where([
                ['type', $deduction], ['name', 'Basic Pay']
            ])->first();

            if ($this->sector == 'public') {
                $pera_pay = $employee_payroll->earning_deduction()->where([
                    ['type', $deduction], ['name', 'PERA Pay']
                ])->first();
            }

            if ($basic_pay){
                $total_lwop_deduction = $employee_payroll->total_basic_pay_deduction + $employee_payroll->total_pera_pay_deduction;
            }

            $minutes = $employee_payroll->timekeeping_deductions()->sum('minutes');
            $total_lwop_time = floor($minutes / 60) . ' hours, ' . ($minutes -   floor($minutes / 60) * 60) . ' minutes';
        }

        $details = array(
            'total_lwop_time'       => $total_lwop_time,
            'basic_pay_deduction'   => $basic_pay == true ? $basic_pay->amount : 0,
            'total_lwop_deduction'  => number_format($total_lwop_deduction, 2)
        );

        if($this->sector == 'public') {
            $details['pera_pay_deduction'] = $pera_pay == true ? $pera_pay->amount : 0;
        }
        return $details;
    }

    // Fetch Employee's Payroll Deductions
    public function fetch_deductions($employee_payroll)
    {

        $basic_pay = $employee_payroll->earning_deduction()->where('type', 1)->where('name', 'Basic Pay')->first();
        $gsis_sss = $employee_payroll->earning_deduction()->where('type', 2)->where('category', 1)->where('name', $this->sector == 'public' ? 'GSIS' : 'SSS')->first();
        $ecip = $employee_payroll->earning_deduction()->where('type', 2)->where('category', 1)->where('name', 'ECIP')->first();
        $pagibig = $employee_payroll->earning_deduction()->where('type', 2)->where('category', 1)->where('name', 'Pag-Ibig')->first();
        $philhealth = $employee_payroll->earning_deduction()->where('type', 2)->where('category', 1)->where('name', 'Philhealth')->first();
        $withholding_tax = $employee_payroll->earning_deduction()->where('type', 2)->where('category', 1)->where('name', 'Withholding Tax')->first();

        $union_dues = 0;
        $pera_pay = 0;
        if ($this->sector == 'public') {
            $union_dues = $employee_payroll->earning_deduction()->where('type', 2)->where('category', 1)->where('name', 'Union Dues')->first();
            $pera_pay = $employee_payroll->earning_deduction()->where('type', 1)->where('name', 'PERA Pay')->first();
            $union_dues = number_format($union_dues->amount, 2);
            $pera_pay = number_format($pera_pay->amount, 2);
        }
        $loans = $employee_payroll->earning_deduction()->where('type', 2)->where('category', 2)->get();

        return array(
            'gsis'              => number_format($gsis_sss->amount, 2),
            'ecip'              => number_format($ecip->amount, 2),
            'pagibig'           => number_format($pagibig->amount, 2),
            'philhealth'        => number_format($philhealth->amount, 2),
            'withholding_tax'   => number_format($withholding_tax->amount, 2),
            'basic_pay'         => number_format($basic_pay->amount, 2),
            'withholding_tax'   => number_format($withholding_tax->amount, 2),
            'pera_pay'          => $pera_pay,
            'union_dues'        => $union_dues,
            'loans'             => $loans
        );
    }

    // Get payroll earnings
    public function fetch_allowances($employee_payroll)
    {
        $earnings = $employee_payroll->earning_deduction()->where('type', 1)->where('name', '!=', 'Basic Pay')->get();
        $sum = 0;
        $array = array();

        foreach ($earnings as $key => $earning) {
            $array[] = [
                'title' => $earning->name,
                'amount' => number_format($earning->amount, 2),
                'allowance_breakdown' => $earning->allowance_breakdown,
                'pay_breakdown' => $earning->pay_breakdown,
                'holiday_breakdown' => $earning->holiday_breakdown,
                'refund_breakdown' => $earning->refund_breakdown
            ];
            $sum += $earning->amount;
        }
        $array['total_earnings'] = number_format($sum, 2);

        return $array;
    }


    // ---------------------------------------------------------------- //
    //                      PROTECTED FUNCTIONS                         //
    // ---------------------------------------------------------------- //

    // ------------------- FETCHING EMPLOYEE DETAILS FUNCTIONS ------------------- //

    // Get Users with schedules and attendances
    public function get_users($type, $entity_type, $entity)
    {
        $users = User::select('id', 'name', 'email')
            ->where(function ($query) use ($type) {
                // Check on getting employees with temporary schedule based on the active schedule from the date periods selected
                $query->whereHas('temporary_schedule_record')
                    ->orWhereHas('employeeSchedules', function ($query) use ($type) {
                        $query->whereHas('schedule_template', function ($query) use ($type) {
                            $query->when($type == PayrollType::WEEKLY, function ($query) {
                                $query->where('basis', ScheduleBasis::HOUR);
                            })->when($type == PayrollType::SEMIMONTHLY || $type == PayrollType::MONTHLY, function ($query) {
                                $query->where('basis', ScheduleBasis::SCHEDULE);
                            });
                        });
                    });
            })->whereHas('salary')->whereHas('employeeMetaInfo', function ($query) use ($entity_type, $entity) {
                $query->when($entity_type == PayrollEntityType::BRANCH, function ($query) use ($entity) {
                    $query->where('branch_id', $entity);
                })->when($entity_type == PayrollEntityType::DEPARTMENT,  function ($query) use ($entity) {
                    $query->where('department_id', $entity);
                })->when($entity_type == PayrollEntityType::POSITION,  function ($query) use ($entity) {
                    $query->where('position_id', $entity);
                });
            })
            ->with(['employeeMetaInfo' => function ($query) {
                $query->select('id', 'user_id', 'employee_id', 'position_id', 'date_hired')
                    ->with(['position' => function ($query) {
                        $query->select('id', 'title');
                    }])
                    ->without(['branch', 'corporation', 'division', 'department', 'employeeType']);
            }, 'salary' => function ($query) {
                $query->without('user');
            }, 'temporary_schedule_record'])
            ->when($entity_type == PayrollEntityType::INDIVIDUAL, function ($query) use ($entity) {
                $query->where('id', $entity);
            })
            ->without(['roles', 'permissions', 'storage', 'training_latest', 'training_filter'])
            ->get();

        return $users;
    }

    // Get loans and compute
    protected function get_computed_loans($user, $is_semi_monthly = false)
    {
        $current_month = date('m', strtotime($this->end_date));
        $approved = 4;
        $loans = $user->loan()
            ->select('id', 'loan_id', 'amount', 'number_of_payments', 'submission_date', 'status')
            ->whereIn('status', [$approved])
            ->where('from', '<=', $this->start_date)
            ->where('to', '>=', $this->end_date)
            ->with(['loanType' => function ($query) {
                $query->select('id', 'name', 'status');
            }])
            ->without(['user']);

        $total_amount = 0;
        if ($loans->count() > 0) {
            $total_amount = $is_semi_monthly ? $loans->sum('amount') / 2 : $loans->sum('amount');
        }

        return array(
            'loans'         => $loans->get(),
            'total_amount'  => $total_amount
        );
    }

    // Get Total Work days of current month
    protected function get_work_days()
    {
        // get the current period details
        $current_year = date('Y', strtotime($this->end_date));
        $current_month = date('m', strtotime($this->end_date));
        $total_days = date('t', mktime(0, 0, 0, $current_month, 1, $current_year));

        // check the work days for a 5 and 6 day work week
        $working_days = 0;
        $working_days_with_saturday = 0;
        for ($i = 1; $i <= $total_days; $i++) {
            $cur_date = date('w', strtotime($current_year . '-' . $current_month . '-' . $i));
            if ($cur_date >= 1 && $cur_date <= 5) {
                $working_days++;
                $working_days_with_saturday++;
            }
            if ($cur_date == 6) {
                $working_days_with_saturday++;
            }
        }
        return array(
            'working_days'                  => $working_days,
            'working_days_with_saturday'    => $working_days_with_saturday
        );
    }

    // ------------------- LEAVES FUNCTIONS - Employee Leave Management ---------------------- //
    public function get_leave_deductions($user, $start_date, $end_date)
    {
        $leaves = $user->leaves()
            ->whereBetween('start_date', [$start_date, $end_date])
            ->whereHas('leave_without_pay')
            ->with('leave_without_pay')
            ->get();

        $total_points = 0;
        foreach ($leaves as $leave) {
            $total_points += $leave->leave_without_pay->deducted_points;
        }

        return [
            'leaves'        => $leaves,
            'total_points'  => $total_points,
        ];
    }


    // ------------------- TIMEKEEPING COMPUTATION FUNCTIONS - Attendances ------------------- //

    /**
     * Get timekeeping scheduled based on the timetables
     *
     */
    public function get_timekeeping_scheduled($employee_payroll, $user, $calendar_days, $working_days, $salary) {

        $absences = [];
        $late_values = [];
        $undertime_values = [];
        $halfday_values = [];

        // Minutes Variables
        $absent_minutes = 0;
        $total_late_minutes = 0;
        $total_undertime_minutes = 0;
        $total_halfday_minutes = 0;
        $total_lwop_points = 0;

        $total_basic_pay_deduction = 0;
        $total_pera_pay_deduction = 0;

        $leave = LeaveBalance::where('user_id', $employee_payroll->user_id)->where('leave_type_id', LeaveType::VACATION_LEAVE)->first();

        $attendanceService = new AttendanceService(User::find($user['id']));
        $attendances = Attendance::where('user_id', $user['id'])->where(function ($query) {
            $query->whereDate('time_in', '>=', $this->start_date)
                ->whereDate('time_in', '<=', $this->end_date);
        })->get();

        // Temporary Schedule Details
        $schedule = $user->employeeSchedules()->without('user')->first()->schedule_template;

        // For computation of absences
        $currentDay = Carbon::parse($this->start_date);
        $lastDay = Carbon::parse($this->end_date);

        $dtr = $attendanceService->getTimekeepingAttendanceDisplay($currentDay, $lastDay);
        $absences = $attendanceService->getAbsentValues($dtr);
        $leave_balance_remaining = $leave->balance;

        foreach ($absences as $absent) {
            $absent_minutes += $absent['minutes'];
            $total_lwop_points += 1;
            $basic_pay = (($salary / $calendar_days) * 1);
            $pera = 2000;
            $pera = $employee_payroll->payroll->type == PayrollType::SEMIMONTHLY ? $pera/2 : $pera;
            $pera_deduct = (($pera / $working_days) * 1);

            $leave_balance_deducted = 1;
            $leave_balance_result = $leave_balance_remaining - $leave_balance_deducted;

            TimekeepingDeduction::create([
                'payroll_employee_id'   => $employee_payroll->id,
                'date'          => $absent['date'],
                'minutes'       => $absent['minutes'],
                'type'          => LeaveWithoutPay::ABSENT,
                'points'        => 1,
                'basic_pay_deduction'  => $basic_pay,
                'pera_deduction'     => $pera_deduct,
                'leave_balance_remaining'   => $leave_balance_remaining,
                'leave_balance_deduction'   => $leave_balance_deducted,
                'leave_balance_result'   => $leave_balance_result,
                'created_at'    => now(),
                'updated_at'    => now()
            ]);
            $leave_balance_remaining = $leave_balance_result;
            
            if ($leave_balance_result < 0) {
                $total_basic_pay_deduction += $basic_pay;
                $total_pera_pay_deduction += $pera_deduct;
            }
        }
        /**
         *  TODO: Verify if we still need the halfday values and halfday minutes
         *  * The computation for half day is in the Attendance Model
         */


        /**
         *  * This is just to match the existing expectation of this function for undertime values,
         *  TODO: Verify if still needed. Otherwise, remove since we have total undertime/late minutes.
         */

        $attendancesWithLate = $attendanceService->getLates($this->start_date, $this->end_date);
        $late_values = [];

        foreach ($attendancesWithLate as $attendanceWithLate) {
            $value = [
                'date'      => date('Y-m-d', strtotime($attendanceWithLate->time_in)),
                'minutes'   => TimeService::hoursToMinutes($attendanceService->totalLates($attendanceWithLate))
            ];
            $fraction = $this->get_fraction(TimeService::hoursToMinutes($attendanceService->totalLates($attendanceWithLate)));
            $basic_pay = (($salary / $calendar_days) * $fraction);
            $pera = 2000;
            $pera = $employee_payroll->payroll->type == PayrollType::SEMIMONTHLY ? $pera/2 : $pera;
            $pera_deduct = (($pera / $working_days) * $fraction);

            $leave_balance_deducted = $fraction;
            $leave_balance_result = $leave_balance_remaining - $leave_balance_deducted;

            TimekeepingDeduction::create([
                'payroll_employee_id'   => $employee_payroll->id,
                'date'          => $value['date'],
                'minutes'       => $value['minutes'],
                'type'          => LeaveWithoutPay::LATE,
                'points'        => $fraction,
                'basic_pay_deduction'  => $basic_pay,
                'pera_deduction'     => $pera_deduct,
                'leave_balance_remaining'   => $leave_balance_remaining,
                'leave_balance_deduction'   => $leave_balance_deducted,
                'leave_balance_result'   => $leave_balance_result,
                'created_at'    => now(),
                'updated_at'    => now()
            ]);
            $leave_balance_remaining = $leave_balance_result;
            array_push($late_values, $value);
            $total_lwop_points += $fraction;

            if ($leave_balance_result < 0) {
                $total_basic_pay_deduction += $basic_pay;
                $total_pera_pay_deduction += $pera_deduct;
            }
        };

        $total_late_minutes = TimeService::hoursToMinutes($attendanceService->totalLates($attendancesWithLate));

        $attendancesWithUndertime = $attendanceService->getUndertimes($this->start_date, $this->end_date);

        $undertime_values = [];

        foreach ($attendancesWithUndertime as $attendanceWithUndertime) {
            $value = [
                'date'      => date('Y-m-d', strtotime($attendanceWithUndertime->time_in)),
                'minutes'   => TimeService::hoursToMinutes($attendanceService->totalUndertime($attendanceWithUndertime))
            ];
            $fraction = $this->get_fraction(TimeService::hoursToMinutes($attendanceService->totalUndertime($attendanceWithUndertime)));
            $basic_pay = (($salary / $calendar_days) * $fraction);
            $pera = 2000;
            $pera = $employee_payroll->payroll->type == PayrollType::SEMIMONTHLY ? $pera/2 : $pera;
            $pera_deduct = (($pera / $working_days) * $fraction);

            $leave_balance_deducted = $fraction;
            $leave_balance_result = $leave_balance_remaining - $leave_balance_deducted;

            TimekeepingDeduction::create([
                'payroll_employee_id'   => $employee_payroll->id,
                'date'          => $value['date'],
                'minutes'       => $value['minutes'],
                'type'          => LeaveWithoutPay::UNDERTIME,
                'points'        => $fraction,
                'basic_pay_deduction'  => $basic_pay,
                'pera_deduction'     => $pera_deduct,
                'leave_balance_remaining'   => $leave_balance_remaining,
                'leave_balance_deduction'   => $leave_balance_deducted,
                'leave_balance_result'   => $leave_balance_result,
                'created_at'    => now(),
                'updated_at'    => now()
            ]);
            $leave_balance_remaining = $leave_balance_result;
            array_push($undertime_values, $value);
            $total_lwop_points += $fraction;

            if ($leave_balance_result < 0) {
                $total_basic_pay_deduction += $basic_pay;
                $total_pera_pay_deduction += $pera_deduct;
            }
        }

        $total_undertime_minutes = TimeService::hoursToMinutes($attendanceService->totalUndertime($attendancesWithUndertime));

        $days_per_week = count($schedule->schedule_template);

        $total_days = count($attendances);

        $work_days = $days_per_week == 5 ? $this->work_days['working_days'] : $this->work_days['working_days_with_saturday'];

        $employee_payroll->update([
            'total_basic_pay_deduction' => $total_basic_pay_deduction,
            'total_pera_pay_deduction' => $total_pera_pay_deduction,
        ]);

        return array(
            'schedule'              => $schedule,
            'attendances'           => $attendances,
            'absences'              => $absences,
            'total_absent_minutes'  => $absent_minutes, // * We can remove the absent minutes since it is not used for calculating the LWOP
            'total_late_minutes'    => $total_late_minutes,
            'total_undertime_minutes'  => $total_undertime_minutes,
            'total_halfday_minutes' => $total_halfday_minutes, // TODO: Verify if we still need the halfday minutes
            'lates'                 => $late_values,
            'undertimes'            => $undertime_values,
            'half_days'             => $halfday_values, // TODO: Verify if we still need the halfday values
            'expected_work_days'    => $work_days,
            'total_days'            => $total_days,
            'total_lwop_points' => $total_lwop_points
        );
    }

    /**
     * Computing approved overtime pays
     */
    protected function compute_overtime($employee_payroll, $user, $salary)
    {

        /**
         *  1. Get all overtime records of user within the dates
         *  2. Get the days within the records and compute for the overtime set in hours or minutes
         */

        $overtimes = UserOvertime::where('user_id', $user->id)
            ->where(function ($query) use ($user) {
                $query->whereDate('start_date', '>=', $this->start_date)
                    ->whereDate('end_date', '<=', $this->end_date);
            })
            ->where('purpose', OvertimePurpose::PAY)
            ->where('status', 2)
            ->get();

        $total = 0;
        $payrollPayBreakdownIds = [];
        foreach ($overtimes as $overtime) {
            $start = strtotime($overtime->start_date);
            $end = strtotime($overtime->end_date);
            $diff = $end - $start;


            $start_time = strtotime($overtime->start_time);
            $end_time = strtotime($overtime->end_time);
            $time = ($end_time - $start_time) / 60;

            $days = round($diff / (60 * 60 * 24)) + 1;

            /**
             * Overtime formula:
             * ( ( Salary Basic / 22 / 8 ) / 60 ) * ot minutes total
             */

            $rate = 1.25;

            $isHoliday = Holiday::whereDate('date', $overtime->start_date)->first();
            $isDayOff = Attendance::where([
                ['user_id', $user->id],
                ['schedule_day_id', null],
                ['timetable_id', null],
                ['shift_id', null],
            ])->whereDate('time_in', $overtime->start_date)->first();

            $remarks = 'Regular overtime Pay';

            if ($isHoliday || $isDayOff) {
                $rate = 1.5;
            }

            if ($isHoliday) {
                $remarks = "{$isHoliday->title} overtime pay";
            }

            if ($isDayOff) {
                $remarks = "Day-off overtime pay";
            }

            $ot = (($salary / 22 / 8) / 60) * $rate * ($days * $time);

            $payrollPayBreakdown = PayrollPayBreakdown::create([
                'date' => $overtime->start_date,
                'pay_hours' => $time / 60,
                'working_hours' => 8,
                'working_days' => 22,
                'hourly_rate' => ($salary / 22 / 8),
                'percentage' => $rate,
                'resulting_amount' => $ot,
                'remarks' => $remarks
            ]);

            $payrollPayBreakdownIds[] = $payrollPayBreakdown->id;

            $total += $ot;
        }

        $payroll_earning_deduction = PayrollEarningDeduction::create([
            'payroll_employee_id'   => $employee_payroll->id,
            'name'          => 'Overtime Pay',
            'amount'        => $total ?? 0,
            'type'          => 1,
            'category'      => 1,
            'created_at'    => now(),
            'updated_at'    => now(),
        ]);

        PayrollPayBreakdown::whereIn('id', $payrollPayBreakdownIds)->update(['earning_deduction_id' => $payroll_earning_deduction->id]);

        return $total;
    }

    public function compile_holidays($holidays, $user_attendances, $user_id, $overtimes)
    {
        $holidayCollection = collect();


        // Count the occurrences of each date
        $dateCounts = $holidays->countBy('date');
        $isPresent = false;
        $nextDay = false;
        $totalHours = 0;
        $isOvertime = false;
        $overtimeHours = 0;
        $nightDiffHours = 0;
        foreach ($holidays as $item) {
            $date = $item['date'];
            $isDoubleHoliday = $dateCounts[$date] > 1 ? true : false;
            $carbonDate = Carbon::createFromFormat('Y-m-d', $date);
            $isWeekday = $carbonDate->isWeekday();
            foreach ($overtimes as $overtime) {
                $overtime_date_in = Carbon::parse($overtime->start_date)->format('Y-m-d');
                $overtime_date_out = Carbon::parse($overtime->end_date)->format('Y-m-d');
                if (($overtime_date_in == $date) || ($overtime_date_out == $date)) {
                    $isOvertime = true;
                    $start_time_ot = strtotime($overtime->start_time);
                    $end_time_ot = strtotime($overtime->end_time);
                    $overtimeHours = ($end_time_ot - $start_time_ot) / 60;
                }
            }
            foreach ($user_attendances as $attendance) {
                $time_in = Carbon::parse($attendance->time_in)->format('Y-m-d');
                $time_out = Carbon::parse($attendance->time_out)->format('Y-m-d');
                //Get night differential hours
                $startTime = Carbon::parse($attendance->time_in);
                $endTime = $startTime->copy()->endOfDay();
                $totalHours = $startTime->diffInHours($endTime);
                if ($time_in == $date) {
                    $isPresent = true;
                    $nightDiffHours = $this->compute_night_diff($attendance);
                    if ($time_in != $time_out) {
                        $nextDay = true;
                        $startTime = Carbon::parse($attendance->time_in);
                        $endTime = $startTime->copy()->endOfDay();
                    }
                }
                if (($time_out == $date) && ($time_in != $time_out)) {
                    $isPresent = true;
                    $nextDay = true;
                    $endTime = Carbon::parse($attendance->time_out);
                    $startTime = $endTime->copy()->startOfDay();
                    $totalHours = $startTime->diffInHours($endTime);
                    $holidayInfo = collect([
                        'holiday_id' => $item['id'],
                        'title' => $item['title'],
                        'date' => $date,
                        'isDoubleHoliday' => $isDoubleHoliday,
                        'isWeekday' => $isWeekday,
                        'purpose' => $item['purpose'],
                        'status' => $item['status'],
                        'isPresent' => $isPresent,
                        'nextDay'   => $nextDay,
                        'hours'     => $totalHours,
                        'isOvertime' => $isOvertime,
                        'overtimeHours' => $overtimeHours,
                        'nightDiffHours' => $this->compute_night_diff($attendance)

                    ]);
                    $holidayCollection->push($holidayInfo);
                    $nextDay = false;
                    $totalHours = 0;
                }
            }
            $holidayInfo = collect([
                'holiday_id' => $item['id'],
                'title' => $item['title'],
                'date' => $date,
                'isDoubleHoliday' => $isDoubleHoliday,
                'isWeekday' => $isWeekday,
                'purpose' => $item['purpose'],
                'status' => $item['status'],
                'isPresent' => $isPresent,
                'nextDay'   => $nextDay,
                'hours'     => $totalHours,
                'isOvertime' => $isOvertime,
                'overtimeHours' => $overtimeHours,
                'nightDiffHours' => $nightDiffHours

            ]);
            $holidayCollection->push($holidayInfo);
            $isPresent = false;
            $nextDay = false;
            $totalHours = 0;
            $isOvertime = false;
            $overtimeHours = 0;
            $nightDiffHours = 0;
        }
        return $holidayCollection;
    }

    public function compute_holiday($employee_payroll, $user, $salary)
    {
        //Get all generated holidays
        $holidays = Holiday::where('status', 1)->where(function ($query) {
            $query->whereDate('date', '>=', $this->start_date)
                ->whereDate('date', '<=', $this->end_date);
        })->get();
        //Get user attendances
        $user_attendances = $user->attendance()->where(function ($query) {
            $query->whereDate('time_in', '>=', $this->start_date)
                ->whereDate('time_in', '<=', $this->end_date);
        })->get();

        //Get overtimes for overtime
        $overtimes = UserOvertime::where('user_id', $user->id)
            ->where(function ($query) use ($user) {
                $query->whereDate('start_date', '>=', $this->start_date)
                    ->whereDate('end_date', '<=', $this->end_date);
            })
            ->where('purpose', OvertimePurpose::PAY)
            ->where('status', OvertimeStatus::APPROVED)
            ->get();

        //Get employee type
        $user_id = $user->id;
        $allowed = ['probationary', 'regular', 'full time service', 'permanent'];
        $user_here = EmployeeMetaInfo::where('user_id', $user_id)->first();
        $emp_type = $user_here->employeeType->title;
        $emp_type = strtolower($emp_type);

        //compile holidays with employee's attendances
        $compiled_holidays = $this->compile_holidays($holidays, $user_attendances, $user_id, $overtimes);
        //set variables for total and daily wage
        $total = 0;
        //Legal computation of daily salary is = salary multiplied by 12 months in a year and divided by 261 working days in a year
        $daily = ($salary * 12) / 261;
        $regular_working_hours = 8;
        $hourly = $daily / $regular_working_hours;
        $nightDiffRate = .10;
        $OTHolidayRate = .30;

        $payroll_earning_deduction = PayrollEarningDeduction::create([
            'payroll_employee_id'   => $employee_payroll->id,
            'name'          => 'Holiday Pay',
            'amount'        => 0,
            'type'          => 1,
            'category'      => 1,
            'created_at'    => now(),
            'updated_at'    => now(),
        ]);

        // Compute pay for double holiday
        $double_holidays = $compiled_holidays->where('isDoubleHoliday', true)
            ->where('isWeekday', true)
            ->where('isPresent', true);
        foreach ($double_holidays as $dates) {
        $hours = $dates->get('hours');
            if (($dates->get('purpose') == HolidayPurpose::SPECIAL_NON_WORKING) && (in_array($emp_type, $allowed))) {
                if (($dates->get('nextDay') == true) || ($dates->get('isOvertime') == true)) {
                    //Since this computes holiday pay for the day including the wage, it is divided by 2 to only cover the holiday pay
                    // $total += (($hourly * $hours) / 2) + (($nightDiffRate * $hourly) * $dates->get('nightDiffHours')) + (($OTHolidayRate * $hourly) * $dates->get('overtimeHours'));
                    $pay = (($hourly * $hours) / 2);
                    $total += $pay;

                    PayrollHolidayBreakdown::create([
                        'earning_deduction_id' => $payroll_earning_deduction->id,
                        'holiday_id' => $dates->get('holiday_id'),
                        'date' => $dates->get('date'),
                        'holiday_pay' => $pay,
                    ]);

                } else {
                    $pay = ($daily * .3);
                    $total += $pay;

                    PayrollHolidayBreakdown::create([
                        'earning_deduction_id' => $payroll_earning_deduction->id,
                        'holiday_id' => $dates->get('holiday_id'),
                        'date' => $dates->get('date'),
                        'holiday_pay' => $pay,
                    ]);
                }
            } else {
                // $total += $daily + (($nightDiffRate * $hourly) * $dates->get('nightDiffHours')) + (($OTHolidayRate * $hourly) * $dates->get('overtimeHours'));
                $total += $daily;

                PayrollHolidayBreakdown::create([
                    'earning_deduction_id' => $payroll_earning_deduction->id,
                    'holiday_id' => $dates->get('holiday_id'),
                    'date' => $dates->get('date'),
                    'holiday_pay' => $daily,
                ]);
            }
        }
        // //Compute pay for holidays that are not double and are regular (meaning they are entitled whether they are present or not)
        $regular_holidays = $compiled_holidays->where('isDoubleHoliday', false)
            ->where('isWeekday', true)
            ->where('purpose', HolidayPurpose::REGULAR)
            ->where('nextDay', false);
        foreach ($regular_holidays as $dates) {
            $hours = $dates->get('hours');
            // $total += $daily + (($nightDiffRate * $hourly) * $dates->get('nightDiffHours')) + (($OTHolidayRate * $hourly) * $dates->get('overtimeHours'));
            $total += $daily;
            PayrollHolidayBreakdown::create([
                'earning_deduction_id' => $payroll_earning_deduction->id,
                'holiday_id' => $dates->get('holiday_id'),
                'date' => $dates->get('date'),
                'holiday_pay' => $daily,
            ]);
        }

        // //Compute pay for regular holidays that the employee was present
        $regular_present = $compiled_holidays->where('isDoubleHoliday', false)
            ->where('isWeekday', true)
            ->where('purpose', HolidayPurpose::REGULAR)
            ->where('isPresent', true);
        foreach ($regular_present as $dates) {
            $hours = $dates->get('hours');
            if (($dates->get('nextDay') == true) || ($dates->get('isOvertime') == true)) {
                // $total += ($hourly * $hours) / 2 + (($nightDiffRate * $hourly) * $dates->get('nightDiffHours')) + (($OTHolidayRate * $hourly) * $dates->get('overtimeHours'));
                $pay = ($hourly * $hours) / 2;
                $total += $pay;
                PayrollHolidayBreakdown::create([
                    'earning_deduction_id' => $payroll_earning_deduction->id,
                    'holiday_id' => $dates->get('holiday_id'),
                    'date' => $dates->get('date'),
                    'holiday_pay' => $pay,
                ]);
            } else {
                // $total += $daily + (($nightDiffRate * $hourly) * $dates->get('nightDiffHours')) + (($OTHolidayRate * $hourly) * $dates->get('overtimeHours'));
                $total += $daily;
                PayrollHolidayBreakdown::create([
                    'earning_deduction_id' => $payroll_earning_deduction->id,
                    'holiday_id' => $dates->get('holiday_id'),
                    'date' => $dates->get('date'),
                    'holiday_pay' => $daily,
                ]);
            }
        }

        // //Compute pay for special holidays(employee should be present)
        $special_holidays = $compiled_holidays->where('isDoubleHoliday', false)
            ->where('isWeekday', true)
            ->where('purpose', HolidayPurpose::SPECIAL_NON_WORKING)
            ->where('isPresent', true);
        if (in_array($emp_type, $allowed)) {
            foreach ($special_holidays as $dates) {
                $hours = $dates->get('hours');
                if (($dates->get('nextDay') == true) || ($dates->get('isOvertime') == true)) {
                    // $total += ($hourly * $hours) / 2 + (($nightDiffRate * $hourly) * $dates->get('nightDiffHours')) + (($OTHolidayRate * $hourly) * $dates->get('overtimeHours'));
                    $pay = ($hourly * $hours) / 2;
                    $total += $pay;

                    PayrollHolidayBreakdown::create([
                        'earning_deduction_id' => $payroll_earning_deduction->id,
                        'holiday_id' => $dates->get('holiday_id'),
                        'date' => $dates->get('date'),
                        'holiday_pay' => $pay,
                    ]);
                } else {
                    // $total += $daily + (($nightDiffRate * $hourly) * $dates->get('nightDiffHours')) + (($OTHolidayRate * $hourly) * $dates->get('overtimeHours'));
                    $total += $daily;
                    PayrollHolidayBreakdown::create([
                        'earning_deduction_id' => $payroll_earning_deduction->id,
                        'holiday_id' => $dates->get('holiday_id'),
                        'date' => $dates->get('date'),
                        'holiday_pay' => $daily,
                    ]);
                }
            }
        }

        $payroll_earning_deduction->update([
            'amount' => $total
        ]);

        return $total;
    }

    public function compute_night_diff($attendance)
    {

        $nextDay = false;
        $time_in = Carbon::parse($attendance->time_in);
        $time_out = Carbon::parse($attendance->time_out);
        $time_out_after = $time_out;
        if ($time_in->format('Y-m-d') != $time_out->format('Y-m-d')) {
            $nextDay = true;
        }
        $startOfDay = Carbon::parse($time_in->format('Y-m-d') . ' 00:00:00');
        $nightShiftStart = Carbon::parse($time_in->format('Y-m-d') . ' 21:00:00');
        $endOfDay = Carbon::parse($time_in->format('Y-m-d') . ' 23:59:59');
        // Check if the time_in is after the night shift start time
        if ($time_in->gte($nightShiftStart) && $time_in->lt($endOfDay)) {
            $time_in = $time_in->copy(); // Use a copy of time_in
        } else {
            $time_in = $nightShiftStart; // Set time_in to the night shift start time
        }
        // Check if the time_out is before the end of the day
        if ($time_out->gt($startOfDay) && $time_out->lte($endOfDay)) {
            $time_out = $time_out->copy(); // Use a copy of time_out
        } else {
            $time_out = $endOfDay; // Set time_out to the end of the day
        }
        // Calculate the duration in hours
        $hoursPresent = $time_in->diffInHours($time_out);
        // Adjust the hoursPresent if the time_in is after 9:00 PM but before 12:00 AM
        if ($nextDay == true) {
            if ($time_in->gte($nightShiftStart) && $time_in->lt($nightShiftStart->copy()->addHours(3))) {
                $hoursPresent++;
                $endOfDay = Carbon::parse($time_out_after->format('Y-m-d') . ' 05:00:00');
                // Calculate the duration in hours
                $hoursUntil5AM = $time_out_after->diffInHours($endOfDay);
                $hoursAfterMidnight = $hoursUntil5AM >= 5 ? 5 : 5 - $hoursUntil5AM;
                $hoursPresent += $hoursAfterMidnight;
            }
        }


        return $hoursPresent;
    }

    public function get_night_diff($employee_payroll, $attendances, $salary, $period_workingDays)
    {
        $payrollPayBreakdownIds = [];
        $pay = 0;

        $night_diff_pay = NightDifferentialPay::where('status', Status::ACTIVE)->first();
        $percentage = $night_diff_pay->value('percentage');

        foreach ($attendances as $attendance) {
            $time_in = strtotime($attendance->time_in);
            $time_out = strtotime($attendance->time_out);

            $night_differential = floatval(str_replace(':', '.', $attendance->night_differential));

            if (!$night_differential) {
                continue;
            }

            $working_hours_seconds = $time_out - $time_in;
            $working_hours = floor($working_hours_seconds / 3600);

            $working_days = $period_workingDays;

            $hourly_rate = ($salary / $working_days) / $working_hours;

            $night_diff_rate = $percentage / 100;

            $pay += $hourly_rate * $night_diff_rate * $night_differential;

            $payrollBreakdown = PayrollPayBreakdown::create([
                'date' => $attendance->time_in,
                'pay_hours' => $night_differential,
                'working_hours' => $working_hours,
                'working_days' => $working_days,
                'hourly_rate' => $hourly_rate,
                'percentage' => $night_diff_rate,
                'resulting_amount' => $pay,
                'remarks' => 'Night Differential Pay'
            ]);

            $payrollPayBreakdownIds[] = $payrollBreakdown->id;
        }

        $payroll_earning_deduction = PayrollEarningDeduction::create([
            'payroll_employee_id'   => $employee_payroll->id,
            'name'          => 'Night Differential Pay',
            'amount'        => $pay ?? 0,
            'type'          => 1,
            'category'      => 1,
            'created_at'    => now(),
            'updated_at'    => now(),
        ]);

        PayrollPayBreakdown::whereIn('id', $payrollPayBreakdownIds)->update(['earning_deduction_id' => $payroll_earning_deduction->id]);

        return $pay;
    }

    // For Weekly Payroll / Hour-based employees
    protected function compute_hours_worked($hourly_rate, $hours_worked, $schedule)
    {
        $expected_hours = $schedule->hours->hours;
        if ($hours_worked > $expected_hours) {
            $excess = $hours_worked - $expected_hours;
            $hours_worked -= $excess;
        }
        return $hourly_rate * $hours_worked;
    }


    // ------------------- DEDUCTIONS COMPUTATIONS - Taxes ------------------- //

    protected function get_annual_tax($total_taxable_compensation)
    {
        $annual_tax = 0;
        if ($total_taxable_compensation > 250000 && $total_taxable_compensation < 400000) {  // 15% of excess over 250,000
            $annual_tax = ($total_taxable_compensation - 250000) * 0.15;
        } else if ($total_taxable_compensation > 400000 && $total_taxable_compensation < 800000) {  // 22,500 + 20% of excess over 400,000
            $annual_tax = 22500 + (($total_taxable_compensation - 400000) * 0.2);
        } else if ($total_taxable_compensation > 800000 && $total_taxable_compensation < 2000000) {  // 102,500 + 25% of excess over 800,000
            $annual_tax = 102500 + (($total_taxable_compensation - 800000) * 0.25);
        } else if ($total_taxable_compensation > 2000000 && $total_taxable_compensation < 8000000) {  // 402,500 + 30% of excess over 2,000,000
            $annual_tax = 402500 + (($total_taxable_compensation - 2000000) * 0.3);
        } else if ($total_taxable_compensation > 8000000) {  // 2,202,500 + 35% of excess over 8,000,000
            $annual_tax = 2202500 + (($total_taxable_compensation - 8000000) * 0.35);
        }
        return $annual_tax;
    }


    // ------------------- DATA RECORDING FUNCTIONS ------------------- //

    // Record LWOPs
    protected function timekeeping_deductions($employee_payroll, $lwop)
    {
        $lwops = [];
        if (empty($lwop)) {
            return $lwops;
        }
        foreach ($lwop['late']['records'] as $late) {
            array_push($lwops, [
                'payroll_employee_id'   => $employee_payroll->id,
                'date'          => $late['date'],
                'minutes'       => $late['minutes'],
                'type'          => LeaveWithoutPay::LATE,
                'points'        => $this->get_fraction($late['minutes']),
                'created_at'    => now(),
                'updated_at'    => now()
            ]);
        }
        foreach ($lwop['undertime']['records'] as $undertime) {
            array_push($lwops, [
                'payroll_employee_id'   => $employee_payroll->id,
                'date'          => $undertime['date'],
                'minutes'       => $undertime['minutes'],
                'type'          => LeaveWithoutPay::UNDERTIME,
                'points'        => $this->get_fraction($undertime['minutes']),
                'created_at'    => now(),
                'updated_at'    => now()
            ]);
        }
        foreach ($lwop['half_days']['records'] as $undertime) {
            array_push($lwops, [
                'payroll_employee_id'   => $employee_payroll->id,
                'date'          => $undertime['date'],
                'minutes'       => $undertime['minutes'],
                'type'          => LeaveWithoutPay::HALFDAY,
                'points'        => $this->get_fraction($undertime['minutes']),
                'created_at'    => now(),
                'updated_at'    => now()
            ]);
        }
        foreach ($lwop['absences']['records'] as $absent) {
            array_push($lwops, [
                'payroll_employee_id'   => $employee_payroll->id,
                'date'          => $absent['date'],
                'minutes'       => $absent['minutes'],
                'type'          => LeaveWithoutPay::ABSENT,
                'points'        => 1.00,
                'created_at'    => now(),
                'updated_at'    => now()
            ]);
        }

        return $lwops;
    }

    // Record Employee's earnings and deductions
    protected function record_earnings_deductions($earnings, $deductions, $loans, $employee_payroll_id)
    {
        $earnings_deductions = [];
        foreach ($earnings as $name => $amount) {
            array_push($earnings_deductions, array(
                'payroll_employee_id'   => $employee_payroll_id,
                'name'          => $name,
                'amount'        => $amount ?? 0,
                'type'          => 1,
                'category'      => 1,
                'created_at'    => now(),
                'updated_at'    => now(),
            ));
        }
        foreach ($deductions as $name => $amount) {
            array_push($earnings_deductions, array(
                'payroll_employee_id'   => $employee_payroll_id,
                'name'          => $name,
                'amount'        => $amount ?? 0,
                'type'          => 2,
                'category'      => 1,
                'created_at'    => now(),
                'updated_at'    => now(),
            ));
        }
        foreach ($loans as $name => $loan) {
            $category = 2; // Loans
            $type = 2; // Deduction
            array_push($earnings_deductions, array(
                'payroll_employee_id'   => $employee_payroll_id,
                'name'          => $name,
                'amount'        => $loan['amount'] ?? 0,
                'loan_id'       => $loan['id'], // ID of user_has_loan
                'number_of_payments' => $loan['number_of_payments'],
                'type'          => $type,
                'category'      => $category,
                'created_at'    => now(),
                'updated_at'    => now(),
            ));
        }

        return $earnings_deductions;
    }

    protected function leave_deductions($leaves, $employee_payroll_id)
    {

        $deductions = [];
        foreach ($leaves as $leave) {
            array_push($deductions, [
                'payroll_employee_id'   => $employee_payroll_id,
                'leave_without_pay_id'  => $leave->leave_without_pay->id,
            ]);
        }

        return $deductions;
    }

    // Record Employee's earnings and deductions (LWOP and Taxes)
    protected function record_employee_payroll($employee_payroll, $earnings_deductions, $timekeeping_deductions, $leave_deductions, $deductions, $user_salary)
    {

       // $employee_payroll->timekeeping_deductions()->insert($timekeeping_deductions);

        foreach ($earnings_deductions as $deduction) {
            $earning_deduction = $employee_payroll->earning_deduction()->create($deduction);
            $category = 2; // Loans
            $approved = 1;
            if ($deduction['category'] == $category) {
                $latest_payment_history = LoanPaymentHistory::where([
                    ['user_has_loan_id', $deduction['loan_id']], ['status', $approved]
                ])->latest()->first();

                $create = [
                    'payroll_earning_deduction_id' => $earning_deduction->id,
                    'user_has_loan_id' => $deduction['loan_id']
                ];

                if ($latest_payment_history) {
                    if ($latest_payment_history->outstanding_balance != 0) {
                        $create['remaining_months'] = $latest_payment_history['remaining_months'] - 1;
                        $create['amortization_amount'] = $deduction['amount'];
                        $create['outstanding_balance'] = $latest_payment_history->outstanding_balance - $deduction['amount'];
                        LoanPaymentHistory::create($create);
                    }
                } else {
                    $create['remaining_months'] = $deduction['number_of_payments'] - 1;
                    $create['amortization_amount'] = $deduction['amount'];
                    $create['outstanding_balance'] = $deduction['amount'] * $deduction['number_of_payments'] - $deduction['amount'];
                    LoanPaymentHistory::create($create);
                }
            }
        }

        $employee_payroll->lwop_deductions()->insert($leave_deductions);
        $employee_payroll->withholding_tax()->create([
            'annual_basic_pay'  => $deductions['annual_basic_pay'],
            'gsis_sss'  => $deductions['annual_gsis_sss'],
            'ecip'  => $deductions['annual_ecip'],
            'pagibig' => $deductions['annual_pagibig'],
            'philhealth' => $deductions['annual_philhealth'],
            'union_dues' => isset($deductions['annual_union_dues']) ? $deductions['annual_union_dues'] : null,
            'taxable_basic_salary' => $deductions['taxable_basic_salary'],
            'mid_year_bonus' => isset($deductions['bonuses']['mid_year_bonus']) ? $deductions['bonuses']['mid_year_bonus'] : null,
            'year_end_bonus' => $deductions['bonuses']['year_end_bonus'],
            'pei' => $deductions['pei'],
            'taxable_13th_month' => $deductions['taxable_13th_month'],
            'taxable_compensation' => $deductions['total_taxable_compensation'],
            'tax_exemption' => $deductions['tax_exemption'],
            'annual_tax_due' => $deductions['annual_tax_due'],
            'tax_option'    => $user_salary['user_salary']->withholding_tax_option,
        ]);
    }


    // --------------------------- UTILITY / CONVERSION FUNCTIONS --------------------------- //

    // Get LWOP Fraction from minutes based on LWOP Table
    public function get_lwop_fraction($late_minutes, $undertime_minutes, $halfday_minutes, $absences)
    {

        $late_fraction = $this->get_fraction($late_minutes);
        $undertime_fraction = $this->get_fraction($undertime_minutes);
        $halfday_fraction = $this->get_fraction($halfday_minutes);

        // return the fractions with the absences number, since an absent is equal to 1 point
        return ($late_fraction + $undertime_fraction + $halfday_fraction + $absences);
    }

    protected function target_date($date, $days)
    {
        return date('Y-m-d', strtotime($date . ' +' . $days . ' day'));
    }

    protected function get_hourly_rates($salary, $schedule)
    {
        $annual_salary =  $salary * 12;
        $weekly_pay = $annual_salary / 52;
        $hourly_rate = $weekly_pay / $schedule->hours->hours;

        return $hourly_rate;
    }

    /**
     * Get the fractions of day based on the minutes of late/lwop
     */
    protected function get_fraction($minutes)
    {
        // Check first if there are late/undertime minutes
        // If there are late/undertime minutes, get the fraction.
        // If late/undertime minutes exceed 60, get add additional fraction values
        $fraction = 0;
        if($minutes > 0 && $minutes <= 60) {
            $fraction = self::FRACTIONS_OF_DAY[$minutes];
        } else if ($minutes > 60) {
            $hours = floor($minutes / 60);
            $fraction = (self::FRACTIONS_OF_DAY[60] * $hours);
            $minutes_left = $minutes - ($hours * 60);
            if ($minutes_left > 0) $fraction += self::FRACTIONS_OF_DAY[$minutes_left];
        }
        return $fraction;
    }


    // ------------------- CONSTANTS and VARIABLES ------------------- //

    // Working Hours/Minutes to Fractions of Day for TARDY/LATE and ABSENCES Computations
    protected const FRACTIONS_OF_DAY = [
        1 => 0.002,
        2 => 0.004,
        3 => 0.006,
        4 => 0.008,
        5 => 0.010,
        6 => 0.012,
        7 => 0.015,
        8 => 0.017,
        9 => 0.019,
        10 => 0.021,
        11 => 0.023,
        12 => 0.025,
        13 => 0.027,
        14 => 0.029,
        15 => 0.031,
        16 => 0.033,
        17 => 0.035,
        18 => 0.037,
        19 => 0.040,
        20 => 0.042,
        21 => 0.044,
        22 => 0.046,
        23 => 0.048,
        24 => 0.050,
        25 => 0.052,
        26 => 0.054,
        27 => 0.056,
        28 => 0.058,
        29 => 0.060,
        30 => 0.062,
        31 => 0.065,
        32 => 0.067,
        33 => 0.069,
        34 => 0.071,
        35 => 0.073,
        36 => 0.075,
        37 => 0.077,
        38 => 0.079,
        39 => 0.081,
        40 => 0.083,
        41 => 0.085,
        42 => 0.087,
        43 => 0.090,
        44 => 0.092,
        45 => 0.094,
        46 => 0.096,
        47 => 0.098,
        48 => 0.100,
        49 => 0.102,
        50 => 0.104,
        51 => 0.106,
        52 => 0.108,
        53 => 0.110,
        54 => 0.112,
        55 => 0.115,
        56 => 0.117,
        57 => 0.119,
        58 => 0.121,
        59 => 0.123,
        60 => 0.125,
    ];

    // Leave Credits Points
    protected const LEAVE_CREDITS = [
        1 => 0.042,
        2 => 0.083,
        3 => 0.125,
        4 => 0.167,
        5 => 0.208,
        6 => 0.250,
        7 => 0.292,
        8 => 0.333,
        9 => 0.375,
        10 => 0.417,
        11 => 0.458,
        12 => 0.500,
        13 => 0.542,
        14 => 0.583,
        15 => 0.625,
        16 => 0.667,
        17 => 0.708,
        18 => 0.750,
        19 => 0.792,
        20 => 0.833,
        21 => 0.875,
        22 => 0.917,
        23 => 0.958,
        24 => 1.000,
        25 => 1.042,
        26 => 1.083,
        27 => 1.125,
        28 => 1.167,
        29 => 1.208,
        30 => 1.250,
    ];
}
