<?php

namespace Suiterus\Adg\Controllers\PDS;

use Exception;
use App\Models\User;
use App\Enums\PDS_Status;
use Illuminate\Http\Request;
use App\Enums\Log\PDSLogType;
use App\Http\Classes\NotificationMessage;
use InvalidArgumentException;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Position_history;
use Suiterus\Adg\Models\PDS\PDSPerPosition;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\PDS\UpdatePDS as UPDS;
use Suiterus\Adg\Models\EMI\EmployeeExtraField;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldGroupHistory;

class PDSController extends Controller
{

    use HasCustomLogs;
    
    //approve
    public function approve(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.employee_extra_field_columns,id'
        ]);
        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $employee= EmployeeExtraFieldColumn::where('id', $request->id)->with('employeeExtraField')->withTrashed()->first();
            EmployeeExtraFieldColumn::where('id', $request->id)->withTrashed()->restore();
            $update = UPDS::where('eefc_id', $employee->id)->whereIn('status', [PDS_Status::PENDING, PDS_Status::ONHOLD])->first();
        
            EmployeeExtraFieldColumn::where('id', $employee->id)->update([
                'field_value'   =>  $update['field_value'] ? $update['field_value'] : null,
                'modified_by'   =>  Auth::id()
            ]);

            $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $employee->employeeExtraField->user_id)->first();
            $position_history = Position_history::where('user_id', $employeeMetaInfo->user_id)->where('position_id', $employeeMetaInfo->position_id)->orderBy('created_at','desc')->first();
            
            
            PDSPerPosition::where('eefc_id', $employee->id)->where('position_history_id', $position_history->id)->update([
                'field_value' => $update->field_value,
                'updated_by'   => Auth::id(),
            ]);


            $update = UPDS::where('eefc_id', $employee->id)->update([
                'status'     => PDS_Status::APPROVE,
                'updated_by' => Auth::id()
            ]);

            $newRecord = UPDS::where('eefc_id', $employee->id)->first();

            $newRecord->old = collect($update);
            $newRecord->attributes = collect($newRecord);
            
            $this->logCustomMessage(
                'approve_update_request',
                $newRecord,
                Auth::user()->name . ' approved the update request of ' . $employeeMetaInfo->user->name . ' for field: ' . $newRecord->employeeExtraFieldColumn->field_name,
                $newRecord,
                PDSLogType::APPROVE,
                new Activity()
            );

            $empExtraFields = EmployeeExtrafield::select('id')->where('user_id', $employee->employeeExtraField->user_id)->where('table_name', 'personal_information')->first();
            $employeeFields= EmployeeExtraFieldColumn::where('eef_id', $empExtraFields->id)->get();

            $userField = User::findOrFail($employee->employeeExtraField->user_id);
            $userFirstname = "";
            $userLastname = "";

            foreach($employeeFields as $employeeField){

                if($employeeField->field_name == 'first_name'){

                    $userFirstname = $employeeField->field_value;
                }
                else if($employeeField['field_name'] == 'last_name'){

                    $userLastname = $employeeField['field_value'];

                }else{ 

                }  
            }

            $userField->name = $userFirstname . ' ' . $userLastname;
            $userField->save();

            NotificationMessage::notifySenderAndReceiver('Approve PDS', $employeeMetaInfo->user_id, Auth::id());
            DB::commit();
            return response()->json([
                'text'  => 'PDS update request is now approved.',
            ]);
            
        }catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the records'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    
    //approve all
    public function approve_all(Request $request) {
        $validate = Validator::make($request->all(), [
            'user_id'       =>  'required|exists:adg_db.employee_extra_fields,user_id',
            'table_name'    =>  'required' 
        ]);
        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
                $updates = UPDS::whereHas('employeeExtraFieldColumn', function($query) use($request){
                    $query->whereHas('employeeExtraField', function($query) use($request){
                        $query->where('user_id', $request->user_id)->whereIn('table_name', $request->table_name);
                    })->withTrashed();
                })->whereIn('status', [PDS_Status::PENDING, PDS_Status::ONHOLD])->get();
            
                foreach($updates as $update){
                    
                    EmployeeExtraFieldColumn::where('id', $request->id)->withTrashed()->restore();
                    EmployeeExtraFieldColumn::where([  
                        'id'        =>  $update->eefc_id
                    ])->update([
                        'field_value'   =>  $update->field_value,
                        'modified_by'   =>  Auth::id()
                    ]);

                    $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $update->employeeExtraFieldColumn->employeeExtraField->user_id)->first();
                    $position_history = Position_history::where('user_id', $employeeMetaInfo->user_id)->where('position_id', $employeeMetaInfo->position_id)->orderBy('created_at','desc')->first();

                    PDSPerPosition::where('eefc_id', $update->eefc_id)->where('position_history_id', $position_history->id)->update([
                        'field_value' => $update->field_value,
                        'updated_by'   => Auth::id(),
                    ]);

                    UPDS::whereId($update->id)->update([
                        'status'       => PDS_Status::APPROVE,
                        'updated_by'   => Auth::id(),
                        'updated_at'   => now()
                    ]);
                    
                    $newRecord = UPDS::find($update->id);

                    $newRecord->old = collect($update);
                    $newRecord->attributes = collect($newRecord);
                    
                    $this->logCustomMessage(
                        'approve_all_update_request',
                        $newRecord,
                        Auth::user()->name . ' approved the update request of ' . $employeeMetaInfo->user->name . ' for field: ' . $newRecord->employeeExtraFieldColumn->field_name,
                        $newRecord,
                        PDSLogType::APPROVE_ALL,
                        new Activity()
                    );

                $empExtraFields = EmployeeExtrafield::select('id')->where('user_id', $update->employeeExtraFieldColumn->employeeExtraField->user_id)->where('table_name', 'personal_information')->first();
                $employeeFields= EmployeeExtraFieldColumn::where('eef_id', $empExtraFields->id)->get();
    
                $userField = User::findOrFail($update->employeeExtraFieldColumn->employeeExtraField->user_id);
                $userFirstname = "";
                $userLastname = "";

                foreach($employeeFields as $employeeField){

                    if($employeeField->field_name == 'first_name'){
    
                        $userFirstname = $employeeField->field_value;
                    }
                    else if($employeeField['field_name'] == 'last_name'){
    
                        $userLastname = $employeeField['field_value'];
    
                    }else{ 
    
                    }  
                }
    
                $userField->name = $userFirstname . ' ' . $userLastname;
                $userField->save();
                
                }
                NotificationMessage::notifySenderAndReceiver('All PDS Approve', $request->user_id, Auth::id());
                DB::commit();
                return response()->json([
                    'text'  => "All pending request has been approved."
                ]);            

        }catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

     //decline
     public function decline(Request $request)
     {
 
         $validate = Validator::make($request->all(), [
             'id'    => 'required|exists:adg_db.new_updates_pds,id'
         ]);
 
         if ($validate->fails()) {
             return response()->json([
                 'errors'  => $validate->errors()
             ], 400);
         }
 
         DB::beginTransaction();
         try {
 
             $PDS = UPDS::find($request->id);
             $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $PDS->employeeExtraFieldColumn->employeeExtraField->user_id)->first();
 
             if ($PDS->status == PDS_Status::DECLINE) {
                 throw new InvalidArgumentException('The record is already declined');
             }
 
            UPDS::whereId($request->id)->update([
                'status'        =>  PDS_Status::DECLINE,
                'reason'        =>  $request->reason,
                'updated_by'    =>  Auth::id()
            ]);

            $newRecord = UPDS::find($request->id);

            $newRecord->old = collect($PDS);
            $newRecord->attributes = collect($newRecord);
                    
            $this->logCustomMessage(
                'decline_update_request',
                $newRecord,
                Auth::user()->name . ' declined the update request of ' . $employeeMetaInfo->user->name . ' for field: ' . $newRecord->employeeExtraFieldColumn->field_name,
                $newRecord,
                PDSLogType::DECLINE,
                new Activity()
            );
            NotificationMessage::notifySenderAndReceiver('Decline PDS', $employeeMetaInfo->user_id, Auth::id());
             DB::commit();
             return response()->json([
                 'text'  => 'PDS declined.'
             ]);
         } catch (InvalidArgumentException $e) {
             DB::connection('adg_db')->rollBack();
             return response()->json([
                 'errors'    => $e->getMessage(),
                 'message'   => $e->getMessage()
             ], 500);
         } catch (Exception $e) {
             DB::connection('adg_db')->rollBack();
             return response()->json([
                 'errors'    => ['There was a problem in declining the records'],
                 'message'   => $e->getMessage()
             ], 500);
         }
     }

    //decline all
    public function decline_all(Request $request) {
        $validate = Validator::make($request->all(), [
            'user_id'       =>  'required|exists:adg_db.employee_extra_fields,user_id',
            'table_name'    =>  'required' 
        ]);
        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $record = UPDS::whereHas('employeeExtraFieldColumn', function($query) use($request){
                $query->whereHas('employeeExtraField', function($query) use($request){
                    $query->where('user_id', $request->user_id)->whereIn('table_name', $request->table_name);
                });
            })->whereIn('status', [PDS_Status::PENDING, PDS_Status::ONHOLD])->get();

                foreach($record as $update){
                    UPDS::whereId($update->id)->update([
                        'status'       => PDS_Status::DECLINE,
                        'reason'       => $request->reason,
                        'updated_by'   => Auth::id(),
                        'updated_at'   => now()
                    ]);

                    $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $update->employeeExtraFieldColumn->employeeExtraField->user_id)->first();

                    $newRecord = UPDS::find($update->id);

                    $newRecord->old = collect($update);
                    $newRecord->attributes = collect($newRecord);
                            
                    $this->logCustomMessage(
                        'decline_all_update_request',
                        $newRecord,
                        Auth::user()->name . ' declined all the update request of ' . $employeeMetaInfo->user->name . ' for field: ' . $newRecord->employeeExtraFieldColumn->field_name,
                        $newRecord,
                        PDSLogType::DECLINE_ALL,
                        new Activity()
                    );
                }
                NotificationMessage::notifySenderAndReceiver('All PDS Decline', $request->user_id, Auth::id());
                DB::commit();
                return response()->json([
                    'text'  => "All pending request has been declined."
                ]);            

        }catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in declining the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    // on-hold
    public function on_hold(Request $request){
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.new_updates_pds,id'
        ]);
        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            
            $update = UPDS::whereId($request->id)->where('status', PDS_Status::PENDING)->first();

            $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $update->employeeExtraFieldColumn->employeeExtraField->user_id)->first();

            $update = UPDS::whereId($update->id)->update([
                'status'     => PDS_Status::ONHOLD,
                'reason'     => $request->reason,
                'updated_by' => Auth::id()
            ]);

            $newRecord = UPDS::find($request->id);

            $newRecord->old = collect($update);
            $newRecord->attributes = collect($newRecord);
                    
            $this->logCustomMessage(
                'on_hold_update_request',
                $newRecord,
                Auth::user()->name . ' put the update request of ' . $employeeMetaInfo->user->name . ' for field: ' . $newRecord->employeeExtraFieldColumn->field_name . ' on-hold',
                $newRecord,
                PDSLogType::ON_HOLD,
                new Activity()
            );
            NotificationMessage::notifySenderAndReceiver('On Hold PDS', $employeeMetaInfo->user_id, Auth::id());
            DB::commit();
            return response()->json([
                'text'  => 'The PDS update request is still on-hold.'
            ]);
            
        }catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in this function'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    //fetch
    public function fetch_filter_update(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return User::whereHas('employeeExtraField', function($query){
            $query->whereHas('employeeExtraFieldColumn', function($query){
                $query->whereHas('updatePDS', function($query){
                    $query->whereIn('status', [PDS_Status::PENDING, PDS_Status::ONHOLD]);
                });
            });
        })->where('name', 'LIKE', '%'.$req->employeeName.'%')->whereHas('employeeMetaInfo',function($query) use($req){
            $query->when($req->employee_id != '' && $req->employee_id != null, function($query) use($req){
                if(count($req->employee_id) > 0){
                    return $query->whereIn('employee_id', $req->employee_id);
                }else{
                    return $query->where('employee_id', $req->employee_id);
                }
            })->when($req->position != '' && $req->position != null, function($query) use($req){
                return $query->where('position_id', $req->position);
            })->when($req->department != '' && $req->department !=null, function($query) use($req){
                return $query->where('department_id', $req->department);
            })->when($req->employeeType != '' && $req->employeeType != null, function($query) use($req){
                return $query->where('employee_type', $req->employeeType);
            });
        })->when($req->supervisor, function($query) use($req){
            $query->whereHas('user_supervisor', function($query) use($req){
                $query->when($req->supervisor != '' && $req->supervisor != null, function($query) use($req){
                    return $query->where('supervisor_id', $req->supervisor);
                });
            });
        })->with(['user_supervisor', 'personalInformation'=>function($query){
            $this->display($query);
        }, 'pdsChoices'=>function($query){
            $this->display($query);
        }, 'otherInformation'=>function($query){
            $this->display($query);
        }, 'permanentAddress'=>function($query){
            $this->display($query);
        }, 'residentialAddress'=>function($query){
            $this->display($query);
        },'spouse'=>function($query){
            $this->display($query);
        }, 'children'=>function($query){
            $this->display($query);
        }, 'father'=>function($query){
            $this->display($query);
        }, 'maidenName'=>function($query){
            $this->display($query);
        }, 'elementary'=>function($query){
            $this->display($query);
        }, 'secondary'=>function($query){
            $this->display($query);
        }, 'vocational'=>function($query){
            $this->display($query);
        }, 'college'=>function($query){
            $this->display($query);
        }, 'graduate'=>function($query){
            $this->display($query);
        }, 'civilService'=>function($query){
            $this->display($query);
        },  'workExperience'=>function($query){
            $this->display($query);
        }, 'voluntaryWork'=>function($query){
            $this->display($query);
        }, 'learningDevelopment'=>function($query){
            $this->display($query);
        }, 'skillHobby'=>function($query){
            $this->display($query);
        }, 'choice'=>function($query){
            $this->display($query);
        }, 'reference'=>function($query){
            $this->display($query);
        }, 'governmentIssuedId'=>function($query){
            $this->display($query);
        }, 'fileAttachment'=>function($query){
            $this->display($query);
        }])->paginate($paginate);
    }

    public function display($query){
        $query->with(['employeeExtraFieldColumn' => function($query){
            $query->with(['updatePDS' => function($query){
                $query->whereIn('status', [PDS_Status::PENDING, PDS_Status::ONHOLD]);
            }]);
        }]);
    }

    public function fetch_update_request(Request $request){
        return UPDS::whereIn('status', [PDS_Status::PENDING, PDS_Status::ONHOLD])->whereHas('employeeExtraFieldColumn', function($query) use($request){
            $query->whereHas('employeeExtraField', function($query) use($request){
                $query->where('user_id', $request->user_id)->whereIn('table_name', $request->table_name);
            });
        })->with('employeeExtraFieldGroupHistory')->get();
    }

    public function download_proof(Request $request)
    {
        $record = UPDS::where('id', $request->id)->first();

        $attachment = EmployeeExtraFieldGroupHistory::where('id', $record->eefg_history_id)->first();

        if (!$attachment) {
            return response()->json([
                'errors' => ["Employee haven't uploaded any file yet."],
            ], 404);
        }

        $filePath = $attachment->file_attachment . '.enc';

        $this->logCustomMessage(
            'download_proof_of_id',
            $attachment,
            Auth::user()->name . ' downloaded the proof of ID for ' . $attachment->employeeExtraFieldGroup->name,
            $attachment,
            PDSLogType::DOWNLOAD_PROOF,
            new Activity()
        );

        return response()->streamDownload(function () use ($filePath) {
            FileVault::disk('employee_proof_attachment')->streamDecrypt($filePath);
        }, '');
    }
}
