<?php

namespace Suiterus\Adg\Controllers\PDS\Employee;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Enums\Log\PDSLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\PDS\UpdatePDS;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Position_history;
use Suiterus\Adg\Models\PDS\PDSPerPosition;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\EMI\EmployeeExtraField;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;
use SoareCostin\FileVault\Facades\FileVault;
use App\Enums\PDS_Status;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldGroup;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldGroupHistory;
use Suiterus\Adg\Services\PDS\PDSEmployeeService;

class PDSEmployeeController extends Controller
{
    use HasCustomLogs;

    private $pdsEmployeeService;

    public function __construct(PDSEmployeeService $pdsEmployeeService)
    {
        $this->pdsEmployeeService = $pdsEmployeeService;
    }

    public function create(Request $request) {

        $valid = Validator::make($request->all(), [
            'new_row'   => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        DB::beginTransaction();
        try {
            foreach($request->new_row as $row){
                if($row['is_add_more_field']){

                    $group = $this->pdsEmployeeService->determineGroup($row['table_name'], Auth::id());

                    $employeeExtraField = EmployeeExtraField::create([
                        'user_id'       =>  Auth::id(),
                        'eefg_id'       =>  $group->id,
                        'table_name'    =>  $row['table_name'],
                        'created_by'    =>  Auth::id(),
                        'modified_by'   =>  Auth::id()
                    ]);

                    foreach($row['new_value'] as $value){
                        $employeeExtraFieldColumn = EmployeeExtraFieldColumn::create([
                            'eef_id'         =>  $employeeExtraField->id,
                            'field_name'     =>  $value['field_name'],
                            'field_label'    =>  $value['field_label'],
                            'field_value'    =>  '',
                            'field_type'     =>  $value['field_type'],
                            'field_status'   =>  1,
                            'created_by'     =>  Auth::id(),
                            'modified_by'    =>  Auth::id()
                        ]);

                        $employeeMetaInfo = EmployeeMetaInfo::where('user_id', Auth::id())->first();

                        $record = UpdatePDS::updateOrCreate([
                            'eefc_id'       =>  $employeeExtraFieldColumn->id,
                            'status'        =>  PDS_Status::PENDING
                        ],
                        [
                            'eefc_id'       =>  $employeeExtraFieldColumn->id,
                            'position_id'   =>  $employeeMetaInfo->position_id,
                            'field_value'   =>  $value['value'],
                            'old_value'     =>  $employeeExtraFieldColumn->field_value,
                            'status'        =>  PDS_Status::PENDING,
                            'created_by'    =>  Auth::id(),
                            'updated_by'    =>  Auth::id()
                        ]);

                        $position_history = Position_history::where('position_id', $employeeMetaInfo->position_id)->orderBy('created_at','desc')->first();

                        PDSPerPosition::create([
                            'position_history_id'   =>    $position_history->id,
                            'eefc_id'               =>    $employeeExtraFieldColumn->id,
                            'field_name'            =>    $employeeExtraFieldColumn->field_name,
                            'field_value'           =>    $employeeExtraFieldColumn->field_value,
                            'created_by'            =>    Auth::id()
                        ]);

                        $updated = UpdatePDS::where([
                            ['eefc_id', $employeeExtraFieldColumn->id], ['status', PDS_Status::PENDING]
                        ])->first();

                        $updated->old = collect($record);
                        $updated->attributes = collect($updated);

                        $this->logCustomMessage(
                            'update_pds',
                            $updated,
                            Auth::user()->name . ' updated the field: ' . $value['field_name'] . ' in PDS',
                            $updated,
                            PDSLogType::UPDATE,
                            new Activity()
                        );
                    }
                }else{
                    foreach($row['new_value'] as $value){
                        $group = $this->pdsEmployeeService->determineGroup($row['table_name'], Auth::id());

                        $employeeExtraField = EmployeeExtraField::where([
                            ['user_id' , Auth::id()], ['table_name' , $row['table_name']]
                        ])->first();

                        if ($employeeExtraField) {
                            $employeeExtraField->update([
                                'eefg_id' => $group->id
                            ]);
                        }
                        
                        $employeeExtraFieldColumn = EmployeeExtraFieldColumn::where('id', $value['id'])->withTrashed()->first();
                        EmployeeExtraFieldColumn::where('id', $value['id'])->withTrashed()->restore();
                        $employeeMetaInfo = EmployeeMetaInfo::where('user_id', Auth::id())->first();

						if($value['id'] != null){
                            $attachment = EmployeeExtraFieldGroupHistory::where('eefg_id', $employeeExtraField->eefg_id)->latest()->first();
                            $attachmentId = null;
                            if ($attachment) {
                                $attachmentId = $attachment->id;
                            }
							$record = UpdatePDS::updateOrCreate([
								'eefc_id'       =>  $value['id'],
								'status'        =>  PDS_Status::PENDING
							],
							[
								'eefc_id'       =>  $value['id'],
                                'eefg_history_id' => $attachmentId,
                                'position_id'   =>  $employeeMetaInfo->position_id,
								'field_value'   =>  $value['value'],
                                'old_value'     =>  $employeeExtraFieldColumn->field_value,
								'status'        =>  PDS_Status::PENDING,
								'created_by'    =>  Auth::id(),
								'updated_by'    =>  Auth::id()
							]);

                            $updated = UpdatePDS::where([
                                ['eefc_id', $employeeExtraFieldColumn->id], ['status', PDS_Status::PENDING]
                            ])->first();
    
                            $updated->old = collect($record);
                            $updated->attributes = collect($updated);
    
                            $this->logCustomMessage(
                                'update_pds',
                                $updated,
                                Auth::user()->name . ' updated the field: ' . $value['field_name'] . ' in PDS',
                                $updated,
                                PDSLogType::UPDATE,
                                new Activity()
                            );

						}else{
                            $employeeExtraField_id = null;
                            if(!$employeeExtraFieldColumn){
                                $employeeExtraField_id = EmployeeExtraField::where([
                                    [
                                        'user_id' , Auth::id()
                                    ],
                                    [
                                        'table_name' , $row['table_name']
                                    ]
                                ])->first()->id;
                            }

							$extraFieldColumn = EmployeeExtraFieldColumn::create([
								'eef_id'         =>   $employeeExtraField_id ?? $employeeExtraFieldColumn->eef_id,
								'field_name'     =>  $value['field_name'],
								'field_label'    =>  $value['field_label'],
								'field_value'    =>  '',
								'field_type'     =>  $value['field_type'],
								'field_status'   =>  1,
								'created_by'     =>  Auth::id(),
								'modified_by'    =>  Auth::id()
							]);

                            $employeeExtraField = EmployeeExtraField::where('id', $employeeExtraField_id ?? $employeeExtraFieldColumn->eef_id)->first();
    
                            $employeeExtraField->update([
                                'eefg_id' => $group->id
                            ]);

                            $attachment = EmployeeExtraFieldGroupHistory::where('eefg_id', $employeeExtraField->eefg_id)->latest()->first();
                            $attachmentId = null;
                            if ($attachment) {
                                $attachmentId = $attachment->id;
                            }

							$record = UpdatePDS::updateOrCreate([
								'eefc_id'       =>  $extraFieldColumn->id,
								'status'        =>  PDS_Status::PENDING
							],
							[
								'eefc_id'       =>  $extraFieldColumn->id,
                                'eefg_history_id ' => $attachmentId,
                                'position_id'   =>  $employeeMetaInfo->position_id,
								'field_value'   =>  $value['value'],
                                'old_value'     =>  $extraFieldColumn->field_value,
								'status'        =>  PDS_Status::PENDING,
								'created_by'    =>  Auth::id(),
								'updated_by'    =>  Auth::id()
							]);

                            $updated = UpdatePDS::where([
                                ['eefc_id', $extraFieldColumn->id], ['status', PDS_Status::PENDING]
                            ])->first();
    
                            $updated->old = collect($record);
                            $updated->attributes = collect($updated);
    
                            $this->logCustomMessage(
                                'update_pds',
                                $updated,
                                Auth::user()->name . ' updated the field: ' . $value['field_name'] . ' in PDS',
                                $updated,
                                PDSLogType::UPDATE,
                                new Activity()
                            );

						}
                    }
                }

            }

            DB::commit();
            return response()->json([
                'text' => 'New record have been created.',
            ]);

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(
                [
                    'errors'    => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                    'msg'       => $e->getMessage(),
                    'line'      => $e->getLine()
                ],
                500
            );
        }
    }

    public function update_file(Request $request){
        $extraFieldColumn = EmployeeExtraFieldColumn::whereId($request->id)->with('employeeExtraField')->first();

        Storage::delete($extraFieldColumn->field_value);
        $path = Storage::disk('local')->put('pds_file_attachment/'.$extraFieldColumn->employeeExtraField['user_id'].'/'.$extraFieldColumn->field_name,$request->file_attachment);

        EmployeeExtraFieldColumn::where('id', $extraFieldColumn->id)->update([
            'field_value'   => $path,
            'modified_by'   => Auth::id()
        ]);

        return response()->json([
            'text' => 'File successfully updated.'
        ]);
    }

    public function fetch_new_update(Request $request){
        return UpdatePDS::whereIn('status', [PDS_Status::PENDING, PDS_Status::ONHOLD])->whereHas('employeeExtraFieldColumn', function($query) use($request){
            $query->whereHas('employeeExtraField', function($query) use($request){
                $query->where('user_id', Auth::id())->whereIn('table_name', $request->table_name);
            });
       })->with('employeeExtraFieldGroupHistory')->orderBy('status', 'desc')->get();
    }

    public function fetch_result(Request $request){
        return response()->json([
            'data' => UpdatePDS::whereId($request->id)->first()
        ]);
    }

    public function set_as_acknowledged(Request $request){

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.new_updates_pds,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {

            $acknowledge = UpdatePDS::whereId($request->id)->first();
            $acknowledge->result_status = 1;
            $acknowledge->save();

            $update = UpdatePDS::whereId($request->id)->first();
            $info = EmployeeExtraFieldColumn::where('id', $update->eefc_id)->where('field_value', "")->orWhereNull('field_value')->delete();
            $info = EmployeeExtraFieldColumn::where('id', $update->eefc_id)->withTrashed()->first();

            $tableName = EmployeeExtraField::whereId($info->eef_id)->first();

            $extraFieldColumn = EmployeeExtraFieldColumn::where('eef_id', $tableName->id)->count();
            $fieldDeleted = EmployeeExtraFieldColumn::where('eef_id', $tableName->id)->whereNotNull('deleted_at')->count();

            if($extraFieldColumn <= $fieldDeleted){
                EmployeeExtraField::whereId($info->eef_id)->delete();
            }
            DB::commit();
            return response()->json([
                'text'  => 'Acknowledged successfully'
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function expiration_date_checker() {
        // Retrieve the data for the current user's government ID expiration fields
        $data = EmployeeExtraFieldColumn::where('field_name', 'government_id_expiration')
            ->whereHas('employeeExtraField', function($query) {
                $query->where('user_id', Auth::id());
            })
            ->orderBy('id', 'desc')
            ->get();
    
        // Check if there's any data
        if ($data->isNotEmpty()) {
            $closest_date_difference = null;
    
            // Loop through each item in the collection
            foreach ($data as $item) {
                // Parse the expiration date
                $expiration_date = Carbon::parse($item->field_value)->startOfDay();
                // Calculate the difference in days
                $date_difference = $expiration_date->diffInDays(Carbon::now()->startOfDay());
    
                // If the expiration date is within 90 days
                if ($date_difference <= 90) {
                    // Update closest date if it is null or the current difference is smaller
                    if (is_null($closest_date_difference) || $date_difference < $closest_date_difference) {
                        $closest_date_difference = $date_difference;
                    }
                }
            }
    
            // Return the closest date difference if found
            if (!is_null($closest_date_difference)) {
                return $closest_date_difference;
            }
        }
    
        // If no expiration date is within 90 days, return null
        return null;
    }

    public function upload_proof(Request $request) {

        $validate = Validator::make($request->all(), [
            'group_name' => 'required|string',
            'ids' => 'required|array|nullable:' . env('ADG_DB_CONNECTION') . '.new_updates_pds,id',
            'file' => 'required|max:50000|mimes:pdf',
        ]);
    
        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }
    
        DB::beginTransaction();

        try {
            $attachment = $request->file('file');
            $filePath = Storage::disk('employee_proof_attachment')->put(Auth::id(), $attachment);
            FileVault::disk('employee_proof_attachment')->encrypt($filePath);

            $groupId = $this->pdsEmployeeService->fetchGroupId($request->group_name, Auth::id());

            $attachmentHistory = EmployeeExtraFieldGroupHistory::create([
                'eefg_id' => $groupId,
                'file_attachment' => $filePath
            ]);

            foreach($request->ids as $id) {
                if ($id) {
                    $record = UpdatePDS::where('id', $id)->with('employeeExtraFieldColumn')->first();

                    if ($record) {
                        $record->update([
                            'eefg_history_id' => $attachmentHistory->id
                        ]);
                
                        $this->logCustomMessage(
                            'upload_proof_of_id',
                            $record,
                            Auth::user()->name . ' uploaded the proof of ID for ' . $record->employeeExtraFieldColumn->field_name,
                            $record,
                            PDSLogType::UPLOAD_PROOF,
                            new Activity()
                        );

                    }
                }
            }
            
            DB::commit();
            return response()->json([
                'text'  => 'File Uploaded Successfully',
            ]);
    
        } catch(Exception $e) {
            DB::rollBack();
            Storage::disk('employee_proof_attachment')->delete($filePath + '.enc');
            return response()->json([
                'errors'    => $e->getMessage()
            ], 500);
        }
    }

    public function download_proof(Request $request)
    {
        $record = UpdatePDS::where('id', $request->id)->first();

        $attachment = EmployeeExtraFieldGroupHistory::where('id', $record->eefg_history_id)->first();

        if (!$attachment) {
            return response()->json([
                'errors' => ["You haven't uploaded any file yet."],
            ], 404);
        }

        $filePath = $attachment->file_attachment . '.enc';

        $this->logCustomMessage(
            'download_proof_of_id',
            $attachment,
            Auth::user()->name . ' downloaded the proof of ID for ' . $attachment->employeeExtraFieldGroup->name,
            $attachment,
            PDSLogType::DOWNLOAD_PROOF,
            new Activity()
        );

        return response()->streamDownload(function () use ($filePath) {
            FileVault::disk('employee_proof_attachment')->streamDecrypt($filePath);
        }, '');
    }
}
