<?php

namespace Suiterus\Adg\Controllers\Mentorship;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Suiterus\Adg\Models\SM\Position;
use Suiterus\Adg\Models\SM\Department;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Mentorship\Mentorship;
use Suiterus\Adg\Models\Mentorship\MentorshipDate;
use Suiterus\Adg\Models\Mentorship\MentorshipSession;
use Suiterus\Adg\Services\Mentorship\MentorshipService;
use Suiterus\Adg\Requests\Mentorship\Session\UpdateRequest;
use Suiterus\Adg\Requests\Mentorship\Session\PaginateRequest;
use Suiterus\Adg\Requests\Mentorship\Session\StoreRequest;
use Suiterus\Adg\Requests\Mentorship\Session\UpdateDateRequest;

class MentorshipSessionController extends Controller
{
    use HasCustomLogs;

    private $mentorshipService;

    public function __construct(MentorshipService $mentorshipService)
    {
        $this->mentorshipService = $mentorshipService;
    }
    /**
     * The function `store` creates a mentorship session with associated entities and logs the
     * activity, handling exceptions with rollback and error messages.
     * 
     * @param Request request The `store` function you provided seems to handle the creation of
     * mentorship sessions based on the data received in the `` parameter. Here is a breakdown
     * of the key components in the function:
     * 
     * @return The code is returning a JSON response with a success message 'Mentorship created
     * successfully' if the mentorship creation process is successful. If an exception occurs during
     * the process, it catches the exception, rolls back the transaction, deletes any attachments
     * related to the mentorship, and then returns a JSON response with the error message from the
     * exception including the line number where the exception occurred.
     */
    public function store(StoreRequest $request)
    {
        DB::beginTransaction();

        try {
                $mentorship = Mentorship::find($request->mentorship_id);
    
                $entity = [
                    'division' => Division::class,
                    'department' => Department::class,
                    'employees' => User::class,
                    'position' => Position::class,
                ];

                for ($i = 0; $i < count($request->mentorship_dates['program_venue']); $i++) {
                
                    if ($request->mentorship_entity['entity_type'][$i] && $request->mentorship_entity['entity_id'][$i]) {
                        $mentorship->update([
                            'entity_type' => $entity[$request->mentorship_entity['entity_type'][$i]],
                            'entity_id' =>  $request->mentorship_entity['entity_id'][$i] == 'undefined' ? null : $request->mentorship_entity['entity_id'][$i],
                        ]);
                    } 
                        
                    $sessionNumber = MentorshipSession::where('mentorship_id', $mentorship->id)->max('session_number') ?? 0;
                    $nextSessionNumber = $sessionNumber + 1;
        
                    $session = MentorshipSession::create([
                        'mentorship_id' => $mentorship->id,
                        'session_number' => $nextSessionNumber,
                        'created_by' => Auth::id()
                    ]);

                    foreach ($request->mentorship_mentors[$i] as $mentors) {
                        foreach ($mentors as $mentor) {
                            $this->mentorshipService->addMentor(
                                $mentor,
                                $session
                            );
                        } 
                    }

                    foreach ($request->mentorship_mentees[$i] as $mentees) {
                        foreach ($mentees as $mentee) {
                            $this->mentorshipService->addMentee(
                                $mentee,
                                $session
                            );
                        } 
                    }

                    foreach ($request->mentorship_attachments[$i] ?? [] as $attachments) {
                        foreach ($attachments as $attachment) {
                            $this->mentorshipService->addAttachment(
                                $attachment,
                                $session
                            );
                        } 
                    }

                    $this->mentorshipService->addDate(
                        $request->mentorship_dates['program_venue'][$i],  
                        $request->mentorship_dates['start_date'][$i], 
                        $request->mentorship_dates['end_date'][$i], 
                        $request->mentorship_dates['start_time'][$i], 
                        $request->mentorship_dates['end_time'][$i], 
                        $session
                    );

                    $this->logCustomMessage('create_mentorship_session', $session, Auth::user()->name . ' has created a mentorship session named ' . $session->mentorship->program_title, $session, 'Create a mentorship session', new Activity());
                }
                
            DB::commit();

            return response()->json([
                'message' => 'Mentorship created successfully',
            ], 200);

        } catch (Exception $e) {
            DB::rollBack();
            Storage::disk('mentorship_attachment')->deleteDirectory($mentorship->id);
            return $e;
            return response()->json([
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * This PHP function paginates MentorshipSession records based on the provided PaginateRequest
     * parameters.
     * 
     * @param PaginateRequest request The `paginate` function takes a `PaginateRequest` object as a
     * parameter. This object likely contains information needed for pagination, such as the page count
     * and mentorship ID.
     * 
     * @return The code is returning a paginated list of MentorshipSession records based on the
     * provided PaginateRequest. It retrieves MentorshipSession records where the mentorship_id matches
     * the one provided in the request. It also eager loads relationships with mentorship, mentors,
     * invites, dates, and attachments. The number of records per page is determined by the page_count
     * parameter in the request, or it falls back to
     */
    public function paginate(PaginateRequest $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return MentorshipSession::where('mentorship_id', $request->mentorship_id)->with(['mentorship', 'mentors.mentor', 'invites.mentee', 'dates', 'attachments'])->paginate($paginate);
    }

    /**
     * The function retrieves a mentorship session with related mentorship, mentors, invites, dates,
     * and attachments based on the provided session ID.
     * 
     * @param Request request The `` parameter in the `show` function is an instance of the
     * `Illuminate\Http\Request` class in Laravel. It represents an HTTP request that is sent to the
     * server and contains information such as request parameters, headers, and more. In this context,
     * it is used to retrieve the `
     * 
     * @return The `show` function is returning a MentorshipSession model with the specified
     * relationships loaded. The MentorshipSession is retrieved based on the provided `session_id` from
     * the request. The relationships being loaded are `mentorship`, `mentors` with their associated
     * `mentor` models, `invites` with their associated `mentee` models, `dates`, and `attachments`.
     * The function returns
     */
    public function show (Request $request) {
        return MentorshipSession::where('id', $request->session_id)->with(['mentorship', 'mentors.mentor', 'invites.mentee', 'dates', 'attachments'])->first();
    }

    /**
     * The function updates a mentorship session in a PHP application, handling database transactions
     * and logging activities.
     * 
     * @param UpdateRequest request In the provided code snippet, the `update` function is responsible
     * for updating a mentorship session based on the data provided in the `UpdateRequest` object.
     * Let's break down the parameters used in this function:
     * 
     * @return A JSON response with a success message 'Mentorship session updated successfully' and a
     * status code of 200 if the update operation is successful. If an exception occurs during the
     * update process, a JSON response with an error message 'Something went wrong' and a status code
     * of 500 is returned.
     */
    public function updateMentorshipDate(UpdateDateRequest $request)
    {
        DB::beginTransaction();

        try {
            $session = MentorshipSession::findOrFail($request->session_id);
            $mentorshipDate = MentorshipDate::where('session_id', $session->id)->first();

            $oldMentorshipDate = clone $mentorshipDate;

            $this->mentorshipService->deleteMentorshipDate($session);
            $this->mentorshipService->addDate($request->program_venue, $request->start_date, $request->end_date, $request->start_time, $request->end_time, $session);

            $mentorshipDate->attributes = collect($mentorshipDate);
            $mentorshipDate->old = collect($oldMentorshipDate);

            $this->logCustomMessage('update_mentorship_date', $mentorshipDate, $mentorshipDate->session->mentorship->program_title . " program's session date/venue has been updated by " . Auth::user()->name, $mentorshipDate, 'Update a mentorship date', new Activity());

            DB::commit();

            return response()->json([
                'message' => 'Mentorship date updated successfully',
            ], 200);

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Something went wrong',
            ], 500);
        }
    }

    public function updateSession(UpdateRequest $request) 
    {
        DB::beginTransaction();

        try {
            $session = MentorshipSession::findOrFail($request->session_id);

            $oldSession = clone $session;

            $session->update([
                'key_points' => $request->key_points,
                'process_observation' => $request->process_observation,
                'updated_by' => Auth::id(),
                'updated_at' => Carbon::now()
            ]);

            $session->attributes = collect($session);
            $session->old = collect($oldSession);

            $this->logCustomMessage('update_mentorship_session', $session, Auth::user()->name . "updated the " . $session->id, $session, 'Update a mentorship session', new Activity());

            DB::commit();

            return response()->json([
                'message' => 'Mentorship session updated successfully',
            ], 200);
            
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Something went wrong',
            ], 500);
        }
        
    }


}
