<?php

namespace Suiterus\Adg\Controllers\Mentorship;


use Exception;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Mentorship\MentorshipReport;
use Suiterus\Adg\Requests\Mentorship\Report\StoreRequest;
use Suiterus\Adg\Requests\Mentorship\Report\UpdateRequest;

class MentorshipReportController extends Controller
{
    use HasCustomLogs;

    /**
     * The `store` function creates a mentorship report, logs the action, and handles database
     * transactions in a Laravel application.
     * 
     * @param StoreRequest request The `store` function you provided is responsible for storing a
     * Mentorship Report in the database. It begins a database transaction, attempts to create a new
     * Mentorship Report using the data from the request, logs a custom message related to the creation
     * of the report, and then commits the transaction if successful.
     * 
     * @return The `store` function is returning a JSON response with a success message if the
     * mentorship report is created successfully. If an exception occurs during the creation process,
     * it will return a JSON response with an error message.
     */
    public function store(StoreRequest $request) 
    {
        DB::beginTransaction();

        try {
            $report = $request->all();

            $report = MentorshipReport::create($report);

            $this->logCustomMessage('create_mentorship_report', $report, Auth::user()->name . ' has created a mentorship report for ' . $report->mentorship->program_title , $report, 'Create a mentorship report', new Activity());

            DB::commit();

            return response()->json([
                'message' => 'Mentorship report created successfully',
            ], 200);

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Something went wrong',
            ], 500);
        }
    }

    /**
     * The function updates a Mentorship Report in a database transaction and logs the update action.
     * 
     * @param UpdateRequest request The `update` function you provided is responsible for updating a
     * Mentorship Report based on the data provided in the `UpdateRequest` object. It begins a database
     * transaction, attempts to update the report with the new data, and then commits the transaction
     * if successful. If an error occurs during the update process
     * 
     * @return A JSON response with a success message 'Successfully updated' and a status code of 200
     * if the update operation is successful. If an exception occurs during the update process, a JSON
     * response with an error message 'Something went wrong' and a status code of 500 is returned.
     */
    public function update(UpdateRequest $request) 
    {
        DB::beginTransaction();

        try {
            $report = MentorshipReport::where('id', $request->id)->first();

            $oldReport = clone $report;
            $report->update([
                'goal_details' => $request->goal_details,
                'goal_timeline' => $request->goal_timeline,
                'goal_remarks' => $request->goal_remarks,
                'realities_details' => $request->realities_details,
                'realities_timeline' => $request->realities_timeline,
                'realities_remarks' => $request->realities_remarks,
                'options_details' => $request->options_details,
                'options_timeline' => $request->options_timeline,
                'options_remarks' => $request->options_remarks,
                'way_forward_details' => $request->way_forward_details,
                'way_forward_timeline' => $request->way_forward_timeline,
                'way_forward_remarks' => $request->way_forward_remarks,
                'updated_by' => Auth::id(),
            ]);

            $report->attributes = collect($report);
            $report->old = collect($oldReport);

            $this->logCustomMessage('update_mentorship_report', $report, $report->mentorship()->first()->program_title . " program's report has been updated by " . Auth::user()->name, $report, 'Update a mentorship report', new Activity());

            DB::commit();
            return response()->json([
                'message' => 'Successfully updated',
            ], 200);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Something went wrong',
            ], 500);
        }
    }

    /**
     * The function retrieves a MentorshipReport with its associated mentorship based on the provided
     * mentorship_id from the request.
     * 
     * @param Request request The `Request ` parameter in the `show` function is a parameter of
     * type `Illuminate\Http\Request`. This parameter is used to access the incoming HTTP request data,
     * such as form input, headers, and other request information. In this case, the function is
     * retrieving a MentorshipReport based
     * 
     * @return The `show` function is returning a MentorshipReport model where the mentorship_id
     * matches the value from the request, along with the related `mentorship` model. The `first()`
     * method is used to retrieve the first result that matches the query conditions.
     */
    public function show(Request $request) {
        return MentorshipReport::where('mentorship_id', $request->mentorship_id)->with(['mentorship'])->first();
    }
}
