<?php

namespace Suiterus\Adg\Controllers\Mentorship;

use Exception;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use App\Enums\Mentorship\MentorshipInviteStatus;
use Suiterus\Adg\Models\Mentorship\MentorshipInvite;
use Suiterus\Adg\Requests\Mentorship\Invite\ListRequest;
use Suiterus\Adg\Requests\Mentorship\Invite\UpdateRequest;
use Suiterus\Adg\Requests\Mentorship\Invite\PaginateRequest;

class MentorshipInviteController extends Controller
{

    use HasCustomLogs;
    public function paginate(PaginateRequest $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return MentorshipInvite::with(['session', 'session.mentorship', 'session.dates', 'session.mentors.mentor', 'session.invites.mentee', 'session.attachments'])->when($request->user_id != null && isset($request->user_id) && $request->user_id != '', function ($query) use ($request) {
            $query->where('mentee_id', $request->user_id);
        })->paginate($paginate);
    }

    public function list(ListRequest $request)
    {
        return MentorshipInvite::when($request->status != null && isset($request->status) && $request->status != '', function($query) use($request){
            $query->where('status', $request->status);
        })->whereHas('session', function ($query) use ($request) {
            $query->where('mentorship_id', $request->mentorship_id);
        })->whereHas('mentee', function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->mentee_name . '%');
        })->whereHas('mentee.employeeMetaInfo', function ($query) use ($request) {
            $query->when($request->employee_id != null && isset($request->employee_id) && $request->employee_id != '', function ($query) use ($request) {
                if (count($request->employee_id) > 0) {
                    $query->whereIn('employee_id', $request->employee_id);
                } else {
                    $query->where('employee_id', $request->employee_id);
                }
            });
            $query->when($request->position != null && isset($request->position) && $request->position != '', function ($query) use ($request) {
                $query->where('position_id', $request->position);
            });
            $query->when($request->department != null && isset($request->department) && $request->department != '', function ($query) use ($request) {
                $query->where('department_id', $request->department);
            });
            $query->when($request->division != null && isset($request->division) && $request->division != '', function ($query) use ($request) {
                $query->where('division_id', $request->division);
            });
        })->with(['mentee'])->get();
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateRequest $request, $id)
    {
        DB::beginTransaction();
        try {
            $mentorshipInvite = MentorshipInvite::where('id', $id)->first();

            if ($request->status == $mentorshipInvite->status) {
                return response()->json([
                    'message' => 'Nothing to update',
                ], 200);
            }

            $oldMentorshipInvite = clone $mentorshipInvite;
            $mentorshipInvite->update([
                'status' => $request->status
            ]);

            $mentorshipInvite->attributes = collect($mentorshipInvite);
            $mentorshipInvite->old = collect($oldMentorshipInvite);

            $statusMessage = [
                MentorshipInviteStatus::ACCEPTED => 'has been accepted',
                MentorshipInviteStatus::DECLINED => 'has been declined',
            ];

            $this->logCustomMessage('update_mentorship_invite', $mentorshipInvite, $mentorshipInvite->session->mentorship()->first()->program_title . " program {$statusMessage[$request->status]} by mentee " . Auth::user()->name, $mentorshipInvite, 'Update a mentorship invite', new Activity());

            DB::commit();
            return response()->json([
                'message' => 'Successfully updated',
            ], 200);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => $e->getMessage(),
            ], 500);
        }

    }
}
