<?php

namespace Suiterus\Adg\Controllers\Loan;

use Exception;
use App\Models\User;
use App\Models\UserLoan;
use Illuminate\Http\Request;
use App\Enums\Log\UserLoanLogType;
use App\Http\Classes\NotificationMessage;
use App\Models\LoanPaymentHistory;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;

class UserLoanController extends Controller
{

    use HasCustomLogs;
    
    public function create_user_loan(Request $req) {

        $valid = Validator::make($req->all(),[
            'userID' => 'integer|required|exists:mysql.users,id',
            'loanID' => 'integer|required|exists:adg_db.loans,id',
            'amortizationAmount' => 'numeric|required',
            'submissionDate' => 'required|date',
            'from' => 'required|date',
            'to' => 'required|date',
            'paymentDurationMonths' => 'numeric|required',
            'outstandingBalance' => 'numeric|required'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try {
            $userLoan = UserLoan::updateOrCreate(
            [
                ['id', $req->id],
            ],
            [
                'user_id' => $req->userID,
                'loan_id' => $req->loanID,
                'amount' => $req->amortizationAmount,
                'purpose_of_loan' => $req->purpose,
                'number_of_payments' => $req->paymentDurationMonths,
                'outstanding_balance' => $req->outstandingBalance,
                'submission_date' => $req->submissionDate,
                'status' => $req->status ? $req->status : 4, //approved as default status
                'from' => $req->from,
                'to' => $req->to,
                'created_by' => Auth::id(),
            ]);

            $this->logCustomMessage(
                'create_update_loan',
                $userLoan,
                Auth::user()->name . ' created/updated a loan for ' . $userLoan->user->name,
                $userLoan,
                UserLoanLogType::CREATE_UPDATE,
                new Activity()
            );

            isset($req->id) ? NotificationMessage::notifySenderAndReceiver('Update Loan', $req->userID, Auth::id()) : NotificationMessage::notifySenderAndReceiver('Create Loan', $req->userID, Auth::id());

            DB::commit();
            return response()->json([
                'text'  => 'Loan application created/updated successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }

       
    }
    
    public function search_user_loan(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return UserLoan::when($req->loanType != '' && $req->loanType != null, function($query) use($req){
            return $query->where('loan_id', $req->loanType);
        })->when($req->submissionDate != '' && $req->submissionDate != null, function($query) use($req){
            return $query->where('submission_date', $req->submissionDate);
        })->when($req->loanStatus != '' && $req->loanStatus != null, function($query) use($req){
            return $query->where('status', $req->loanStatus);
        })->when($req->loanStatusNotIncluded != '' && $req->loanStatusNotIncluded != null, function($query) use($req){
            return $query->where('status', '!=' , $req->loanStatusNotIncluded);
        })->whereHas('user', function($query) use($req) {
            $query->whereHas('employeeMetaInfo', function($query) use($req) {
                $query->when($req->employee_id != null && isset($req->employee_id) && $req->employee_id != '', function($query) use($req) {
                    if(count($req->employee_id) > 0){
                        $query->whereIn('employee_id', $req->employee_id);
                    } else{
                        $query->where('employee_id', $req->employee_id);
                    }
                });
            });
            
            $query->when($req->employeeName != '' && $req->employeeName != null, function($query) use($req) {
                return $query->where('name', 'LIKE', '%' . $req->employeeName . '%');
            });
        })->paginate($paginate);
    }

    public function update_user_loan(Request $req){
        $valid = Validator::make($req->all(),[
            'id' => 'integer|required',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try {
            $userLoan = UserLoan::where(
                [
                    ['id', $req->id],
                ]
            )->update([
                'status' => 4 // approved
            ]);

            $updated = UserLoan::find($req->id);

            $updated->old = collect($userLoan);
            $updated->attributes = collect($updated);
                    
            $this->logCustomMessage(
                'update_loan_application',
                $updated,
                Auth::user()->name . ' updated the loan of ' . $userLoan->user->name,
                $updated,
                UserLoanLogType::UPDATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Loan application updated successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function approved_loans_list(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  UserLoan::where('status', 4)->whereHas('user', function($query) use($request){
                
                $query->select('id','name')->with(['employeeMetaInfo' => function($query){
                    $query->select('id','user_id','employee_id');
                }]);

                $query->when($request->employeeName != '' && $request->employeeName != null, function($query) use($request) {
                    return $query->where('name', 'LIKE', '%' . $request->employeeName . '%');
                });

            })->paginate($paginate)
        ]);
    } 
    
    public function paginate_loan_transaction_history(Request $request){
        $status = 1; // Approved
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return LoanPaymentHistory::where([
            ['user_has_loan_id', $request->user_loan_id], ['status', $status]
        ])->with([
            'payrollEarningDeduction', 'userHasLoan'
        ])->paginate($paginate);
    }

    public function loan_summary(Request $request){
       return UserLoan::findOrFail($request->id);
    }

    public function get_loan_transaction_history(Request $request){
        $status = 1; // Approved
        return LoanPaymentHistory::where([
            ['user_has_loan_id', $request->user_loan_id], ['status', $status]
        ])->with([ 'payrollEarningDeduction', 'userHasLoan'])->get();
    }

}
