<?php


namespace Suiterus\Adg\Controllers\Services;

use Carbon\Carbon;
use App\Models\User;
use App\Enums\Constants;
use App\Enums\LeaveType;
use UnexpectedValueException;
use App\Jobs\IncreaseLeaveBalance;
use Suiterus\Adg\Models\SM\Sector;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Controllers\LeaveManagement\Table\Credits;
use Suiterus\Adg\Models\LeaveManagement\ForcedLeaveCounter;
use Suiterus\Adg\Models\LeaveManagement\LeaveCreditHistory;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveCreditEarning;
use Suiterus\Adg\Models\LeaveManagement\LeaveType as LeaveManagementLeaveType;

class LeaveCreditService
{

    public function fetchBalance($user_id, $leave_type)
    {
        return LeaveBalance::where('user_id', $user_id)->where('leave_type_id', $leave_type)->first();
    }

    // Leave Monetization

    public function monetizeLeave($leave, $creator)
    {
        // Get balances
        $sick_leave_balance = LeaveBalance::where('user_id', $leave->user_id)->where('leave_type_id', LeaveType::SICK_LEAVE)->first();
        $vacation_leave_balance = LeaveBalance::where('user_id', $leave->user_id)->where('leave_type_id', LeaveType::VACATION_LEAVE)->first();
        $forced_leave_balance = LeaveBalance::where('user_id', $leave->user_id)->where('leave_type_id', LeaveType::FORCED_LEAVE)->first();

        $sector = Sector::where('status', 'active')->first();
        $salary = $leave->user->salary;

        // Get Salary
        if ($sector->name == 'public') {
            $salary = str_replace(",", '', $salary->publicSalary->salaryGrade->value);
        } else {
            $salary = str_replace(",", '', $salary->privateSalary->salary);
        }

        $constant = Constants::LEAVE_MONETIZATION;

        $forced_deduction = 5.0;

        if ($forced_leave_balance->balance < 5) {
            throw new UnexpectedValueException('Insufficient forced leave balance to monetize');
        }

        $monetization = $leave->monetization;
        if ($monetization->type == 1) {
            $vl_days = $monetization->vacation_days;
            $sl_days = $monetization->sick_days;
            $vacation_deduction = (float) $vl_days;
            $sick_deduction = (float) $sl_days;
            $sick_result = $sick_leave_balance->balance - $sick_deduction;
            $vacation_deduction = $forced_deduction + $vacation_deduction;
            $vacation_result = $vacation_leave_balance->balance - $vacation_deduction;
            $total_points =  $vacation_deduction + $sick_deduction;

            if ($vacation_result < 5 || $sick_result < 5) {
                throw new UnexpectedValueException('Cannot monetize with insufficient balance after deduction');
            }

            if (($vl_days + $sl_days) < 10) {
                throw new UnexpectedValueException('Monetization in days should have a minimum of 10 days');
            }

            if (($vl_days + $sl_days) > 30) {
                throw new UnexpectedValueException('Monetization in days should not exceed 30 days');
            }

            $computed = (float)$salary * $constant * $total_points;

            $monetization->monetized_leave()->create([
                'deducted_vacation_leave_balance' => $vacation_deduction,
                'deducted_sick_leave_balance'      => $sick_deduction,
                'vacation_leave_balance'            => $vacation_result,
                'sick_leave_balance'                => $sick_result,
                'total_points'                      => $total_points,
                'monetized_amount'                  => $computed
            ]);
        } else {
            $sick_deduction = (float) floor($sick_leave_balance->balance / 2);
            $vacation_deduction = (float) floor(($vacation_leave_balance->balance - $forced_deduction) / 2);
            $sick_result = $sick_leave_balance->balance - $sick_deduction;
            $vacation_result = $vacation_leave_balance->balance - $vacation_deduction;

            if ($vacation_result < 5 || $sick_result < 5 || $forced_leave_balance->balance < 5) {
                throw new UnexpectedValueException('Cannot monetize with insufficient balance after deduction');
            }

            $total_points =  $sick_deduction + $vacation_deduction;
            $computed = (float)$salary * $constant * $total_points;

            $monetization->monetized_leave()->create([
                'deducted_vacation_leave_balance' => $vacation_deduction,
                'deducted_sick_leave_balance'      => $sick_deduction,
                'vacation_leave_balance'            => $vacation_result,
                'sick_leave_balance'                => $sick_result,
                'total_points'                      => $total_points,
                'monetized_amount'                  => $computed
            ]);
        }

        $this->deductBalance($leave, $creator, $sick_deduction, $sick_leave_balance);
        $this->deductBalance($leave, $creator, $vacation_deduction, $vacation_leave_balance);
        $this->deductBalance($leave, $creator, $forced_deduction, $forced_leave_balance);

        return [
            'vacation_deduction' => $vacation_deduction,
            'sick_deduction' => $sick_deduction
        ];
    }

    // Compute Terminal Leave

    public function computeTerminalLeave($leave, $creator)
    {

        $vacation_leave_balance = LeaveBalance::where('user_id', $leave->user_id)->where('leave_type_id', LeaveType::VACATION_LEAVE)->first();
        $sick_leave_balance = LeaveBalance::where('user_id', $leave->user_id)->where('leave_type_id', LeaveType::SICK_LEAVE)->first();
        $forced_leave_balance = LeaveBalance::where('user_id', $leave->user_id)->where('leave_type_id', LeaveType::FORCED_LEAVE)->first();
        $special_leave_balance = LeaveBalance::where('user_id', $leave->user_id)->where('leave_type_id', LeaveType::SPECIAL_LEAVE)->first();

        $vacation_balance = $vacation_leave_balance->balance;
        $sick_balance = $sick_leave_balance->balance;
        $forced_balance = $forced_leave_balance->balance;
        $special_balance = $special_leave_balance->balance;

        $sector = Sector::where('status', 'active')->first();
        $salary = $leave->user->salary;

        // Get Salary
        if ($sector->name == 'public') {
            $salary = str_replace(",", '', $salary->publicSalary->salaryGrade->value);
        } else {
            $salary = str_replace(",", '', $salary->privateSalary->salary);
        }

        $constant = Constants::LEAVE_MONETIZATION;

        // Formula: Salary x (VL + SL) x constant
        $terminal_amount = $salary * ($vacation_balance + $sick_balance) * $constant;

        $leave->terminal_leave_benefit()->create([
            'computed_amount'   => $terminal_amount
        ]);

        $this->deductBalance($leave, $creator, $vacation_balance, $vacation_leave_balance);
        $this->deductBalance($leave, $creator, $sick_balance, $sick_leave_balance);
        $this->deductBalance($leave, $creator, $forced_balance, $forced_leave_balance);
        $this->deductBalance($leave, $creator, $special_balance, $special_leave_balance);
    }

    // Increase / Decrease Balance
    public function deductBalance($leave, $creator, $credits, $balance = null)
    {
        $user_id = $leave->user_id;
        $period = date('Y-m-t');
        if ($balance == null) {
            $balance = LeaveBalance::where('user_id', $user_id)->where('leave_type_id', $leave->leave_type_id)->first();
        }

        $vacation_balance = LeaveBalance::where('user_id', $user_id)->where('leave_type_id', LeaveType::VACATION_LEAVE)->first();
        $sick_balance = LeaveBalance::where('user_id', $user_id)->where('leave_type_id', LeaveType::SICK_LEAVE)->first();
        $special_privilege_balance = LeaveBalance::where('user_id', $user_id)->where('leave_type_id', LeaveType::SPECIAL_LEAVE)->first();

        $decreased_vacation_balance = $vacation_balance->balance; // ? can we removed this variable?
        $decreased_sick_balance = null;
        $decreased_balance = null;
        $decreased_total_points = null;
        $difference = null;
        $remaining_bal = null;

        if (strtolower($balance->leave_type->title) == 'mandatory/forced leave') { // forced leave - vacation leave
            /**
             * ? Can we remove this condition as it is connected to decreased_vacation_balance?
             */

            // if ($decreased_vacation_balance < 0) {
            //     $lwop_points = abs($decreased_vacation_balance);

            //     $leave->leave_without_pay()->create([
            //         'deducted_points'   => $lwop_points,
            //     ]);

            //     $decreased_vacation_balance = 0;
            // }

            // $vacation_balance->balance = $decreased_vacation_balance;
            // $vacation_balance->save();

            $decreased_balance = $balance->balance - $credits;

            $forcedLeaveCounter = ForcedLeaveCounter::where('user_id', $user_id)->first();

            $forcedLeaveCounter->update([
                'counter' => ($forcedLeaveCounter->counter - $credits)  <= 0 ? 0 : ($forcedLeaveCounter->counter - $credits),
            ]);

            if ($decreased_balance < 0) {
                $lwop_points = abs($decreased_balance);

                $leave->leave_without_pay()->create([
                    'deducted_points'   => $lwop_points,
                ]);

                $decreased_balance = 0;
            }
            $this->recordDeduction($period, $leave, $credits, $decreased_balance, $decreased_vacation_balance, $decreased_sick_balance, $decreased_total_points, $creator, $vacation_balance, $sick_balance, $special_privilege_balance, $balance, $difference, $remaining_bal);
            $balance->balance = $decreased_balance;
            $balance->save();
        } else if ($balance->leave_type_id == 3) { // sick leave - vacation leave
            $decreased_total_points = $sick_balance->balance - $credits;
            $difference = $decreased_total_points < 0 ? 0 : $decreased_total_points;

            if ($sick_balance->balance < $credits) {
                $remaining_bal = abs($decreased_total_points);
                $decreased_vacation_balance = $vacation_balance->balance - $remaining_bal;
                if ($decreased_vacation_balance < 0) {
                    $lwop_points = abs($decreased_vacation_balance);

                    $leave->leave_without_pay()->create([
                        'deducted_points'   => $lwop_points,
                    ]);

                    $decreased_vacation_balance = 0;
                }
                $vacation_balance->balance = $decreased_vacation_balance;
                $vacation_balance->save();
            }
            $this->recordDeduction($period, $leave, $credits, $decreased_balance, $decreased_vacation_balance, $decreased_sick_balance, $decreased_total_points, $creator, $vacation_balance, $sick_balance, $special_privilege_balance, $balance, $difference, $remaining_bal);
            $sick_balance->balance = $difference;
            $sick_balance->save();
        }
        else if ($balance->leave_type_id == 6) { // special privilege leave - vacation leave
            $decreased_total_points = $special_privilege_balance->balance - $credits;
            $difference = $decreased_total_points < 0 ? 0 : $decreased_total_points;

            if ($special_privilege_balance->balance < $credits) {
                $remaining_bal = abs($decreased_total_points);
                $decreased_vacation_balance = $vacation_balance->balance - $remaining_bal;
                if ($decreased_vacation_balance < 0) {
                    $lwop_points = abs($decreased_vacation_balance);

                    $leave->leave_without_pay()->create([
                        'deducted_points'   => $lwop_points,
                    ]);

                    $decreased_vacation_balance = 0;
                }
                $vacation_balance->balance = $decreased_vacation_balance;
                $vacation_balance->save();
            }
            $this->recordDeduction($period, $leave, $credits, $decreased_balance, $decreased_vacation_balance, $decreased_sick_balance, $decreased_total_points, $creator, $vacation_balance, $sick_balance, $special_privilege_balance, $balance, $difference, $remaining_bal);
            $special_privilege_balance->balance = $difference;
            $special_privilege_balance->save();
        } else {
            $decreased_balance = $balance->balance - $credits;

            if ($decreased_balance < 0) {
                $lwop_points = abs($decreased_balance);

                $leave->leave_without_pay()->create([
                    'deducted_points'   => $lwop_points,
                ]);

                $decreased_balance = 0;
            }

            if (strtolower($balance->leave_type->title) == 'vacation leave') {
                $forcedLeaveCounter = ForcedLeaveCounter::where('user_id', $user_id)->first();
                $forcedLeaveCounter->update([
                    'counter' => ($forcedLeaveCounter->counter - $credits)  <= 0 ? 0 : ($forcedLeaveCounter->counter - $credits)
                ]);
            }

            $this->recordDeduction($period, $leave, $credits, $decreased_balance, $decreased_vacation_balance, $decreased_sick_balance, $decreased_total_points, $creator, $vacation_balance, $sick_balance, $special_privilege_balance, $balance, $difference, $remaining_bal);
            $balance->balance = $decreased_balance;
            $balance->save();
        }
    }

    // Generate Monthly Balance
    public function generateMonthlyBalance($isResetPoints = false)
    {

        $employees = User::whereHas('employeeMetaInfo')->whereHas('leave_balance.leave_type', function ($query) {
            $monthly = 1;
            $query->where('increase_type', $monthly);
        })->get();

        foreach ($employees as $employee) {
            foreach ($employee->leave_balance as $employeeBalance) {
                dispatch(new IncreaseLeaveBalance($employee->id, $employeeBalance->leave_type_id, $isResetPoints));
            }
        }
    }

    // Generate Annual Leave Balance
    public function generateAnnualBalance()
    {

        $employees = User::whereHas('employeeMetaInfo')->whereHas('leave_balance.leave_type', function ($query) {
            $yearly = 2;
            $query->where('increase_type', $yearly);
        })->get();

        foreach ($employees as $employee) {
            foreach ($employee->leave_balance as $employeeBalance) {
                dispatch(new IncreaseLeaveBalance($employee->id, $employeeBalance->leave_type_id, true));
            }
        }

        $this->generateMonthlyBalance(true);
    }

    // Initialize balances (to use when testing)
    public function initializeBalances()
    {
        $employees =  User::whereHas('employeeMetaInfo')->get();
        $leave_types = LeaveManagementLeaveType::all();
        $balances = [];
        foreach ($employees as $employee) {
            if ($employee->leave_balance()->count() == 0) {
                foreach ($leave_types as $type) {
                    $balance = [
                        'user_id'   => $employee->id,
                        'leave_type_id' => $type->id,
                        'balance'       => 0,
                        'created_by'    => 1,
                        'updated_by'    => 1,
                        'created_at'    => now(),
                        'updated_at'    => now()
                    ];
                    array_push($balances, $balance);
                }
            }
        }
        LeaveBalance::insert($balances);
    }

    // Initialize balance of employee when record is created
    public function initializeEmployeeBalance($employee)
    {
        $leave_types_monthly = LeaveManagementLeaveType::where('increase_type', 1)->get();
        $leave_type_annual = LeaveManagementLeaveType::where('increase_type', 2)->get();

        $period_last_day = date('t');
        $date_hired_day = date('d', strtotime($employee->employeeMetaInfo->date_hired));
        $days = $period_last_day - $date_hired_day;
        $credits_object = new Credits;
        $credits_days = $credits_object->getCreditsEarnedDaily();
        $initial_monthly_balance = $credits_days[$days];
        if ($employee->leave_balance()->count() == 0) {
            foreach ($leave_types_monthly as $type) {
                LeaveBalance::create([
                    'user_id'   => $employee->id,
                    'leave_type_id' => $type->id,
                    'balance'       => 0,
                    'created_by'    => 1,
                    'updated_by'    => 1,
                ]);
            }
            foreach ($leave_type_annual as $type) {
                $annual_increase = $type->default_increase;
                if ($employee->employeeMetaInfo->employeeType->title == 'Probationary') {
                    $annual_increase = 0;
                }
                LeaveBalance::create([
                    'user_id'   => $employee->id,
                    'leave_type_id' => $type->id,
                    'balance'       => $annual_increase,
                    'created_by'    => 1,
                    'updated_by'    => 1,
                ]);
            }
        }
    }

    // Record Earnings and Deductions

    private function recordDeduction($period, $leave, $credits, $decreased_balance, $decreased_vacation_balance, $decreased_sick_balance, $decreased_total_points, $creator, $vacation_balance, $sick_balance, $special_privilege_balance, $balance, $difference, $remaining_bal)
    {

        $deducted = 2;
        if (strtolower($leave->leaveType->title) == 'mandatory/forced leave') {
            /**
             * ? Can we remove this code as it is connected to decreased_vacation_balance
             */
            // $leave->credit_deductions()->create([
            //     'period'            => $period,
            //     'remarks'           => $vacation_balance->leave_type->title,
            //     'credits'           => $credits,
            //     'resulting_balance' => $decreased_vacation_balance,
            //     'created_by'        => $creator,
            //     'updated_by'        => $creator
            // ]);

            // LeaveCreditHistory::create([
            //     'user_id' => $leave->user_id,
            //     'leave_type_id' => $vacation_balance->leave_type->id,
            //     'credits' => $credits,
            //     'resulting_balance' => $decreased_vacation_balance,
            //     'leave_adjustment_type' => $deducted,
            //     'remarks' => $vacation_balance->leave_type->title,
            //     'period' => Carbon::now(),
            //     'created_by' => Auth::id(),
            // ]);

            $leave->credit_deductions()->create([
                'period'            => $period,
                'remarks'           => $balance->leave_type->title,
                'credits'           => $credits,
                'resulting_balance' => $decreased_balance,
                'created_by'        => $creator,
                'updated_by'        => $creator
            ]);
            LeaveCreditHistory::create([
                'user_id' => $leave->user_id,
                'leave_type_id' => $balance->leave_type->id,
                'credits' => $credits,
                'resulting_balance' => $decreased_balance,
                'leave_adjustment_type' => $deducted,
                'remarks' => $balance->leave_type->title,
                'period' => Carbon::now(),
                'created_by' => Auth::id(),
            ]);
        } else if (strtolower($leave->leaveType->title) == 'sick leave') {
            if ($sick_balance->balance < $credits) {
                $leave->credit_deductions()->create([
                    'period'            => $period,
                    'remarks'           => $sick_balance->leave_type->title,
                    'credits'           => $sick_balance->balance,
                    'resulting_balance' => $difference,
                    'created_by'        => $creator,
                    'updated_by'        => $creator
                ]);
                $leave->credit_deductions()->create([
                    'period'            => $period,
                    'remarks'           => $vacation_balance->leave_type->title,
                    'credits'           => $remaining_bal,
                    'resulting_balance' => $decreased_vacation_balance,
                    'created_by'        => $creator,
                    'updated_by'        => $creator
                ]);
                LeaveCreditHistory::create([
                    'user_id' => $leave->user_id,
                    'leave_type_id' => $sick_balance->leave_type->id,
                    'credits' => $sick_balance->balance,
                    'resulting_balance' => $difference,
                    'leave_adjustment_type' => $deducted,
                    'remarks' => $sick_balance->leave_type->title,
                    'period' => Carbon::now(),
                    'created_by' => Auth::id(),
                ]);
                LeaveCreditHistory::create([
                    'user_id' => $leave->user_id,
                    'leave_type_id' => $vacation_balance->leave_type->id,
                    'credits' => $remaining_bal,
                    'resulting_balance' => $decreased_vacation_balance,
                    'leave_adjustment_type' => $deducted,
                    'remarks' => 'Decreased from ' . $sick_balance->leave_type->title,
                    'period' => Carbon::now(),
                    'created_by' => Auth::id(),
                ]);
            }
            else {
                $leave->credit_deductions()->create([
                    'period'            => $period,
                    'remarks'           => $balance->leave_type->title,
                    'credits'           => $credits,
                    'resulting_balance' => $difference,
                    'created_by'        => $creator,
                    'updated_by'        => $creator
                ]);
                LeaveCreditHistory::create([
                    'user_id' => $leave->user_id,
                    'leave_type_id' => $balance->leave_type->id,
                    'credits' => $credits,
                    'resulting_balance' => $difference,
                    'leave_adjustment_type' => $deducted,
                    'remarks' => $balance->leave_type->title,
                    'period' => Carbon::now(),
                    'created_by' => Auth::id(),
                ]);
            }
        } else if ($balance->leave_type_id == 6) {
            if ($special_privilege_balance->balance < $credits) {
                $leave->credit_deductions()->create([
                    'period'            => $period,
                    'remarks'           => $special_privilege_balance->leave_type->title,
                    'credits'           => $special_privilege_balance->balance,
                    'resulting_balance' => $difference,
                    'created_by'        => $creator,
                    'updated_by'        => $creator
                ]);
                $leave->credit_deductions()->create([
                    'period'            => $period,
                    'remarks'           => $vacation_balance->leave_type->title,
                    'credits'           => $remaining_bal,
                    'resulting_balance' => $decreased_vacation_balance,
                    'created_by'        => $creator,
                    'updated_by'        => $creator
                ]);
                LeaveCreditHistory::create([
                    'user_id' => $leave->user_id,
                    'leave_type_id' => $special_privilege_balance->leave_type->id,
                    'credits' => $special_privilege_balance->balance < 0 ? 0 : $special_privilege_balance->balance,
                    'resulting_balance' => $difference,
                    'leave_adjustment_type' => $deducted,
                    'remarks' => $special_privilege_balance->leave_type->title,
                    'period' => Carbon::now(),
                    'created_by' => Auth::id(),
                ]);
                LeaveCreditHistory::create([
                    'user_id' => $leave->user_id,
                    'leave_type_id' => $vacation_balance->leave_type->id,
                    'credits' => $remaining_bal,
                    'resulting_balance' => $decreased_vacation_balance,
                    'leave_adjustment_type' => $deducted,
                    'remarks' => 'Deduction from Special Privilege Leave',
                    'period' => Carbon::now(),
                    'created_by' => Auth::id(),
                ]);
            }
            else {
                $leave->credit_deductions()->create([
                    'period'            => $period,
                    'remarks'           => $balance->leave_type->title,
                    'credits'           => $credits,
                    'resulting_balance' => $difference,
                    'created_by'        => $creator,
                    'updated_by'        => $creator
                ]);
                LeaveCreditHistory::create([
                    'user_id' => $leave->user_id,
                    'leave_type_id' => $balance->leave_type->id,
                    'credits' => $credits,
                    'resulting_balance' => $difference,
                    'leave_adjustment_type' => $deducted,
                    'remarks' => $balance->leave_type->title,
                    'period' => Carbon::now(),
                    'created_by' => Auth::id(),
                ]);
            }
        } else {
            $leave->credit_deductions()->create([
                'period'            => $period,
                'remarks'           => $balance->leave_type->title,
                'credits'           => $credits,
                'resulting_balance' => $decreased_balance,
                'created_by'        => $creator,
                'updated_by'        => $creator,
            ]);
            LeaveCreditHistory::create([
                'user_id' => $leave->user_id,
                'leave_type_id' => $balance->leave_type->id,
                'credits' => $credits,
                'resulting_balance' => $decreased_balance,
                'leave_adjustment_type' => $deducted,
                'remarks' => $balance->leave_type->title,
                'period' => Carbon::now(),
                'created_by' => Auth::id(),
            ]);
        }
    }

    private function recordIncrease($period, $remarks, $credits, $increased_balance, $balance)
    {
        LeaveCreditEarning::create([
            'leave_balance_id'  => $balance->id,
            'period'            => $period,
            'remarks'           => $remarks,
            'credits'           => $credits,
            'resulting_balance' => $increased_balance,
        ]);
    }
}
