<?php

namespace Suiterus\Adg\Controllers\LeaveManagement;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\LeaveManagement\LeaveEligibleGroup;
use Suiterus\Adg\Models\LeaveManagement\LeaveEligibleUser;
use Suiterus\Adg\Models\LeaveManagement\LeaveType;
use Suiterus\Adg\Requests\LeaveTypes\StoreUpdateRequest;
use Suiterus\Adg\Services\LeaveManagement\LeaveEligibleService;
use Suiterus\Adg\Models\Activity\Activity;
use App\Traits\Logs\HasCustomLogs;
use App\Enums\Log\LeaveManagementLogType;

class LeaveTypeController extends Controller
{
    use HasCustomLogs;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return LeaveType::get();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreUpdateRequest $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $leaveType = LeaveType::create([
                'title' => $request->title,
                'description' => $request->description,
                'initial_points' => $request->initial_points,
                'file_attachment' => $request->file_attachment,
                'remark' => $request->remark,
                'default_increase' => $request->leave_points,
                'increase_type' => $request->increase_type,
                'status' => $request->status,
                'type' => $request->type,
                'created_by' => Auth::id(),
                'updated_by' => Auth::id(),
            ]);

            (new LeaveEligibleService)->assignLeaveType($request->group_ids, $request->user_ids, $leaveType);

            $this->logCustomMessage(
                'create_leave_type',
                $leaveType,
                Auth::user()->name . ' created a new leave type ' . $leaveType->title,
                $leaveType,
                LeaveManagementLogType::CREATE_TYPE,
                new Activity()
            );

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text' => 'New leave type has been created.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\LeaveType  $leaveType
     * @return \Illuminate\Http\Response
     */
    public function show($leaveTypeId)
    {
        $validate = Validator::make(['id' => $leaveTypeId], [
            'id'  =>  'required|exists:' . env('ADG_DB_CONNECTION') . '.leave_types,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    =>  $validate->errors()
            ], 400);
        }

        return LeaveType::where('id', $leaveTypeId)->first();
    }

    // paginate
    public function paginate(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return LeaveType::where('title', 'LIKE', '%' . $request->keyword . '%')->with(['eligibleGroups', 'eligibleUsers'])->paginate($paginate);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\LeaveType  $leaveType
     * @return \Illuminate\Http\Response
     */
    public function update(StoreUpdateRequest $request, $leaveTypeId)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();

        try {

            $leaveType = LeaveType::where('id', $leaveTypeId);
            $leaveType->update([
                'title' => $request->title,
                'description' => $request->description,
                'initial_points' => $request->initial_points,
                'file_attachment' => $request->file_attachment,
                'remark' => $request->remark,
                'default_increase' => $request->leave_points,
                'increase_type' => $request->increase_type,
                'status' => $request->status,
                'type' => $request->type,
                'updated_by' => Auth::id(),
            ]);

            LeaveEligibleGroup::where('leave_type_id', $leaveTypeId)->delete();
            LeaveEligibleUser::where('leave_type_id', $leaveTypeId)->delete();

            (new LeaveEligibleService)->assignLeaveType($request->group_ids, $request->user_ids, $leaveType->first());

            $updateLeaveType = LeaveType::find($leaveTypeId);

            $updateLeaveType->old = collect($leaveType);
            $updateLeaveType->attributes = collect($updateLeaveType);
                    
            $this->logCustomMessage(
                'update_leave_type',
                $updateLeaveType,
                Auth::user()->name . ' updated the leave type ' . $updateLeaveType->title,
                $updateLeaveType,
                LeaveManagementLogType::UPDATE_TYPE,
                new Activity()
            );

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text' => 'Leave type has been updated.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }
}
