<?php

namespace Suiterus\Adg\Controllers\LeaveManagement;

use Exception;
use mikehaertl\pdftk\Pdf;
use App\Enums\LeaveStatus;
use Illuminate\Http\Request;
use App\Enums\LeaveApproverStatus;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Enums\Log\LeaveManagementLogType;
use Suiterus\Adg\Models\Activity\Activity;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\LeaveManagement\LeaveApprover;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Services\LeaveManagement\LeaveService;
use Suiterus\Adg\Requests\LeaveApprover\LeaveApproverRequest;

class LeaveApproverController extends Controller
{
    use HasCustomLogs;

    public function paginate(Request $request)
    {
        return LeaveApprover::where('approver_id', Auth::id())->when($request->name, function ($query) use ($request) {
            $query->whereHas('leave', function ($query) use ($request) {
                $query->whereHas('user', function ($query) use ($request) {
                    $query->where('name', 'LIKE', "%{$request->name}%");
                });
            });
        })->with(['leave.user.personalInformation', 'leave.user.salary.publicSalary', 'leave.user.salary.privateSalary', 'leave.attachments'])->orderBy('created_at', 'desc')->paginate(10);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\LeaveApproverRequest  $request
     * @param  \App\Models\LeaveApprover  $leaveApprover
     * @return \Illuminate\Http\Response
     */
    public function update(LeaveApproverRequest $request, LeaveService $leaveService)
    {

        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $leaveApprover = LeaveApprover::where([
                ['id', $request->id],
                ['approver_id', Auth::id()]
            ]);

            if (!$leaveApprover->first()) {
                return response([
                    'error' => 'You are not authorized to do this action!'
                ], 403);
            }

            $leave = $leaveApprover->first()->leave;

            $formData = [
                'authorized_officer_7a' => $leave->metadata->authorized_officer_7a ?? $request->metadata['authorized_officer_7a'],
                'authorized_officer_7b' => $leave->metadata->authorized_officer_7b ?? $request->metadata['authorized_officer_7b'],
                'for_disapproval_details' => $leave->metadata->for_disapproval_details ?? $request->metadata['for_disapproval_details']
            ];

            $metadata = $request->metadata;

            if ($leave->metadata) {
                foreach ($leave->metadata as $key => $value) {
                    $metadata[$key] = $value ?? $request->metadata[$key];
                }
            }

            if (!isset($leave->metadata['approval']) && $metadata['approval']) {
                $recommendation = $metadata['approval'] == LeaveStatus::DECLINED ? 'for_disapproval' : 'for_approval';
                $formData[$recommendation] = 'On';

                if ($metadata['approval'] == LeaveStatus::DECLINED) {
                    $formData['for_disapproval_details'] = $request->remarks;
                }
            }

            if (!$leave) {
                return response()->json([
                    'message' => 'Leave application cannot be found, please verify if it was deleted.'
                ], 404);
            }

            if ($request->status == LeaveApproverStatus::DECLINED) {
                $leaveApprover->first()->leave()->update([
                    'status' => LeaveStatus::DECLINED
                ]);
            }

            $leaveApprover->update([
                'status' => $request->status,
                'remarks' => $request->remarks
            ]);

            $leave->update([
                'metadata' => $metadata
            ]);

            $leaveService->updateLeaveFormFile($formData, $leave->file_path, $leave->file_name);

            $updated = LeaveApprover::where([
                ['id', $request->id],
                ['approver_id', Auth::id()]
            ])->with('leave')->first();

            $updated->old = collect($leave);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'approver_update_leave',
                $updated,
                Auth::user()->name . ' approved the request of ' . $updated->leave->user->name . ' for ' . $updated->leave->leaveType->title,
                $updated,
                LeaveManagementLogType::APPROVER_UPDATE,
                new Activity()
            );

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();

            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }
}
