<?php

namespace Suiterus\Adg\Controllers\LeaveManagement;

use Exception;
use App\Models\User;
use App\Enums\LeaveStatus;
use Illuminate\Http\Request;
use UnexpectedValueException;
use App\Enums\MonetizationLeave;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Sector;
use Suiterus\Hrjp\Models\Position;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Department;
use Suiterus\Adg\Models\Payroll\Payroll;
use App\Enums\Log\LeaveManagementLogType;
use App\Http\Classes\NotificationMessage;
use Illuminate\Support\Facades\Validator;
use App\Enums\LeaveType as EnumsLeaveType;
use Suiterus\Adg\Models\Activity\Activity;
use App\Enums\LeaveAllowance as EnumsLeaveAllowance;
use Suiterus\Adg\Models\Payroll\TimekeepingDeduction;
use Suiterus\Adg\Models\LeaveManagement\LeaveAllowance;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Services\LeaveManagement\LeaveService;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterLegend;
use Suiterus\Adg\Controllers\Services\LeaveCreditService;
use Suiterus\Adg\Models\LeaveManagement\LeaveApplication;
use Suiterus\Adg\Models\LeaveManagement\ForcedLeaveCounter;
use Suiterus\Adg\Models\LeaveManagement\Requests\LeaveRequest;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadEmployeeShift;
use Suiterus\Adg\Models\LeaveManagement\Requests\LeaveForCancellation;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveCreditDeduction;
use Suiterus\Adg\Controllers\LeaveManagement\Services\LeaveRefundService;

class LeaveApplicationController extends Controller
{
    use HasCustomLogs;

    protected $leaveRefundService;

    public function __construct() {
        $this->leaveRefundService = new LeaveRefundService();
    }

    // List Requests
    public function fetch_records(Request $request)
    {

        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        try {

            $leaves = Leave::when(count($request->employees) > 0, function ($query) use ($request) {
                $query->whereIn('user_id', $request->employees);
            })
                ->when($request->position != null, function ($query) use ($request) {
                    $query->whereHas('user.employeeMetaInfo', function ($query) use ($request) {
                        $query->where('position_id', $request->position);
                    });
                })
                ->when($request->department != null, function ($query) use ($request) {
                    $query->whereHas('user.employeeMetaInfo', function ($query) use ($request) {
                        $query->where('department_id', $request->department);
                    });
                })
                ->when(isset($request->filter_dates) && count($request->filter_dates) > 0, function ($query) use ($request) {
                    $query->when(count($request->filter_dates) == 1, function ($query) use ($request) {
                        $query->whereDate('start_date', '<=', $request->filter_dates[0])
                            ->whereDate('end_date', '>=', $request->filter_dates[0]);
                    })
                        ->when(count($request->filter_dates) == 2, function ($query) use ($request) {
                            $query->where(function ($query) use ($request) {
                                $query->whereDate('start_date', '<=', $request->filter_dates[0])
                                    ->whereDate('end_date', '>=', $request->filter_dates[1]);
                            })->orWhere(function ($query) use ($request) {
                                $query->whereDate('start_date', '<=', $request->filter_dates[1])
                                    ->whereDate('end_date', '>=', $request->filter_dates[0]);
                            });
                        });
                })
                ->when($request->status != null, function ($query) use ($request) {
                    $query->where('status', $request->status);
                })
                ->whereHas('user', function ($query) use ($request) {
                    $keyword = '%' . $request->employee_name . '%';
                    $query->where('name', 'LIKE', $keyword)
                        ->when(isset($request->supervisor) && $request->supervisor != 0, function ($query) use ($request) {
                            $query->whereHas('user_supervisor.supervisor.user', function ($query) use ($request) {
                                $query->whereId($request->supervisor);
                            });
                        });
                })
                ->with([
                    'user'  => function ($query) {
                        $query->select('id', 'name')
                            ->without([
                                'roles',
                                'permissions',
                                'storage',
                            ])
                            ->with([
                                'user_supervisor' => function ($query) {
                                    $query->with([
                                        'supervisor' => function ($query) {
                                            $query->with(['user' => function ($query) {
                                                $query->without(['roles', 'permissions', 'supervisor', 'storage', 'employeeMetaInfo']);
                                            }]);
                                        },
                                    ]);
                                },
                            ]);
                    },
                    'user.personalInformation', 'user.salary.publicSalary', 'user.salary.privateSalary', 'attachments', 'leaveAllowances.employeeAllowance.allowance', 'forCancellation'
                ])
                ->orderBy('updated_at', 'desc')->paginate($paginate);

            return response()->json([
                'data'  => $leaves
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function fetch_leave(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.leaves,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    =>  $validate->errors()
            ], 400);
        }

        try {

            $sector = Sector::where('status', 'active')->first();

            $leave = Leave::whereId($request->id)->with([
                'detail',
                'monetization' => function ($query) {
                    $query->with('monetized_leave');
                }
            ])->first();

            $user = $leave->user()
                ->with([
                    'salary' => function ($query) {
                        $query->without('user');
                    }
                ])
                ->without(['permissions', 'roles', 'storage'])->first();

            $salary = $user->salary;
            if ($sector->sector == 1) {
                $salary = $salary->publicSalary->salaryGrade->value;
            } else {
                $salary = $salary->privateSalary->salary;
            }

            $first_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'first_name')->first()->field_value;
            $middle_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'middle_name')->first()->field_value;
            $last_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'last_name')->first()->field_value;

            $name = [
                'first_name'    => $first_name,
                'middle_name'   => $middle_name,
                'last_name'     => $last_name,
            ];

            $monetization = $leave->monetization()->with('monetized_leave')->first();

            $certification_of_credits = [
                'vacation_leave_initial'    => 0,
                'sick_leave_initial'        => 0,
                'vacation_leave_deduction'  => 0,
                'sick_leave_deduction'      => 0,
                'vacation_leave_balance'    => 0,
                'sick_leave_balance'        => 0,
            ];

            $vacation_leave_balance = $leave->user->leave_balance()->where('leave_type_id', EnumsLeaveType::VACATION_LEAVE)->first();
            $sick_leave_balance = $leave->user->leave_balance()->where('leave_type_id', EnumsLeaveType::SICK_LEAVE)->first();

            if ($leave->status == 1) {

                $sick_deduction = $leave->credit_deductions()->where('remarks', __('leaves.sick'))->first();
                $vacation_deduction = $leave->credit_deductions()->where('remarks', __('leaves.vacation'))->first();

                $sick_deduction = $sick_deduction == null ? 0 : $sick_deduction->credits;
                $vacation_deduction = $vacation_deduction == null ? 0 : $vacation_deduction->credits;

                $certification_of_credits = [
                    'vacation_leave_initial'    => ($vacation_leave_balance->balance + $vacation_deduction),
                    'sick_leave_initial'        => ($sick_leave_balance->balance + $sick_deduction),
                    'vacation_leave_deduction'  => $vacation_deduction,
                    'sick_leave_deduction'      => $sick_deduction,
                    'vacation_leave_balance'    => $vacation_leave_balance->balance,
                    'sick_leave_balance'        => $sick_leave_balance->balance
                ];
            } else if ($leave->status == 2) {

                $certification_of_credits = [
                    'vacation_leave_initial'    => $vacation_leave_balance->balance,
                    'sick_leave_initial'        => $sick_leave_balance->balance,
                    'vacation_leave_deduction'  => 0,
                    'sick_leave_deduction'      => 0,
                    'vacation_leave_balance'    => $vacation_leave_balance->balance,
                    'sick_leave_balance'        => $sick_leave_balance->balance,
                ];
            }

            $data = array(
                'leave'     => $leave,
                'name'          => $name,
                'monetizations' => $monetization,
                'user'      => $user,
                'salary'    => $salary,
                'certification' => $certification_of_credits
            );

            return response()->json([
                'data'  => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the record'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    // Approve, Decline and Cancel
    public function approve(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.leaves,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {

            $leave = Leave::with(['leaveType'])->findOrFail($request->id);

            $message = 'Only pending requests can be changed';
            if ($leave->status == LeaveStatus::FOR_REVIEW) {

                if ((strtolower($leave->leaveType->title) != 'monetization of leave credits' && strtolower($leave->leaveType->title) != 'terminal leave')) {
                    foreach ($request->selected_allowances as $allowance) {
                        LeaveAllowance::create([
                            'leave_id' => $leave->id,
                            'allowance_id' => $allowance,
                            'type' => EnumsLeaveAllowance::PAID
                        ]);
                    }

                    foreach ($request->unselected_allowances as $allowance) {
                        LeaveAllowance::create([
                            'leave_id' => $leave->id,
                            'allowance_id' => $allowance,
                            'type' => EnumsLeaveAllowance::NOT_PAID
                        ]);
                    }
                }

                $this->update_leave_roster_table($leave);

                $message = 'The request has been approved.';

                $payroll = Payroll::where('status', 1)->whereHas('payroll_employee', function ($query) use ($leave) {
                    $query->where('user_id', $leave->user_id);
                })->get();

                if ($payroll->isNotEmpty()) {
                    foreach ($payroll as $payroll) {
                        TimekeepingDeduction::where('type', 1)->whereHas('employee_payroll', function ($query) use ($leave) {
                            $query->where('user_id', $leave->user_id)->whereBetween('date', [$leave->start_date, $leave->end_date]);
                        })->delete();
                    }
                }

                $leaveCreditService = new LeaveCreditService;
                $leaveService = new LeaveService();

                // $leaveService->monetizeLeave($leave, $request->user()->id);
                $credits = null;

                $formData = [
                    'authorized_official' => $request->authorized_official,
                    'certification_date'     => date('Y-m-d'),
                    'total_vacation_leave_balance' => '-',
                    'vacation_leave_deduction' => '-',
                    'remaining_vacation_leave_balance' => '-',
                    'total_sick_leave_balance' => '-',
                    'sick_leave_deduction' => '-',
                    'remaining_sick_leave_balance' => '-'
                ];

                $balance = round($leaveService->getLeaveBalance($leave->user_id, $leave->leaveType->id));

                $userId = $leave->user_id;

                $vacationLeaveBalance = LeaveBalance::where('user_id', $userId)->whereHas('leave_type', function ($query) {
                    $query->where('title', 'Vacation Leave');
                })->first();

                $sickLeaveBalance = LeaveBalance::where('user_id', $userId)->whereHas('leave_type', function ($query) {
                    $query->where('title', 'Sick Leave');
                })->first();

                $isSickAndVacationDeducted = false;

                if (strtolower($leave->leaveType->title) == 'monetization of leave credits') {
                    $deductions = $leaveCreditService->monetizeLeave($leave, $request->user()->id);
                    $vacationDeduction = $deductions['vacation_deduction'];
                    $sickDeduction = $deductions['sick_deduction'];
                    $isSickAndVacationDeducted = true;
                } else if (strtolower($leave->leaveType->title) == 'terminal leave') {
                    $leaveCreditService->computeTerminalLeave($leave, $request->user()->id);
                    $vacationDeduction = $vacationLeaveBalance->balance;
                    $sickDeduction = $sickLeaveBalance->balance;
                    $isSickAndVacationDeducted = true;
                } else {
                    $credits = $leave->number_of_days;

                    $leaveCreditService->deductBalance($leave, $request->user()->id, $credits);
                }

                $formData['total_vacation_leave_balance'] = $remainingVacationBalance = number_format($vacationLeaveBalance ? $vacationLeaveBalance->balance : 0, 2);
                $formData['total_sick_leave_balance'] = $remainingSickBalance = number_format($sickLeaveBalance ? $sickLeaveBalance->balance : 0, 2);

                $vacationDeduction = $credits ?? $vacationDeduction;

                if ($vacationLeaveBalance) {
                    $formData['vacation_leave_deduction'] = (isset($vacationDeduction) && strtolower($leave->leaveType->title) == 'vacation leave') || $isSickAndVacationDeducted ? number_format($vacationDeduction, 2) : '-';
                    $remainingVacationBalance = (isset($vacationDeduction) && strtolower($leave->leaveType->title) == 'vacation leave') || $isSickAndVacationDeducted ? number_format($vacationLeaveBalance->balance - $vacationDeduction, 2) : $remainingVacationBalance;
                }

                $sickDeduction = $credits ?? $sickDeduction;

                if ($sickLeaveBalance) {
                    $formData['sick_leave_deduction'] = (isset($sickDeduction) && strtolower($leave->leaveType->title) == 'sick leave') || $isSickAndVacationDeducted ? number_format($sickDeduction, 2) : '-';
                    $remainingSickBalance = (isset($sickDeduction) && strtolower($leave->leaveType->title) == 'sick leave') || $isSickAndVacationDeducted ? number_format($sickLeaveBalance->balance - $sickDeduction, 2) : $remainingSickBalance;
                }

                $formData['remaining_vacation_leave_balance'] = $remainingVacationBalance >= 0 ? $remainingVacationBalance : number_format(0, 2);
                $formData['remaining_sick_leave_balance'] = $remainingSickBalance >= 0 ? $remainingSickBalance : number_format(0, 2);

                $statusApproved = 1;
                $leave->update([
                    'status'            => $statusApproved, //approved
                    'date_approved'     => date('Y-m-d'),
                    'reason_purpose'    => $request->reason_purpose,
                    'updated_by'        => $request->user()->id
                ]);

                $this->leaveRefundService->updateLeaveRefund($statusApproved, $leave);

                $updated = Leave::with(['leaveType', 'user'])->findOrFail($request->id);

                $updated->old = collect($leave);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'approve_leave_request',
                    $updated,
                    Auth::user()->name . ' approved the request of ' . $updated->user->name . ' for ' . $updated->leaveType->title,
                    $updated,
                    LeaveManagementLogType::APPROVE,
                    new Activity()
                );

                $leave->save();

                $approvedFor = [
                    1 => 'days_with_pay',
                    2 => 'days_without_pay',
                    3 => 'approved_for'
                ];

                if ($balance < $leave->number_of_days) {
                    $approvedForWithoutPay = 2;
                    $daysWithoutPay = $leave->number_of_days - $balance;
                    $daysWithPay = $leave->number_of_days - $daysWithoutPay;
                    $leave->number_of_days = $daysWithPay;
                    $formData[$approvedFor[$approvedForWithoutPay]] = $daysWithoutPay;
                }

                $formData[$approvedFor[$request->approved_for]] = $request->approved_for == 3 ? $request->reason_purpose : $leave->number_of_days;

                $leaveService->updateLeaveFormFile($formData, $leave->file_path, $leave->file_name);
                NotificationMessage::notifySenderAndReceiver('Approved leave', $leave->user_id, Auth::id());
                DB::connection(env('ADG_DB_CONNECTION'))->commit();
            }
            return response()->json([
                'message'   => $message
            ]);
        } catch (UnexpectedValueException $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => [$e->getMessage()],
                'message'   => $e->getMessage(),
            ], 500);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['The selected record could not be found'],
                'message'   => $e->getMessage(),$e->getLine(), $e->getFile()
            ], 500);
        }
    }

    public function decline(Request $request, LeaveService $leaveService)
    {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.leaves,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $leave = Leave::findOrFail($request->id);

            $message = 'Only pending requests can be changed';
            if ($leave->status == LeaveStatus::FOR_REVIEW) {
                $leave->update([
                    'date_approved'     => date('Y-m-d'),
                    'status'            => 2,
                    'reason_purpose'    => $request->reason_purpose,
                    'updated_by'        => $request->user()->id
                ]);

                $leaveService->updateLeaveFormFile([
                    'disapproved_due_to_details' => $request->reason_purpose,
                    'authorized_official' => $request->authorized_official
                ], $leave->file_path, $leave->file_name);

                $message = 'The request has been disapproved.';

                $this->leaveRefundService->updateLeaveRefund(2, $leave);
            } else if ($leave->status == LeaveStatus::FOR_CANCELLATION) {
                $cancel = LeaveForCancellation::where('leave_id', $leave->id)->where('leave_type_id', $leave->leave_type_id)->first();
                $leave->update([
                    'status'            => $cancel->old_leave_status,
                    'updated_by'        => Auth::id()
                ]);

                $cancel->update([
                    'user_id'           => Auth::id(),
                    'status'            => 2,
                    'updated_by'        => Auth::id(),
                    'updated_at'        => now()
                ]);

                $message = 'The request has been cancelled.';

                $this->leaveRefundService->updateLeaveRefund(2, $leave);
            }

            if (strtolower($leave->leaveType->title) == 'mandatory/forced leave') {
                $forcedLeaveCounter = ForcedLeaveCounter::where('user_id', $leave->user_id)->first();
                $forcedLeaveCounter->update([
                    'counter' => ($forcedLeaveCounter->counter - $leave->number_of_days) <= 0 ? 0 : ($forcedLeaveCounter->counter - $leave->number_of_days)
                ]);
            }

            $updated = Leave::with(['leaveType', 'user'])->findOrFail($request->id);
           
            $updated->old = collect($leave);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'decline_leave_request',
                $updated,
                Auth::user()->name . ' declined the request of ' . $updated->user->name . ' for ' . $updated->leaveType->title,
                $updated,
                LeaveManagementLogType::DECLINE,
                new Activity()
            );

            $leave->save();
            NotificationMessage::notifySenderAndReceiver('Disapproved leave', $leave->user_id, Auth::id());
            DB::commit();

            return response()->json([
                'message'   => $message
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['The selected record could not be found'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function cancel(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.leaves,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $leave = Leave::findOrFail($request->id);

            $message = 'Only for cancellation requests can be changed';

            if ($leave->status == 5) {
                LeaveForCancellation::where('leave_id', $leave->id)->update([
                    'status'             =>  1,
                    'updated_by'         => Auth::id(),
                    'updated_at'         => now(),
                ]);

                $cancel = LeaveForCancellation::where('leave_id', $leave->id)->orderBy('id', 'desc')->first();

                if ($cancel->old_leave_status == 1) {

                    $user = User::whereId($leave->user_id)->without(['permissions', 'storage', 'roles', 'employeeMetaInfo'])->first();
                    $leaves = $user->leaves()->where('status', 5)->pluck('id')->toArray();

                    $points = LeaveCreditDeduction::whereHasMorph('deductable', Leave::class, function ($query) use ($leaves) {
                        $query->whereIn('id', $leaves);
                    })->first();

                    $deduct = LeaveCreditDeduction::whereHasMorph('deductable', Leave::class, function ($query) use ($leaves) {
                        $query->whereIn('deductable_id', $leaves);
                    })->get();

                    $vacation_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::VACATION_LEAVE)->first();
                    $sick_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::SICK_LEAVE)->first();
                    $forced_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::FORCED_LEAVE)->first();
                    $special_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::SPECIAL_LEAVE)->first();

                    if ($leave->leaveType->title == 'Sick Leave') {
                        if ($deduct->count() == 2) {
                            $this->cancel_sick_vacation_request($leave, $sick_balance, $vacation_balance, $leaves, $points);
                        } else {
                            $balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::SICK_LEAVE)->first();
                            $balance_points = $balance->balance;
                            $credit_points  = $points->credits;
                            $total_points   = $balance_points + $credit_points;

                            $balance->update([
                                'balance'   =>  $total_points
                            ]);
                        }
                    } else if ($leave->leaveType->title == 'Forced Leave') {
                        $this->cancel_forced_vacation_request($leave, $forced_balance, $vacation_balance, $leaves, $points);
                    } else if ($leave->leaveType->title == 'Special Privilege Leave') {
                        $balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::SPECIAL_LEAVE)->first();
                        $balance_points = $balance->balance;
                        $credit_points  = $points->credits;
                        $total_points   = $balance_points + $credit_points;

                        $balance->update([
                            'balance'   =>  $total_points
                        ]);
                    } else if ($leave->leaveType->title != 'Terminal Leave' && $leave->leaveType->title != 'Monetization of Leave Credits') {
                        $balance = $user->leave_balance()->where('leave_type_id', $leave->leave_type_id)->first();
                        $balance_points = $balance->balance;
                        $credit_points  = $points->credits;
                        $total_points   = $balance_points + $credit_points;

                        $balance->update([
                            'balance'   =>  $total_points
                        ]);
                    }

                    LeaveApplication::where('leave_id', $leave->id)->update([
                        'vacation_leave'             =>  $vacation_balance->balance,
                        'sick_leave'                 =>  $sick_balance->balance,
                        'forced_leave'               =>  $forced_balance->balance,
                        'special_privilege_leave'    =>  $special_balance->balance,
                        'updated_by'                 =>  Auth::id()
                    ]);

                    $leave->leave_without_pay()->where('leave_id', $leave->id)->delete();

                    LeaveCreditDeduction::whereHasMorph('deductable', Leave::class, function ($query) use ($leaves) {
                        $query->whereIn('id', $leaves);
                    })->delete();
                }

                $leave->update([
                    'status'            => 4,
                    'reason_purpose'    => $request->reason_purpose,
                    'updated_by'        => Auth::id(),
                    'updated_at'        => now()
                ]);

                $this->leaveRefundService->updateLeaveRefund(2, $leave);
                $updated = Leave::with(['leaveType', 'user'])->findOrFail($request->id);

                $updated->old = collect($leave);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'cancel_leave_request',
                    $updated,
                    Auth::user()->name . ' cancelled the request of ' . $updated->user->name . ' for ' . $updated->leaveType->title,
                    $updated,
                    LeaveManagementLogType::CANCEL,
                    new Activity()
                );

                $leave->save();

                $roster_head_shift = RosterHeadEmployeeShift::whereHas('roster', function ($query) use ($leave) {
                    $query->where('head_nurse_id', $leave->user_id);
                })->whereHas('rosterDay', function ($query) use ($leave) {
                    $query->whereBetween('date', [$leave->start_date, $leave->end_date]);
                })->get();

                $roster_emp_shift = RosterDayEmployeeShift::whereHas('rosterEmployeePerGroup', function ($query) use ($leave) {
                    $query->where('user_id', $leave->user_id);
                })->whereHas('rosterDay', function ($query) use ($leave) {
                    $query->whereBetween('date', [$leave->start_date, $leave->end_date]);
                })->get();

                if ($roster_emp_shift->isNotEmpty()) {
                    $roster_emp_shift->toQuery()->update([
                        'roster_legend_id' => null
                    ]);
                }
                if ($roster_head_shift->isNotEmpty()) {
                    $roster_head_shift->toQuery()->update([
                        'roster_legend_id' => null
                    ]);
                }

                NotificationMessage::notifySenderAndReceiver('Canceled Leave', $leave->user_id, Auth::id());
                DB::commit();

                $message = 'The request has been cancelled.';
            }

            return response()->json([
                'message'   => $message
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The selected record could not be found'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    // cancel request conditions --
    //for forced leave and vacation leave
    public function cancel_forced_vacation_request($leave, $forced_balance, $vacation_balance, $leaves, $points)
    {
        $forced_balance_points = $forced_balance->balance;
        $forced_credit_points = $points->credits;
        $forced_total_points = $forced_balance_points + $forced_credit_points;
        $forced_balance->update([
            'balance' => $forced_total_points
        ]);

        $vacation_balance_points = $vacation_balance->balance;
        $vacation_credit_points = $points->credits;
        $vacation_total_points = $vacation_balance_points + $vacation_credit_points;
        $vacation_balance->update([
            'balance' => $vacation_total_points
        ]);

        $leave->leave_without_pay()->where('leave_id', $leave->id)->delete();

        LeaveCreditDeduction::whereHasMorph('deductable', Leave::class, function ($query) use ($leaves) {
            $query->whereIn('id', $leaves);
        })->delete();
    }

    //for sick leave and vacation
    public function cancel_sick_vacation_request($leave, $sick_balance, $vacation_balance, $leaves, $points)
    {
        $sick_balance_points = $sick_balance->balance;
        $sick_credit_points = $points->credits;
        $sick_total_points = $sick_balance_points + $sick_credit_points;
        $sick_balance->update([
            'balance' => $sick_total_points
        ]);
        $v_points = LeaveCreditDeduction::whereHasMorph('deductable', Leave::class, function ($query) use ($leaves) {
            $query->whereIn('id', $leaves)->where('remarks', 'Vacation Leave');
        })->first();

        $vacation_balance_points = $vacation_balance->balance;
        $vacation_credit_points = $v_points->credits;
        $vacation_total_points = $vacation_balance_points + $vacation_credit_points;
        $vacation_balance->update([
            'balance' => $vacation_total_points
        ]);

        $leave->leave_without_pay()->where('leave_id', $leave->id)->delete();

        LeaveCreditDeduction::whereHasMorph('deductable', Leave::class, function ($query) use ($leaves) {
            $query->whereIn('id', $leaves);
        })->delete();
    }

    // Batch approval
    public function batch_approve(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'reason_purpose'  => 'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            $leaves = Leave::where('status', 3)->get();
            $count = $leaves->count();

            $leaveService = new LeaveCreditService;

            foreach ($leaves as $leave) {

                $leave->update([
                    'status'            => 1,
                    'date_approved'     => date('Y-m-d'),
                    'reason_purpose'    => $request->reason_purpose,
                    'updated_by'        => $request->user()->id
                ]);
                $this->update_leave_roster_table($leave);
                if ($leave->leave_detail_id == MonetizationLeave::MONETIZATION) {
                    $leaveService->monetizeLeave($leave, $request->user()->id);
                } else if ($leave->leave_detail_id == MonetizationLeave::TERMINAL) {
                    $leaveService->computeTerminalLeave($leave, $request->user()->id);
                } else {
                    $credits = $leave->number_of_days;
                    $leaveService->deductBalance($leave, $request->user()->id, $credits);
                }

                $updated = Leave::with(['leaveType', 'user'])->findOrFail($leave->id);

                $updated->old = collect($leave);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'approve_all_leave_request',
                    $updated,
                    Auth::user()->name . ' approved the request of ' . $updated->user->name . ' for ' . $updated->leaveType->title,
                    $updated,
                    LeaveManagementLogType::APPROVE_ALL,
                    new Activity()
                );

                $leave->save();
            }

            DB::connection(env('ADG_DB_CONNECTION'))->commit();

            return response()->json([
                'message'   => $count . ' records have been approved.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'   => ['There was a problem in approving the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function update_leave_roster_table($leave)
    {

        $roster_head_shift = RosterHeadEmployeeShift::whereHas('roster', function ($query) use ($leave) {
            $query->where('head_nurse_id', $leave->user_id);
        })->whereHas('rosterDay', function ($query) use ($leave) {
            $query->whereBetween('date', [$leave->start_date, $leave->end_date]);
        })->get();

        $roster_emp_shift = RosterDayEmployeeShift::whereHas('rosterEmployeePerGroup', function ($query) use ($leave) {
            $query->where('user_id', $leave->user_id);
        })->whereHas('rosterDay', function ($query) use ($leave) {
            $query->whereBetween('date', [$leave->start_date, $leave->end_date]);
        })->get();

        $legend = RosterLegend::where('key', 'leave')->value('id');

        if ($roster_emp_shift->isNotEmpty()) {
            $roster_emp_shift->toQuery()->update([
                'roster_legend_id' => empty($legend) ? null : $legend
            ]);
        }
        if ($roster_head_shift->isNotEmpty()) {
            $roster_head_shift->toQuery()->update([
                'roster_legend_id' => empty($legend) ? null : $legend
            ]);
        }
    }
}
