<?php

namespace Suiterus\Adg\Controllers\LeaveManagement\Employee;

use Exception;
use App\Models\User;
use mikehaertl\pdftk\Pdf;
use App\Enums\LeaveStatus;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Sector;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Enums\Log\LeaveManagementLogType;
use App\Http\Classes\NotificationMessage;
use Illuminate\Support\Facades\Validator;
use App\Enums\LeaveType as EnumsLeaveType;
use Suiterus\Adg\Models\Activity\Activity;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\LeaveManagement\LeaveType;
use Suiterus\Adg\Models\LeaveManagement\LeaveDetail;
use Suiterus\Adg\Models\UserSupervisors\Supervisors;
use Suiterus\Adg\Models\LeaveManagement\LeaveSubtype;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Services\LeaveManagement\LeaveService;
use Suiterus\Adg\Models\LeaveManagement\LeaveAttachment;
use Suiterus\Adg\Requests\Leave\DownloadLeaveFormRequest;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveCreditEarning;
use Suiterus\Adg\Services\EmailNotification\EmailNotificationService;
use Suiterus\Adg\Models\LeaveManagement\Requests\LeaveForCancellation;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveCreditDeduction;
use Suiterus\Adg\Controllers\LeaveManagement\Services\LeaveRefundService;

class LeaveController extends Controller
{
    use HasCustomLogs;

    protected $emailNotificationService;
    protected $leaveRefundService;

    public function __construct() {
        $this->emailNotificationService = new EmailNotificationService();
        $this->leaveRefundService = new LeaveRefundService();
    }

    public function create_request(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'selected_leave_type'  =>  'required',
            'remarks' => 'nullable|max:280',
            'files.*' => 'mimes:pdf,doc,docx,png,jpg,jpeg|max:10000'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();

        try {

            $leaveService = new LeaveService();

            $forReview = 3;
            $system = 1;

            $metadata = json_decode($req->metadata);

            if ($req->selected_leave_type == EnumsLeaveType::SPECIAL_LEAVE) {
                $specialPrivilegeLeave = LeaveBalance::where([
                    ['user_id', $req->user()->id], ['leave_type_id', $req->selected_leave_type]
                ])->first();

                if ($specialPrivilegeLeave->balance == 0) {
                    NotificationMessage::notifyReceiver('Insufficient Special Privilege Leave Balance', Auth::id(), $system);
                    $subject = 'NKTI Leave Management Notification';
                    $message ='You have used all your Special Privilege Leave Balance, it will be deducted from your Vacation Leave Balance';
                    $this->emailNotificationService->sendMailNotification($req->user()->email, $subject, $message, $req->user()->name);
                }
            }

            $leave = Leave::create([
                'user_id'           => $req->user()->id,
                'leave_type_id'  => $req->selected_leave_type,
                'leave_detail_id'   => null,
                'number_of_days'    => $req->working_days,
                'start_date'        => $req->start_date,
                'end_date'          => $req->end_date,
                'commutation'       => $req->commutation,
                'remarks'    => $req->remarks ?? '',
                'date_filed'        => date('Y-m-d'),
                'status'            => $forReview,
                'metadata' => $metadata,
                'created_by'        => $req->user()->id,
                'updated_by'        => $req->user()->id,
            ]);

            $isMonetizedLeave = LeaveType::where([
                ['id', $req->selected_leave_type],
                ['title', 'Monetization of Leave Credits']
            ])->first();

            if ($isMonetizedLeave) {
                $leave->monetization()->create([
                    'type'  => $req->type,
                    'vacation_days'  => $req->vacation_days,
                    'sick_days'  => $req->sick_days
                ]);
            }

            $leaveService->assignApprovers($req->to, $leave->id);

            $fillFormData = $leaveService->getLeaveFormData(User::where('id', Auth::id())->first(), $req->selected_leave_type, $req->commutation, $req->working_days, $req->start_date, $req->end_date);

            if ($metadata) {
                foreach ($metadata as $key => $value) {
                    $fillFormData[$key] = $value;
                }
            }

            $path = Auth::id() . '/' . $leave->id;
            $fileName = uniqid() . '.pdf';

            $storage = Storage::disk('leave_form');

            $filePath = $leaveService->createLeaveFormFile($storage, $path, $fileName, $fillFormData, config('pdftk.options'));

            Leave::find($leave->id)->update([
                'file_name' => $fileName,
                'file_path' => $filePath . '.enc'
            ]);

            $attachments = $req->file('files');

            $leaveService->attachFilesToLeaveForm($storage, $leave->id, $filePath, $attachments, $path, config('pdftk.options'));
            $this->leaveRefundService->createLeaveRefund($leave);
            $this->logCustomMessage(
                'create_request_leave',
                $leave,
                Auth::user()->name . ' created a new ' . $leave->leaveType->title . ' request',
                $leave,
                LeaveManagementLogType::CREATE,
                new Activity()
            );

            foreach ($req->to as $receiver) {
                NotificationMessage::notifyReceiver('Request leave',$receiver, Auth::id());
            }

            DB::connection(env('ADG_DB_CONNECTION'))->commit();

            return response()->json([
                'text' => 'New request record has been created.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();

            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
                'line'   => $e->getLine(),
            ], 500);
        }
    }

    public function download_leave_form(DownloadLeaveFormRequest $request)
    {
        $leave = Leave::find($request->leave_id);
        $storage = Storage::disk('leave_form');

        if (!$storage->exists($leave->file_path)) {
            return response()->json([
                'message' => 'Leave form not found'
            ], 404);
        }

        $explodedPath = explode('.', $leave->file_path);
        $fileExt = $explodedPath[count($explodedPath) - 1];

        if ($fileExt == 'enc') {
            FileVault::disk('leave_form')->decryptCopy($leave->file_path);
        }

        $decryptedFilePath = dirname($leave->file_path) . '/' . $leave->file_name;

        $pdf = new Pdf($storage->path($decryptedFilePath), config('pdftk.options'));

        $result = $pdf->flatten()
            ->saveAs($storage->path($decryptedFilePath));

        if (!$result) {
            throw new FileException('There was a problem saving the pdf leave form');
        }

        $this->logCustomMessage(
            'download_view_leave',
            $leave,
            Auth::user()->name . ' downloaded/viewed a leave form',
            $leave,
            LeaveManagementLogType::DOWNLOAD_VIEW,
            new Activity()
        );

        return response()->download($storage->path($decryptedFilePath))->deleteFileAfterSend(true);
    }

    public function download_leave_attachment_form(Request $request)
    {
        $leaveAttachment = LeaveAttachment::find($request->leave_attachment_id);
        $storage = Storage::disk('leave_form');

        if (!$storage->exists($leaveAttachment->file_path)) {
            return response()->json([
                'message' => 'Leave form not found'
            ], 404);
        }

        $explodedPath = explode('.', $leaveAttachment->file_path);
        $fileExt = $explodedPath[count($explodedPath) - 1];

        $this->logCustomMessage(
            'download_leave_attachment',
            $leaveAttachment,
            Auth::user()->name . ' downloaded a leave attachment form',
            $leaveAttachment,
            LeaveManagementLogType::DOWNLOAD_ATTACHMENT,
            new Activity()
        );

        if ($fileExt == 'enc') {
            return response()->streamDownload(function () use ($leaveAttachment) {
                FileVault::disk('leave_form')->streamDecrypt($leaveAttachment->file_path);
            });
        }

        
        return Storage::disk('leave_form')->download($leaveAttachment->file_path);
    }


    public function fetch_balances(Request $request)
    {

        try {

            $user = User::whereId(Auth::id())->without(['permissions', 'storage', 'roles', 'employeeMetaInfo'])->first();

            $sick_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::SICK_LEAVE)->first();
            $vacation_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::VACATION_LEAVE)->first();
            $forced_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::FORCED_LEAVE)->first();
            $special_balance = $user->leave_balance()->where('leave_type_id', EnumsLeaveType::SPECIAL_LEAVE)->first();

            $data = array(
                'user'  => $user,
                'sick_balance'  => $sick_balance->balance,
                'vacation_balance'  => $vacation_balance->balance,
                'forced_balance'  => $forced_balance->balance,
                'special_balance'  => $special_balance->balance,
            );

            return response()->json([
                'data'  => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the data'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    public function fetch_request(Request $req)
    {

        try {

            $leave = Leave::whereId($req->user()->id)->with([
                'user' => function ($query) {
                    $query->without(['permissions', 'roles']);
                },
            ])->first();

            return response()->json([
                'data'  => $leave
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function fetch_viewed_leave(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.leaves,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    =>  $validate->errors()
            ], 400);
        }

        try {
            $sector = Sector::where('status', 'active')->first();

            $leave = Leave::whereId($request->id)->with(['application', 'subtype', 'detail', 'monetization' => function ($query) {
                $query->with('monetized_leave');
            }])->first();

            $user = $leave->user()->with(['salary' => function ($query) {
                $query->without('user');
            }])->without(['permissions', 'roles', 'storage'])->first();

            $salary = $user->salary;
            if ($sector->sector == 1) {
                $salary = $salary->publicSalary->salaryGrade->value;
            } else {
                $salary = $salary->privateSalary->salary;
            }

            $first_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'first_name')->first()->field_value;
            $middle_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'middle_name')->first()->field_value;
            $last_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'last_name')->first()->field_value;

            $name = [
                'first_name'    => $first_name,
                'middle_name'   => $middle_name,
                'last_name'     => $last_name,
            ];

            $monetization = $leave->monetization()->with('monetized_leave')->first();

            $data = array(
                'leave'     => $leave,
                'name'          => $name,
                'monetizations' => $monetization,
                'user'      => $user,
                'salary'    => $salary,
            );

            return response()->json([
                'data'  => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the record'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    public function fetch_saved_leave(Request $request)
    {

        return;
        try {

            $sector = Sector::where('status', 'active')->first();

            $leave = Leave::where('user_id', $request->user()->id)->where('save_status', 1)->with([
                'subtype',
                'detail',
                'monetization'  => function ($query) {
                    $query->with('monetized_leave');
                }
            ])->first();

            if ($leave != null) {
                $user = $leave->user()
                    ->with([
                        'salary' => function ($query) {
                            $query->without('user');
                        }
                    ])
                    ->without(['permissions', 'roles', 'storage'])->first();

                $salary = $user->salary;
                if ($sector->sector == 1) {
                    $salary = $salary->publicSalary->salaryGrade->value;
                } else {
                    $salary = $salary->privateSalary->salary;
                }

                $first_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'first_name')->first()->field_value;
                $middle_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'middle_name')->first()->field_value;
                $last_name = $leave->user->personalInformation->employeeExtraFieldColumn()->where('field_name', 'last_name')->first()->field_value;

                $name = [
                    'first_name'    => $first_name,
                    'middle_name'   => $middle_name,
                    'last_name'     => $last_name,
                ];

                $monetization = $leave->monetization()->with('monetized_leave')->first();

                $data = array(
                    'leave'     => $leave,
                    'name'          => $name,
                    'monetizations' => $monetization,
                    'user'      => $user,
                    'salary'    => $salary
                );
            } else {
                $data = null;
            }


            return response()->json([
                'data'  => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the record'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    //fetchall
    public function fetch_all_request(Request $req)
    {
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        return Leave::with(['user' => function ($query) {
            $query->select('id', 'name')->without(['storage', 'roles', 'permissions', 'employeeMetaInfo', 'branch', 'division', 'department', 'position']);
        }, 'monetization', 'attachments'])->when($req->supervisor != null, function ($query) use ($req) {
            $query->where('supervisor_id', $req->supervisor);
        })->when(count($req->filter_dates) > 0, function ($query) use ($req) {
            //if filter_dates has FROM and TO
            $query->when(count($req->filter_dates) > 1, function ($query) use ($req) {
                $query->whereDate('start_date', '>=', $req->filter_dates[0])
                    ->whereDate('end_date', '<=', $req->filter_dates[1]);
            });
            //if filter_dates has only one date
            $query->when(count($req->filter_dates) == 1, function ($query) use ($req) {
                $query->whereDate('start_date', $req->filter_dates[0]);
            });
        })->when($req->filed_date != null, function ($query) use ($req) {

            $query->whereDate('date_filed', $req->filed_date);
        })->when($req->status != null, function ($query) use ($req) {
            $query->where('status', $req->status);
        })->where('user_id', $req->user()->id)->latest()->paginate($paginate);
    }

    //update
    public function update_request(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'id'               => 'required|exists:adg_db.leaves,id',
            'leave_subtype_id' => 'required',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            $leave = Leave::findOrFail($req->id);
            if ($leave->save_status != 2) {
                $leave->update([
                    'leave_subtype_id'     => $req->leave_subtype_id,
                    'leave_detail_id'   => $req->leave_detail_id,
                    'specified_details' => $req->specified_details,
                    'number_of_days'    => $req->number_of_days,
                    'save_status'       => $req->save_status,
                    'start_date'        => $req->start_date,
                    'end_date'          => $req->end_date,
                    'commutation'       => $req->commutation,
                    'created_by'        => $req->user()->id,
                    'updated_by'        => $req->user()->id,
                ]);

                if ($leave->leave_detail_id == 8) {
                    $leave->monetization()->update([
                        'type'  => $req->monetization['type'],
                        'vacation_days'  => $req->monetization['vacation_days'],
                        'sick_days'  => $req->monetization['sick_days']
                    ]);
                }
                $leave->save();
                DB::connection(env('ADG_DB_CONNECTION'))->commit();
                return response()->json([
                    'text' => 'Updated Successfully',
                ]);
            } else {
                return response()->json([
                    'errors'    => ['The selected leave is already submitted and cannot be updated.']
                ], 400);
            }
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    //delete
    public function delete_request(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            $delete = Leave::findOrFail($req->id);
            $delete->delete();
            DB::connection(env('ADG_DB_CONNECTION'))->commit();

            return response()->json([
                'text'  =>  'Leave Request has been deleted.',
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    public function fetch_types()
    {
        $types = LeaveSubtype::all();

        return response()->json([
            'data'  => $types
        ]);
    }

    public function fetch_details()
    {
        $vacation = LeaveDetail::where('purpose', 1)->get();
        $sick = LeaveDetail::where('purpose', 2)->get();
        $women = LeaveDetail::where('purpose', 3)->get();
        $study = LeaveDetail::where('purpose', 4)->get();
        $other = LeaveDetail::where('purpose', 5)->get();
        $monetization = LeaveDetail::where('id', 8)->first();
        $terminal = LeaveDetail::where('id', 9)->first();

        $details = array(
            'vacation'  => $vacation,
            'sick'  => $sick,
            'women'  => $women,
            'study'  => $study,
            'other'  => $other,
            'monetization'  => $monetization,
            'terminal'      => $terminal
        );

        return response()->json([
            'data'  => $details
        ]);
    }

    public function filter_request(Request $req)
    {
        try {
            $records = Leave::with(['user' => function ($q) {

                $q->select('id', 'name')->without(['storage', 'roles', 'permissions', 'employeeMetaInfo', 'branch', 'division', 'department', 'position']);
            }])->when($req->supervisor != null, function ($q) use ($req) {
                $q->where('supervisor_id', $req->supervisor);
            })->when(count($req->filter_dates) > 0, function ($q) use ($req) {
                //if filter_dates has FROM and TO
                $q->when(count($req->filter_dates) > 1, function ($q) use ($req) {
                    $q->whereDate('start_date', '>=', $req->filter_dates[0])
                        ->whereDate('end_date', '<=', $req->filter_dates[1]);
                });
                //if filter_dates has only one date
                $q->when(count($req->filter_dates) == 1, function ($q) use ($req) {
                    $q->whereDate('start_date', $req->filter_dates[0]);
                });
            })->when($req->filed_date != null, function ($q) use ($req) {

                $q->whereDate('date_filed', $req->filed_date);
            })->when($req->status != null, function ($q) use ($req) {
                $q->where('status', $req->status);
            })->where('user_id', $req->user()->id);

            return response()->json([
                'data' => $records->orderBy('created_at', 'asc')->paginate(10)
            ]);
        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    //  supervisor list
    public function supervisor_list()
    {
        $data = Supervisors::with(['user' => function ($q) {
            $q->select('id', 'name')->without('permissions', 'roles', 'storage', 'employeeMetaInfo');
        }])->get();

        return response()->json([
            'text' => 'fetch successful',
            'data' => $data,
        ]);
    }

    public function cancel_request(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            $leave = Leave::findOrFail($request->id);

            LeaveForCancellation::create([
                'leave_id'           =>  $leave->id,
                'leave_type_id'      =>  $leave->leave_type_id,
                'old_leave_status'  =>  $leave->status,
                'status'             =>  3,
                'created_by'         =>  Auth::id(),
                'created_at'         =>  now()
            ]);

            $data = Leave::where('id', $request->id)->first();

            LeaveForCancellation::where('leave_id', $data->id)->update([
                'old_leave_status'  =>  $data->status,
                'status'             =>  3,
                'updated_by'         => Auth::id(),
                'updated_at'         => now(),
            ]);

            if ($leave->status == 3 || $leave->status == 1) {
                $leave->update([
                    'status'       => 5,
                    'updated_by'   => Auth::id(),
                    'updated_at'   => now(),
                ]);
            }

            $updated = Leave::with('approvers')->find($request->id);

            $updated->old = collect($leave);
            $updated->attributes = collect($updated);
                    
            $this->logCustomMessage(
                'cancel_leave_request',
                $updated,
                Auth::user()->name . ' cancelled the request for ' . $updated->leaveType->title,
                $updated,
                LeaveManagementLogType::CANCEL,
                new Activity()
            );

            $leave->save();

            foreach ($updated->approvers as $receiver) {
                NotificationMessage::notifyReceiver('Request leave cancellation', $receiver->approver_id, $updated->user_id);
            }

            DB::commit();
            return response()->json([
                'text'  =>  'Leave Request has been updated.',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    // Fetch Leave Earnings and Deductions
    public function fetch_credits(Request $request)
    {

        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            $user = User::find(Auth::id());

            $type = $request->type;
            if ($type == 1) {
                $leave_remarks = __('leaves.vacation');
            } else if ($type == 2) {
                $leave_remarks = __('leaves.sick');
            } else if ($type == 3) {
                $leave_remarks = __('leaves.forced');
            } else {
                $leave_remarks = __('leaves.special-privilege');
            }

            $balance = $user->leave_balance()->where('leave_type_id', $type)->first();

            $fields = ['period', 'remarks', 'credits', 'resulting_balance', 'created_at', 'deleted_at'];
            $earning_fields = array_merge($fields, [DB::raw('"leave_credit_earnings" as source')]);
            $deduction_fields = array_merge($fields, [DB::raw('"leave_credit_deductions" as source')]);

            $earnings = LeaveCreditEarning::select($earning_fields)
                ->where('leave_balance_id', $balance->id)
                ->whereYear('period', '=', $request->year);

            $leaves = $user->leaves()->pluck('id')->toArray();

            $points = LeaveCreditDeduction::select($deduction_fields)
                ->where('remarks', $leave_remarks)
                ->whereHasMorph('deductable', Leave::class, function ($query) use ($leaves) {
                    $query->whereIn('id', $leaves)->select('status');
                })->withTrashed()
                ->whereYear('period', '=', $request->year)
                ->union($earnings)->orderBy('created_at')->paginate($paginate);

            return response()->json([
                'data'  => $points
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    public function fetch_leave_dates(Request $request){
        return Leave::where('user_id', Auth::id())->whereYear('start_date', $request->year)->whereMonth('start_date', $request->month)->where(function($query) {
            $query->where('status', LeaveStatus::APPROVED)->orWhere('status', LeaveStatus::FOR_REVIEW)->orWhere('status', LeaveStatus::FOR_CANCELLATION);
        })->with(['leaveType'])->get();
    }
}
