<?php

namespace Suiterus\Adg\Controllers\LearningDevelopment;

use Exception;
use Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Suiterus\Adg\Models\LearningDevelopment\TrainingBond;
use Suiterus\Adg\Models\SM\Department;
use App\Models\User;
use DateTime;
use Suiterus\Adg\Models\SM\EmployeeType;
use Suiterus\hrjp\Models\Position;

class TrainingBondController extends Controller
{
    public function create_trainingbond(Request $request) {

        $valid = Validator::make($request->all(),[
            'employee_id'           => 'required',
            'employee_type'         => 'required',
            'department'            => 'required',
            'position'              => 'required',
            'date_from'             => 'required|date',
            'date_to'               => 'required|date',
            'status'               => 'required',
            'wave_option'           => 'required'
        ],[
            'name.unique'   =>  $request->name.' is already taken.',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        $date_from = new DateTime($request->date_from);
        $date_to = new DateTime($request->date_to);
        $duration = $date_from->diff($date_to);
        DB::beginTransaction();
        try {
            $department = TrainingBond::create([
                'user_id'       => $request->employee_id,
                'employee_type' => $request->employee_type,
                'department'    => $request->department,
                'position'      => $request->position,
                'date_from'     => $request->date_from,
                'date_to'       => $request->date_to,
                'status'        => $request->status,
                'duration'      => $duration->days,
                'wave_option'   => $request->wave_option,
                'created_by'    => $request->user()->id,
                'updated_by'    => $request->user()->id,
                'created_at'    => now()
            ]);
            DB::commit();
            $user = $request->user();
            return response()->json([
                'text'  =>  'Training Bond created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function update_trainingbond(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'                    => 'required|exists:adg_db.training_bonds,id',
            'employee_id'               => 'required',
            'employee_type'         => 'required',
            'department'            => 'required',
            'position'              => 'required',
            'date_from'             => 'required|date',
            'date_to'               => 'required|date',
            'status'               => 'required',
            'wave_option'           => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }
        $date_from = new DateTime($request->date_from);
        $date_to = new DateTime($request->date_to);
        $duration = $date_from->diff($date_to);
        DB::beginTransaction();
        try {

            $TrainingBond = TrainingBond::findOrFail($request->id);
            $TrainingBond->update([
                'user_id'       => $request->employee_id,
                'employee_type' => $request->employee_type,
                'department'    => $request->department,
                'position'      => $request->position,
                'date_from'     => $request->date_from,
                'date_to'       => $request->date_to,
                'status'        => $request->status,
                'duration'      => $duration->days,
                'wave_option'   => $request->wave_option,
                'updated_by'    => $request->user()->id,
                'updated_at'    => now()
            ]);
            $TrainingBond->save();
            DB::commit();
            return response()->json([
                'text'  => 'Training Bond updated successfully!'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Training Bond does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in updating the Training Bond'],
                'message'   => $e->getMessage()
            ], 500);
        }


    }

    public function delete_trainingbond(Request $request) {

        $valid = Validator::make($request->all(), [
            'id'            => 'required|exists:adg_db.training_bonds,id',
        ]);

        if($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $TrainingBond = TrainingBond::findOrFail($request->id);
            $user_id = $TrainingBond->user_id;
            $TrainingBond->delete();
            DB::commit();
            return response()->json([
                'text'  => 'Training Bond deleted successfully!'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Training Bond does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with deleting the Training Bond.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
    
    public function init_training_bond(Request $req){
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  TrainingBond::with(['user', 'department', 'employeeType', 'position'])->orderBy('id', 'desc')->paginate($paginate)
        ]);
    }


    // FILTERING OF RECORDS
    public function filter(Request $req){
        $paginate = $req->paginate ? intval($req->paginate) : 10;
        try{
            $users = TrainingBond::whereHas('user', function($query) use ($req) {
                $query->select('id', 'name')->where('name', 'LIKE', '%' . $req->employee_name . '%')
                    ->without(['storage', 'roles', 'permissions'])
                    ->whereHas('employeeMetaInfo')
                    ->with(['employeeMetaInfo' => function($query) {
                        $query->select('id','user_id', 'employee_id')->without('employeeType','corporation','division','department','branch'); 
                    }]);
            })
            ->with(['user'])
            ->when($req->position!=null, function($q) use ($req){
                $q->where('position', $req->position);
            })->when($req->employee_type!=null, function($q) use ($req){
                $q->where('employee_type', $req->employee_type);
            })->when($req->department!=null, function($q) use ($req){
                $q->where('department', $req->department);
            })
            ->with(['user', 'department', 'employeeType', 'position']);

            //Employee filter
            if(count($req->employees) > 0) {
                $users->whereIn('user_id', $req->employees);
            }

            return response()->json([
                'data' => $users->orderBy('created_at', 'asc')->paginate($paginate)
            ]);
        
        } 
        catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
}

