<?php

namespace Suiterus\Adg\Controllers\ExitManagement\ExitInterview;
use DB;
use Exception;
use Validator;
use App\Models\User;
use App\Enums\Status;
use Illuminate\Http\Request;
use App\Enums\ClearanceStatus;
use App\Enums\ExitInterviewStatus;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\SM\Remark;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Models\PDS\PDSHistory;
use Suiterus\Adg\Models\SM\SalaryGrade;
use App\Enums\Log\ExitManagementLogType;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;
use Suiterus\Adg\Controllers\PDS\PDSHistoryController;
use Suiterus\Adg\Models\HRExit\ExitInterview\ExitInterview;
use Suiterus\Adg\Models\HRExit\ExitInterview\ExitInterview as EI;

class ExitInterviewController extends Controller
{
    use HasCustomLogs;
    
    //create
    public function create_exit_interview(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'user_id'    => 'required|integer|exists:mysql.users,id',
            'date'    => 'required|date'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {
            $file = $req->file('attachment');
            $filename = $file === null ? null : Crypt::encryptString(microtime()) . "." . $file->getClientOriginalExtension();

            $interview = EI::create([
                'user_id'           => $req->user_id,
                'date'              => $req->date,
                'status'            => ExitInterviewStatus::ONGOING,
                'attachment'        => $filename,
                'remarks'           => $req->remarks,
            ]);

            if ($file !== null) {
                Storage::disk('employee')->putFileAs($interview->user_id, $file, $filename);
            }

            $this->logCustomMessage(
                'create_exit_interview',
                $interview,
                Auth::user()->name . ' created an exit interview for ' . $interview->user->name,
                $interview,
                ExitManagementLogType::CREATE_INTERVIEW,
                new Activity()
            );

            return response()->json([
                'message' => "Created successfully!",
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your Clearance as of now. Contact the developer to fix it. '],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //fetchall
    public function fetch_all_interviews(Request $req)
    {

        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        $data = EI::whereHas('user', function ($query) use ($req) {
            $query->where('name', 'LIKE', '%' . $req->employee_name . '%');
        })
            ->with(['user' => function ($q) {
                $q->select('id', 'name')->without('roles', 'permissions', 'storage', 'employeeMetaInfo', 'supervisor', 'exitInterview');
            }])
            ->when(isset($req->date) && $req->date !== null, function ($query) use ($req) {
                $query->whereDate('date', $req->date);
            })
            ->paginate($paginate);

        return response()->json([
            'data'  => $data
        ]);
    }

    //fetch single record
    public function fetch_single_interviews(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'id' => 'required|exists:adg_db.exit_interviews,id',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        try {
            $single_record = EI::where('id', $req->id)->first();
            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $single_record,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your record as of now.'],
                'msg' => $e->getMessage(),
            ]);
        }
    }

    //update
    public function update_interviews(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'id'    => 'required|exists:adg_db.exit_interviews,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            $file = $req->file('attachment');

            $filename = $file === null ? null : Crypt::encryptString(microtime()) . "." . $file->getClientOriginalExtension();

            $interview = EI::findOrfail($req->id);

            $file_path = $interview->user_id . '/' . $interview->attachment;


            $interview->update([
                'date'              => $req->date,
                'status'            => $req->status,
                'attachment'        => $filename,
                'remarks'           => $req->remarks,
                'updated_by'        => $req->user()->id,
            ]);

            Storage::disk('employee')->delete($file_path);


            if ($file !== null) {
                Storage::disk('employee')->putFileAs($interview->user_id, $file, $filename);
            }

            $updated = EI::find($req->id);

            $updated->old = collect($interview);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_exit_interview',
                $updated,
                Auth::user()->name . ' updated the exit interview for ' . $updated->user->name,
                $updated,
                ExitManagementLogType::UPDATE_INTERVIEW,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'message' => "Updated successfully!",
            ]);
        } catch (Exception $e) {

            return response()->json(
                [
                    'errors' => ['Can`t create your Clearance as of now. Contact the developer to fix it. Error Code : '],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //delete
    public function delete_interviews(Request $req)
    {

        DB::beginTransaction();

        $valid = Validator::make($req->all(), [
            'id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        try {

            $delete = EI::findOrFail($req->id);

            $this->logCustomMessage(
                'delete_exit_interview',
                $delete,
                Auth::user()->name . ' deleted the exit interview of ' . $delete->user->name,
                $delete,
                ExitManagementLogType::DELETE_INTERVIEW,
                new Activity()
            );

            $delete->delete();
            DB::commit();

            return response()->json([
                'text'  =>  'Exit Invterview has been deleted.',

            ]);
        } catch (Exception $e) {

            DB::rollback();

            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }


    public function fetch_users_with_clearance()
    {

        try {

            $users = User::whereHas('clearance', function ($query) {
                $query->where('status', ClearanceStatus::APPROVED);
            })
                ->with(['employeeMetaInfo' => function ($query) {
                    $query->without(['corporation', 'branch', 'division', 'department']);
                }])
                ->without(['roles', 'permissions', 'storage', 'user_supervisor'])->get();
            return response()->json([
                'data'  =>  $users
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function set_as_done(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $record = EI::findOrFail($req->id);
            $record->status = ExitInterviewStatus::DONE;

            $updated = EI::findOrFail($req->id);

            $updated->old = collect($record);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'done_exit_interview',
                $updated,
                Auth::user()->name . ' set the exit interview for ' . $updated->user->name . ' as done',
                $updated,
                ExitManagementLogType::DONE_INTERVIEW,
                new Activity()
            );

            $record->save();
            
            DB::commit();
            return response()->json([
                'text'  =>  'Exit Invterview has been set as done.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function fetch_users_interview_done()
    {
        try {
            $exitInterviews = User::whereHas('exitInterview', function ($query){
                $query->where('status', 2);
            })
                ->with(['user_supervisor','employeeMetaInfo' => function ($query) {
                    $query->without(['corporation', 'branch', 'division', 'department']);
                }])
                ->without(['roles', 'permissions', 'storage'])->get();

                $pendingClearances = User::whereHas('clearance', function ($query) {
                    $query->where('status', ClearanceStatus::PENDING);
                })->get()->pluck(['id']);

                $approveClearances = User::whereHas('clearance', function ($query) {
                    $query->where('status', ClearanceStatus::APPROVED);
                })->get()->pluck(['id']);

                $ongoingClearances = User::whereHas('clearance', function ($query) {
                    $query->where('status', ClearanceStatus::ONGOING);
                })->get()->pluck(['id']);

                $verifyClearances = User::whereHas('clearance', function ($query) {
                    $query->where('status', ClearanceStatus::VERIFICATION);
                })->get()->pluck(['id']);


            $pendingIDs = array_filter(json_decode($exitInterviews), function($exitInterview) use($pendingClearances){
                return !in_array($exitInterview->id, json_decode($pendingClearances));

            });
            $approvedIDs = array_filter($pendingIDs , function($exitInterview) use($approveClearances){
                return !in_array($exitInterview->id, json_decode($approveClearances));
            });
            $ongoingIDs = array_filter($approvedIDs , function($exitInterview) use($ongoingClearances){
                return !in_array($exitInterview->id, json_decode($ongoingClearances));
            });
            $verifyIDs = array_filter($ongoingIDs , function($exitInterview) use($verifyClearances){
                return !in_array($exitInterview->id, json_decode($verifyClearances));
            });

            $employees = [];

            foreach($pendingIDs as $pendingID) {
                
                foreach($approvedIDs as $approveID) {
                    
                    foreach($ongoingIDs as $ongoingID) {
                    
                        foreach($verifyIDs as $verifyID) {
                    
                            array_push($employees, $verifyID);
                        }
                    }

                }
            }

            

            return response()->json([
                'data' => $employees,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function file_attachment_download(Request $req){
        $valid = Validator::make($req->all(),[
            'id' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'error' => $valid->errors()
            ], 400);
        }

        $exit_interview = ExitInterview::where('id' , $req->id)->first();

        $this->logCustomMessage(
            'download_file_attachment',
            $exit_interview,
            Auth::user()->name . ' downloaded the file attachment',
            $exit_interview,
            ExitManagementLogType::DOWNLOAD,
            new Activity()
        );

        return Storage::download('employees/' .  $exit_interview->user_id . '/' . $exit_interview->attachment);
    }

    public function CCS_list()
    {
        return response()->json([
            'data' => CCS::all()
        ]);
    }

}
