<?php

namespace Suiterus\Adg\Controllers\ExitManagement;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use Illuminate\Http\Request;
use App\Enums\ClearanceStatus;
use App\Enums\ExitInterviewStatus;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Remark;
use Suiterus\Adg\Models\SM\Sector;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\SalaryGrade;
use App\Enums\Log\ExitManagementLogType;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\HRExit\Clearance;
use Suiterus\Adg\Models\SM\ScheduleTitle;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\Salary\PrivateSalary;
use Suiterus\Adg\Models\SM\DesignationHistory;
use Suiterus\Adg\Models\EMI\EmployeeExtraField;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;
use Suiterus\Adg\Models\Timekeeping\EmployeeSchedule;
use Suiterus\Adg\Models\HRExit\ExitInterview\ExitInterview;
use Suiterus\Adg\Controllers\PDS\PersonalDataSheetController;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;


class ExitController extends Controller
{
    use HasCustomLogs;

    public function fetch_name(Request $req){

        $validate = Validator::make($req->all(), [
            'id'        => 'required|exists:mysql.users,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $user = User::whereId($req->id)
                ->with(['clearance' => function ($query) {
                    $query->with('ClearanceHasForm');
                }])
                ->without(['roles', 'permissions', 'storage'])
                ->first();
            
            return response()->json([
                'data'  => $user
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // FILTER FUNCTION
    public function filter_employees_resigned(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');

        try {

            $users = User::where('status', Status::INACTIVE)->whereHas('clearance', function ($query) use($req) {
                $query->where('status', ClearanceStatus::APPROVED);
            })
            ->whereHas('employeeMetaInfo', function ($query) use ($req) {
                    $query->when($req->department !== null, function ($query) use ($req) {
                        return $query->where('department_id', $req->department);
                    })
                        ->when($req->employee_type !== null, function ($query) use ($req) {
                            return $query->where('employee_type', $req->employee_type);
                        })
                        ->when($req->position !== null, function ($query) use ($req) {
                            return $query->where('position_id', $req->position);
                        });
                })->with(['clearance' => function($query){
                    $query->orderBy('id', 'desc');
                }])
                
                ->without(['roles', 'permissions', 'storage', 'user_supervisor'])
                ->when(isset($req->supervisor) && $req->supervisor != 0, function ($query) use ($req) {
                    $query->whereHas('user_supervisor.supervisor.user', function ($query) use ($req) {
                        $query->whereId($req->supervisor);
                    });
                })
                ->where('name', 'LIKE', '%' . $req->employee_name . '%')->orderBy('updated_at', 'desc');  
                

            return response()->json([
                'data' => $users->paginate($paginate)
            ]);
        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function reemploy(Request $request){
        
        DB::beginTransaction();
        try{

            $activeSector = Sector::where('status', 'active')->first();
            $sector = $activeSector->name;
            
            $resigned_employee = Clearance::whereId($request->id)->where('status', ClearanceStatus::APPROVED)->first();

            EmployeeExtraFieldColumn::whereHas('employeeExtraField', function($query) use($resigned_employee){
                $query->where('user_id', $resigned_employee->user_id);
            })->delete();
            
            EmployeeExtraField::where('user_id', $resigned_employee->user_id)->delete();

            $user = User::whereId($resigned_employee->user_id)->first();

            $createPDS = new PersonalDataSheetController();
            $createPDS->create($request, false, $user);

            $personalInformation = $request->personalInformation;
            foreach ($personalInformation['personalInformationForm']['info'] as $key) {
                if (array_key_exists('entityName', $key)) {
                    if ($key['entityName'] == 'first_name') {
                        $first_name = $key['value'];
                    }else if ($key['entityName'] == 'last_name') {
                        $last_name = $key['value'];
                    }else if ($key['entityName'] == 'email') {
                        $email = $key['value'];
                    }else if ($key['entityName'] == 'employee_no') {
                        $employee_no = $key['value'];
                    }
                }
            }

            User::whereId($resigned_employee->user_id)->update([
                'name'     =>   $first_name . ' ' . $last_name,
                'email'    =>   $email, 
                'password' =>   bcrypt('user'),
                'status'   =>   1 
            ]);

            $employeeInfo = EmployeeMetaInfo::where('user_id', $resigned_employee->user_id)->first();

            if($request->sector == 'public'){
                $division = $request->companyDetails['form'][0]['value'];
                $department = $request->companyDetails['form'][1]['value'];
                $employeeType = $request->employeeTypePosition['form'][0]['value'];
                $position = $request->employeeTypePosition['form'][1]['value'];
                EmployeeMetaInfo::where('user_id', $employeeInfo->user_id)->update([
                    'user_id'           => $resigned_employee->user_id,
                    'employee_id'       => $employeeInfo->employee_id,
                    'employee_type'     => $employeeType,
                    'division_id'       => $division,
                    'department_id'     => $department,
                    'salary_id'         => null,
                    'date_hired'        => date('Y-m-d'), 
                    'position_id'       => $position,
                    'modified_by'       => Auth::id()
                ]);

            }else{
            $corporation = $request->companyDetails['form'][0]['value'];
            $branch = $request->companyDetails['form'][1]['value'];
            $division = $request->companyDetails['form'][2]['value'];
            $department = $request->companyDetails['form'][3]['value'];
            $employeeType = $request->employeeTypePosition['form'][0]['value'];
            $position = $request->employeeTypePosition['form'][1]['value'];
            EmployeeMetaInfo::where('user_id', $employeeInfo->user_id)->update([
                'user_id'           => $resigned_employee->user_id,
                'employee_id'       => $employeeInfo->employee_id,
                'employee_type'     => $employeeType,
                'corp_id'           => $corporation,
                'branch_id'         => $branch,
                'division_id'       => $division,
                'department_id'     => $department,
                'salary_id'         => null,
                'date_hired'        => date('Y-m-d'), 
                'position_id'       => $position,
                'modified_by'       => Auth::id()
            ]);
            }

            DB::commit();
            DB::beginTransaction();

            if($request->sector == 'public'){
            $this->designation_reemployment($resigned_employee, null, null, $division, $department);
            }else{
                $this->designation_reemployment($resigned_employee, $corporation, $branch, $division, $department);
            }
            $active_schedule = ScheduleTitle::where('active_schedule', 1)->first();
        
            EmployeeSchedule::where('user_id', $resigned_employee->user_id)->update([
                'user_id'               => $resigned_employee->user_id,
                'schedule_template_id'  => $active_schedule->id,
                'updated_by'            => Auth::id()
            ]);

            $totalAllowance = 0;
            
            $allowance = EmployeeAllowance::where('user_id', $resigned_employee->user_id)->get();

            foreach($allowance as $allowances) {
                $totalAllowance += $allowances['amount'];
            }

            if($allowance === null){
                $allAllowances = 0;
            }
            else{
                $allAllowances = $totalAllowance;
            }
            
            if($sector == 'public'){
                $newSalary = SalaryGrade::whereHas('publicSalary', function($query) use($request){
                    $query->whereHas('userSalary', function($query) use($request){
                        $query->where([
                            ['user_id', $request->user_id]
                        ]);
                    });
                })->first();

                $salary = $newSalary->value ?? 0;
                $salary = str_replace( ',', '', $salary);
                $annualSalary = $salary * 12;

            }
            else{
                $newSalary = PrivateSalary::whereHas('userSalary', function ($query) use($request){
                $query->where('user_id', $request->user_id);
            })->first();

                $salary = $newSalary->salary;
                $annualSalary = $salary * 12;
            }
            
            $remark = Remark::whereId(26)->first();
            $serviceRecord = ServiceRecord::create([
                'user_id'           =>  $resigned_employee->user_id,
                'start_date'        =>  $resigned_employee->updated_at, 
                'position_id'       =>  $employeeInfo->position_id,
                'salary'            =>  $annualSalary,
                'allowance'         =>  $allAllowances,
                'employee_type_id'  =>  $employeeInfo->employee_type,
                'division_id'       =>  $employeeInfo->division_id,
                'remark_id'         =>  $remark->id,  //* Re-employment
                'created_by'        =>  Auth::id(),
            ]);

            User::where('id', $resigned_employee->id)->update(['status' => Status::ACTIVE]);

            $this->logCustomMessage(
                're_employ_employee',
                $employeeInfo,
                Auth::user()->name . ' re-employed ' . $employeeInfo->user->name,
                $employeeInfo,
                ExitManagementLogType::RE_EMPLOY,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  'Successfully re-employed',
                'user'  =>   $user->id
            ]);
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage(),
                'line'  => $e->getLine()
            ],500);
        }
    }

    public function designation_reemployment($resigned_employee, $corporation, $branch, $division, $department){
        try{
            $history = DesignationHistory::where('user_id', $resigned_employee->user_id)->orderBy('created_at', 'desc')->first();
        
            if(isset($history->date_of_designation) && $history->date_of_designation != null){
                $date1 = Carbon::now(); 
                $date1->toDateString();
                $date2 = Carbon::createFromFormat('Y-m-d', strval($history->date_of_designation)); 

                if($date1->gt($date2)){
                    $status = 2; // for 'previous'
                    DesignationHistory::where('user_id', $resigned_employee->user_id)->orderBy('id', 'desc')->first()
                    ->update([
                        'status'        =>  $status,
                        'updated_by'    =>  Auth::id()
                    ]);
                }
            }

            $designation = DesignationHistory::create([
                'user_id'               =>  $resigned_employee->user_id,
                'corp_id'               =>  $corporation,
                'branch_id'             =>  $branch,
                'division_id'           =>  $division,
                'department_id'         =>  $department,
                'date_of_designation'   =>  date('Y-m-d'),
                'status'                =>  1,
                'employment_status'     =>  1, //* re-employed
                'created_by'            =>  Auth::id()
            ]);

        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
        
    }

    public function fetch_users_clearance_approved()
    {
        try {

            $users = User::whereHas('clearance', function ($query) {
                $query->where('status', ClearanceStatus::APPROVED);
            })
                ->with(['employeeMetaInfo' => function ($query) {
                    $query->without(['corporation', 'branch', 'division', 'department']);
                }])
                ->without(['roles', 'permissions', 'storage', 'user_supervisor'])->get();
            return response()->json([
                'data'  =>  $users
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }

    }

}
