<?php

namespace Suiterus\Adg\Controllers\ExitManagement\Employee;
use DB;

use Exception;
use Validator;
use App\Models\User;
use Illuminate\Http\Request;
use App\Enums\ClearanceStatus;
use App\Traits\Logs\HasCustomLogs;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Storage;
use App\Enums\Log\ExitManagementLogType;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\HRExit\clearance as CS;
use Suiterus\Adg\Models\HRExit\ClearanceCompliance;
use Suiterus\Adg\Models\HRExit\ClearanceCompliance as CCS;

class ClearanceFormController extends Controller
{
    use HasCustomLogs;

    // Fetch Active Clearance
    public function fetchActiveClearance(Request $request)
    {

        try {

            $user = User::findOrFail($request->user()->id);

            $clearance = $user->clearance()
                ->with([
                    'ClearanceHasForm' => function ($query) {
                        $query->with(['department', 'assigned_personnel' => function ($query) {
                            $query->without('roles', 'permissions', 'storage', 'employeeMetaInfo');
                        }]);
                    },
                    'user' => function ($query) {
                        $query->with('user_supervisor')->without('roles', 'permissions', 'storage', 'employeeMetaInfo');
                    }
                ])->first();

            return response()->json([
                'data'  => $clearance
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in getting the data'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    //update
    public function update_clearance(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'id'    => 'required|exists:adg_db.clearances,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $clearance = CS::findOrfail($req->id);

            $clearance->update([
                'status' => ClearanceStatus::VERIFICATION,
                'updated_by' => $req->user()->id,
            ]);

            $updated = CS::find($req->id);

            $updated->old = collect($clearance);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_clearance_form',
                $updated,
                Auth::user()->name . ' updated the clearance form',
                $updated,
                ExitManagementLogType::UPDATE_CLEARANCE,
                new Activity()
            );

            $records_to_keep = [];
            foreach ($req->compliance_list as $compliance) {
                $file = $compliance['attachment'];

                $filename = $file === null ? null : Crypt::encryptString(microtime()) . "." . $file->getClientOriginalExtension();

                $compliance_record = CCS::find($compliance['id']);
                $compliance_record->update([

                    'attachment'    => $filename,
                    'updated_by'    => $req->user()->id
                ]);

                $compliance_record->save();

                $file_path = $clearance->user_id . '/' . $compliance_record->attachment;
                Storage::disk('employee')->delete($file_path);
                array_push($records_to_keep, $compliance_record->id);

                if ($file !== null) {
                    $path = storage_path() . '/app/employees/' . $clearance->user_id;
                    $file->move($path, $filename);
                }
            }

            $clearance->ClearanceHasForm()->whereNotIn('id', $records_to_keep)->delete();
            $clearance->save();

            DB::commit();
            return response()->json([
                'message' => "Updated successfully!",
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your Clearance as of now. Contact the developer to fix it. Error Code : '],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function file_attachment_download(Request $req)
    {
        $validate = Validator::make($req->all(),[
            'id' => 'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'error' => $validate->errors()
            ], 400);
        }

        $compliance_record = ClearanceCompliance::where('id' , $req->id)->first();

        $this->logCustomMessage(
            'download_file_attachment',
            $compliance_record,
            Auth::user()->name . ' downloaded the file attachment',
            $compliance_record,
            ExitManagementLogType::DOWNLOAD,
            new Activity()
        );

        return Storage::download('employees/' .  $compliance_record->user_id . '/' . $compliance_record->attachment);
    }
}
