<?php

namespace Suiterus\Adg\Controllers\ExitManagement;
use Exception;

use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use InvalidArgumentException;
use App\Enums\ClearanceStatus;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Storage;
use App\Enums\Log\ExitManagementLogType;
use App\Enums\Status;
use App\Http\Classes\NotificationMessage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Adg\Models\HREXIT\clearance as CS;
use Suiterus\Adg\Models\HRExit\ClearanceCompliance;
use Suiterus\Dms\Models\Repositories\SectionAccess;
use Suiterus\Adg\Controllers\PDS\PDSHistoryController;
use Suiterus\Adg\Models\HRExit\ClearanceCompliance as CCS;
use Suiterus\Adg\Models\LongevityPay;

class ClearanceController extends Controller
{
    use HasCustomLogs;

    public function create_clearance(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'user_id'       => 'required|integer|exists:mysql.users,id',
            'start_date'    => 'required|date',
            'sent_to_employee' => 'required',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        DB::beginTransaction();
        try {
            $clearance = CS::create([
                'user_id'           => $req->user_id,
                'start_date'        => $req->start_date,
                'status'            => ClearanceStatus::PENDING,
                'sent_to_employee'  => $req->sent_to_employee,

                'created_by'        => $req->user()->id,
            ]);

            foreach ($req->compliance_list as $compliance) {

                $file = $compliance['attachment'];
                $filename = $file === null ? null : Crypt::encryptString(microtime()) . "." . $file->getClientOriginalExtension();

                CCS::create([
                    'user_id'               => $clearance->user_id,
                    'clearance_id'          => $clearance->id,
                    'department_id'         => $compliance['department_id'],
                    'assigned_personal_id'  => $compliance['assigned_personal_id'],
                    'action_item'           => $compliance['action_item'],
                    'attachment'            => $filename,
                    'timestamp'             => now(),
                    'created_by'            => $req->user()->id,
                ]);

                if ($file !== null) {
                    $path = storage_path() . '/app/employees/' . $clearance->user_id;
                    $file->move($path, $filename);
                }
            }

            $this->logCustomMessage(
                'create_clearance_form',
                $clearance,
                Auth::user()->name . ' created a clearance form for ' . $clearance->user->name,
                $clearance,
                ExitManagementLogType::CREATE_CLEARANCE,
                new Activity()
            );

            $clearance->save();

            NotificationMessage::notifySenderAndReceiver('Create Clearance', $req->user_id, Auth::id());
            DB::commit();
            return response()->json([
                'text' => 'Clearance created successfully!.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your Clearance as of now. Contact the developer to fix it. Error Code : '],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    // List all clearances
    public function list_clearances(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');

        $clearances = CS::when(isset($request->status) && $request->status !== null, function ($query) use ($request) {
            $query->where('status', $request->status);
        })
            ->when(isset($request->employee_name), function ($query) use ($request) {
                $query->whereHas('user', function ($query) use ($request) {
                    $query->where('name', 'LIKE', '%' . $request->employee_name . '%');
                });
            })
            ->with(['user' => function ($query) {
                $query->without(['roles', 'storage', 'permissions', 'supervisor'])
                    ->with(['employeeMetaInfo' => function ($query) {
                        $query->without('branch', 'corporation', 'division', 'department', 'position')
                            ->select('user_id', 'employee_id');
                    }, 'user_supervisor']);
            }, 'ClearanceHasForm'])
            ->paginate($paginate);

        return response()->json([
            'data'  => $clearances
        ]);
    }

    // Fetch single clearance
    public function fetch_clearance(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.clearances,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {
            $clearance = CS::whereId($request->id)->with(['user' => function ($query) {
                $query->without(['roles', 'storage', 'permissions'])
                    ->with(['employeeMetaInfo' => function ($query) {
                        $query->without('branch', 'corporation', 'division', 'department', 'position')
                            ->select('user_id', 'employee_id');
                    }, 'user_supervisor']);
            }, 'ClearanceHasForm'])->first();

            return response()->json([
                'data'  => $clearance
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the clearance'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }


    // Approve Clearance
    public function approve(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.clearances,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $clearance = CS::find($request->id);

            if ($clearance->status == ClearanceStatus::APPROVED) {
                throw new InvalidArgumentException('The clearance is already approved');
            }

            if ($clearance->status == ClearanceStatus::ONGOING) {
                throw new InvalidArgumentException('The employee is still uploading the necessary documents');
            }

            $clearance->status = ClearanceStatus::APPROVED;

            $updated = CS::find($request->id);

            $updated->old = collect($clearance);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'approve_clearance_form',
                $updated,
                Auth::user()->name . ' approved the clearance form of ' . $updated->user->name,
                $updated,
                ExitManagementLogType::APPROVE_CLEARANCE,
                new Activity()
            );
            
            $clearance->save();

            $resign = new PDSHistoryController;
            $resign->resignation($clearance);

            $user = User::where('id',$clearance->user_id)->first();

            $inactiveFolderId = Section::where("name","Inactive Employee")->pluck('id')->first();

            $longevity_pays = LongevityPay::where('user_id', $user->id)->get();

            foreach($longevity_pays as $pay){
                $pay->update(['status' => Status::INACTIVE]);
                $pay->delete();
            }

            User::where('id', $clearance->user_id)->update(['status' => Status::INACTIVE]);

            Section::where('name',EmployeeMetaInfo::where('user_id',$user->id)->pluck('employee_id')->first().' - '.$user->name)->update(['parent_id' => $inactiveFolderId]);
            
            UserSalary::where('user_id', $clearance->user_id)->update([
                'status'    =>  1 // inactive
            ]);

            NotificationMessage::notifySenderAndReceiver('Approve Clearance', $clearance->user_id, Auth::id());
            DB::commit();
            return response()->json([
                'text'  => 'Clearance approved.'
            ]);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => $e->getMessage(),
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // Approve all clearance
    public function batch_approve(Request $request)
    {

        DB::beginTransaction();
        try {

            $to_approve = [
                ClearanceStatus::VERIFICATION,
                ClearanceStatus::PENDING,
            ];

            $clearances = CS::whereIn('status', $to_approve)->get();

            foreach ($clearances as $clearance) {
                $clearance->status = ClearanceStatus::APPROVED;

                $updated = CS::find($clearance->id);

                $updated->old = collect($clearance);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'approve_all_clearance_form',
                    $updated,
                    Auth::user()->name . ' approved the clearance form of ' . $updated->user->name,
                    $updated,
                    ExitManagementLogType::APPROVE_ALL_CLEARANCE,
                    new Activity()
                );

                $clearance->save();

                NotificationMessage::notifyReceiver('Approve Clearance', $clearance->user_id, Auth::id());
            }

            DB::commit();
            return response()->json([
                'text'  => 'All clearances approved.'
            ]);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => $e->getMessage(),
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // Decline Clearance
    public function decline(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.clearances,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $clearance = CS::find($request->id);

            if ($clearance->status == ClearanceStatus::DECLINED) {
                throw new InvalidArgumentException('The clearance is already declined');
            }

            if ($clearance->status == ClearanceStatus::ONGOING) {
                throw new InvalidArgumentException('The employee is still uploading the necessary documents');
            }

            $clearance->status = ClearanceStatus::DECLINED;

            $updated = CS::find($clearance->id);

            $updated->old = collect($clearance);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'decline_clearance_form',
                $updated,
                Auth::user()->name . ' declined the clearance form of ' . $updated->user->name,
                $updated,
                ExitManagementLogType::DECLINE_CLEARANCE,
                new Activity()
            );

            $clearance->save();
            
            NotificationMessage::notifySenderAndReceiver('Disapprove Clearance', $clearance->user_id, Auth::id());
            DB::commit();
            return response()->json([
                'text'  => 'Clearance declined.'
            ]);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => $e->getMessage(),
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // Send Clearance
    public function send(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.clearances,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $clearance = CS::find($request->id);

            if ($clearance->status == ClearanceStatus::ONGOING) {
                throw new InvalidArgumentException('The employee is still uploading the necessary documents');
            }

            $clearance->status = ClearanceStatus::ONGOING;

            $updated = CS::find($request->id);

            $updated->old = collect($clearance);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'send_clearance_form',
                $updated,
                Auth::user()->name . ' sent the clearance form of ' . $updated->user->name,
                $updated,
                ExitManagementLogType::SEND_CLEARANCE,
                new Activity()
            );

            $clearance->save();
            DB::commit();
            return response()->json([
                'text'  => 'Clearance sent.'
            ]);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => $e->getMessage(),
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    //UPDATE FUNCTION CLEARANCE
    public function update_clearance(Request $req)
    {

        $validate = Validator::make($req->all(), [
            'id'    => 'required|exists:adg_db.clearances,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $clearance = CS::find($req->id);

            if ($clearance->status == ClearanceStatus::ONGOING) {
                throw new InvalidArgumentException('The employee is still uploading the necessary documents');
            }

            $clearance->status = ClearanceStatus::ONGOING;

            $clearance->update([
                'start_date' => Carbon::now(),
                'updated_by' => $req->user()->id,
            ]);

            $records_to_keep = [];
            foreach ($req->compliance_list as $compliance) {
                $file = $compliance['attachment'];
                $filename = $file === null ? null : Crypt::encryptString(microtime()) . "." . $file->getClientOriginalExtension();
                if ($compliance['id'] === null) {
                    CCS::create([
                        'user_id'               => $clearance->user_id,
                        'clearance_id'          => $clearance->id,
                        'department_id'         => $compliance['department_id'],
                        'assigned_personal_id'  => $compliance['assigned_personal_id'],
                        'action_item'           => $compliance['action_item'],
                        'attachment'            => $filename,
                        'timestamp'             => now(),
                        'created_by'            => $req->user()->id,
                    ]);
                } else {

                    $compliance_record = CCS::findOrFail($compliance['id']);
                    $compliance_record->update([
                        'department_id' => $compliance['department_id'],
                        'assigned_personal_id'  => $compliance['assigned_personal_id'],
                        'action_item'   => $compliance['action_item'],
                        'attachment'    => $filename,
                        'updated_by'    => $req->user()->id
                    ]);

                    $compliance_record->save();
                    $file_path = $clearance->user_id . '/' . $compliance_record->attachment;
                    Storage::disk('employee')->delete($file_path);
                    array_push($records_to_keep, $compliance_record->id);
                }
                if ($file !== null) {
                    $path = storage_path() . '/app/employees/' . $clearance->user_id;
                    $file->move($path, $filename);
                }
            }

            $clearance->ClearanceHasForm()->whereNotIn('id', $records_to_keep)->delete();

            $updated = CS::find($req->id);

            $updated->old = collect($clearance);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_clearance_form',
                $updated,
                Auth::user()->name . ' updated the clearance form of ' . $updated->user->name,
                $updated,
                ExitManagementLogType::UPDATE_CLEARANCE,
                new Activity()
            );

            $clearance->save();

            return response()->json([
                'message' => "Updated successfully!",
            ]);
        } catch (Exception $e) {
            return $e->getmessage();
        }
    }

    public function file_attachment_download(Request $req){
        $valid = Validator::make($req->all(),[
            'id' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'error' => $valid->errors()
            ], 400);
        }

        $clearance_compliance = ClearanceCompliance::where('id' , $req->id)->first();

        $this->logCustomMessage(
            'download_file_attachment',
            $clearance_compliance,
            Auth::user()->name . ' downloaded the file attachment',
            $clearance_compliance,
            ExitManagementLogType::DOWNLOAD,
            new Activity()
        );

        return Storage::download('employees/' .  $clearance_compliance->user_id . '/' . $clearance_compliance->attachment);
    }

    public function CCS_list()
    {
        return response()->json([
            'data' => CCS::all()
        ]);
    }

}
