<?php
namespace Suiterus\Adg\Controllers\EmployeeCases;

use DB;
use Exception;
use Validator;
use App\Models\User;
use Illuminate\Http\Request;
use App\Enums\EmployeeCaseStatus;
use App\Traits\Logs\HasCustomLogs;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Enums\Log\EmployeeCaseLogType;
use App\Http\Classes\NotificationMessage;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\EmployeeCases\EmployeeCase;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;

class EmployeeCaseController extends Controller
{
    use HasCustomLogs;
    
    //CREATE
    public function create(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'user_id' => 'required|integer',
            'complaint_title' => 'required',
            'date_of_complaint' => 'required|date',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();
        try {
            $status = 0;

            if($req->action_taken == null || $req->date_of_action == null){
                    $status = EmployeeCaseStatus::PENDING;          
             }else{
                    $status = EmployeeCaseStatus::FINISHED;
                    }

            $record = EmployeeCase::create([
                'user_id' => $req->user_id,
                'complaint_title' => $req->complaint_title,
                'date_of_complaint' => $req->date_of_complaint,
                'action_taken' => $req->action_taken,
                'date_of_action' => $req->date_of_action,
                'status' => $status,
                'created_by' => $req->user()->id,
                'updated_by' => $req->user()->id,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $this->logCustomMessage(
                'create_employee_case',
                $record,
                $req->user()->name . ' created a new employee case: ' . $req->complaint_title,
                $record,
                EmployeeCaseLogType::CREATE,
                new Activity()
            );

            NotificationMessage::notifySenderAndReceiver('Create Case', $req->user_id, Auth::id());

            DB::commit();

            return response()->json([
                'text' => 'New Employee case record have been created.',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //FETCH ALL
    public function init_list(Request $req)
    {
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        $d = EmployeeCase::whereHas('user.employeeMetaInfo')
            ->with([
                'user' => function ($q) {
                    $q->select('id', 'name')->without(['storage', 'roles', 'permissions']);
                },
                'user.employeeMetaInfo' => function ($q) {
                    $q->select('id', 'user_id', 'employee_id')->without('employeeType', 'corporation', 'division', 'department', 'branch');
                },
            ])
            ->select('id', 'user_id', 'complaint_title', 'date_of_complaint', 'action_taken', 'date_of_action', 'status')
            ->paginate($paginate);

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $d,
        ]);
    }

    //UPDATE
    public function edit(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|exists:adg_db.employee_cases,id',
            'user_id' => 'required|integer',
            'action_taken' => 'required',
            'date_of_action' => 'required|date',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        DB::beginTransaction();
        try {
            $record = EmployeeCase::findOrFail($req->id);

            $record->update([
                'user_id' => $req->user_id,
                'action_taken' => $req->action_taken,
                'date_of_action' => $req->date_of_action,
                'status' => EmployeeCaseStatus::FINISHED,
            ]);

            $updated = EmployeeCase::find($req->id);

            $updated->old = collect($record);
            $updated->attributes = collect($updated);
                    
            $this->logCustomMessage(
                'update_employee_case',
                $updated,
                Auth::user()->name . ' updated the employee case',
                $updated,
                EmployeeCaseLogType::UPDATE,
                new Activity()
            );

            $record->save();

            NotificationMessage::notifySenderAndReceiver('Update Case', $record->user_id, Auth::id());

            DB::commit();
            return response()->json([
                'text' => $record->user->name . "'s record has been updated.",
            ]);
        } catch (ME $e) {
            return response()->json(
                [
                    'errors' => ['The selected record could not be found.'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        } catch (Exception $e) {
            return response()->json(
                [
                    'errors' => ['There was a problem in editing the record.'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //FILTER RECORDS
    public function filter(Request $req)
    {
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        try {
            $users = EmployeeCase::whereHas('user', function ($query) use ($req) {
                $query
                    ->select('id', 'name')
                    ->where('name', 'LIKE', '%' . $req->employee_name . '%')
                    ->without(['storage', 'roles', 'permissions'])
                    ->whereHas('employeeMetaInfo')
                    ->with([
                        'employeeMetaInfo' => function ($query) {
                            $query->select('id', 'user_id', 'employee_id')->without('employeeType', 'corporation', 'division', 'department', 'branch');
                        },
                    ]);
            })
                ->with(['user'])
                ->when($req->date_of_complaint != '' && $req->date_of_complaint != null, function ($query) use ($req) {
                    return $query->where('date_of_complaint', $req->date_of_complaint);
                })
                ->when($req->date_of_action != '' && $req->date_of_action != null, function ($query) use ($req) {
                    return $query->where('date_of_action', $req->date_of_action);
                })
                ->when($req->status != '' && $req->status != null, function ($query) use ($req) {
                    return $query->where('status', $req->status);
                })
                ->select('id', 'user_id', 'complaint_title', 'date_of_complaint', 'action_taken', 'date_of_action', 'status');

            //Employee filter
            if (count($req->employees) > 0) {
                $users->whereIn('user_id', $req->employees);
            }

            return response()->json([
                'data' => $users->orderBy('created_at', 'asc')->paginate($paginate),
            ]);
        } catch (ME $me) {
            return response()->json(
                [
                    'errors' => ['The selected filters could not be found.'],
                    'message' => $me->getMessage(),
                ],
                500
            );
        } catch (Exception $e) {
            return response()->json(
                [
                    'errors' => ['There was a problem in fetching the records.'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //FETCH USERS WITHOUT RECORD
    public function fetch_users_without_record()
    {

        try {
            return User::role(['Employee'])->where('status', 1)->doesntHave('employeeCases')->get();
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    //FETCH USERS WITH RECORD
    public function fetch_users_with_record()
    {

        try {
            $users = User::whereHas('employeeCases', function ($query) {
                $query->whereIn('status', [EmployeeCaseStatus::FINISHED,EmployeeCaseStatus::PENDING]);
            })
                ->with(['employeeMetaInfo' => function ($query) {
                    $query->without(['corporation', 'branch', 'division', 'department']);
                }])
                ->without(['roles', 'permissions', 'storage', 'user_supervisor'])->get();
            return response()->json([
                'data'  =>  $users
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }
}
