<?php


namespace Suiterus\Adg\Controllers\Employee;
use Exception;
use Carbon\Carbon;
use App\Models\User;
use NumberFormatter;
use mikehaertl\pdftk\Pdf;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Models\SM\Remark;
use Suiterus\Adg\Models\SM\Sector;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Models\SM\SalaryGrade;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\Salary\PublicSalary;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\Salary\PrivateSalary;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\SM\SalaryStandardization;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;
use Suiterus\Adg\Models\EMI\EmployeeTypeHistory as ETH;

class EmployeeTypeHistoryController extends Controller
{
   
    use HasCustomLogs;
    
    public function create_records(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'user_id'       => 'required|integer'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();


        try {
            $activeSector = Sector::where('status', 'active')->first();
            $sector = $activeSector->name;

            $record = ETH::where('user_id', $request->user_id)->orderBy('id', 'desc')->first();


            $status = 3; // for 'pending'
         
            if(isset($record->date_of_effectivity) && $record->date_of_effectivity != null){
                $date1 = Carbon::createFromFormat('Y-m-d', strval($request->date_of_effectivity)); 
                $date2 = Carbon::createFromFormat('Y-m-d', strval($record->date_of_effectivity)); 

                if($date1->gt($date2)){
                    $status = 2; // for 'previous'
                    ETH::where('user_id', $request->user_id)->orderBy('id', 'desc')->first()
                    ->update([
                        'status'        =>  $status,
                        'updated_by'    =>  Auth::id()
                    ]);
                }
            }

            $currentDate = Carbon::parse(now())->format('Y-m-d');

            $hasSameEffectivityDate = ETH::where([
                ['user_id', $request->user_id], ['date_of_effectivity', $request->date_of_effectivity]
            ])->first();

            if ($hasSameEffectivityDate) {
                return response()->json([
                    'errors'   =>  ['You have already applied for this date of effectivity']
                ], 400);
            }

            $employeeType = ETH::create([
                'user_id'               => $request->user_id,
                'employee_type_id'      => $request->employee_type_id,
                'date_of_effectivity'   => $request->date_of_effectivity,
                'status'                => ($currentDate == $request->date_of_effectivity) ? 1 : 3,
                'created_by'            => $request->user()->id,
                'updated_by'            => $request->user()->id
            ]);

            $employeeInfo = EmployeeMetaInfo::where('user_id', $request->user_id)->first();

            $employeeInfo->update([
                'employee_type'       => ($employeeType->status == 1) ? $employeeType->employee_type_id : $record->employee_type_id,
                'updated_at'          => now(),
                'updated_by'          => $request->user()->id
            ]); 

            $totalAllowance = 0;
            
            $allowance = EmployeeAllowance::where('user_id', $request->user_id)->get();

            foreach($allowance as $allowances) {
                $totalAllowance += $allowances['amount'];
            }

            if($allowance === null){
                $allAllowances = 0;
            }
            else{
                $allAllowances = $totalAllowance;
            }

            if($sector == 'public'){
                $newSalary = SalaryGrade::whereHas('publicSalary', function($query) use($request){
                    $query->whereHas('userSalary', function($query) use($request){
                        $query->where([
                            ['user_id', $request->user_id]
                        ]);
                    });
                })->first();

                $salary = $newSalary->value;
                $salary = str_replace( ',', '', $salary);
                $annualSalary = $salary * 12;

            }
            else{
                $newSalary = PrivateSalary::whereHas('userSalary', function ($query) use($request){
                $query->where('user_id', $request->user_id);
            })->first();

                $salary = $newSalary->salary;
                $annualSalary = $salary * 12;
            }

            $remark = Remark::whereId(22)->first();
            $oldServiceRecord = ServiceRecord::where('user_id', $request->user_id)->whereNull('end_date')->first();


            if($oldServiceRecord->remark_id === $remark->id){
                $endDate = date('Y-m-d', (strtotime('-1 day', strtotime($request->date_of_effectivity))));
                $oldServiceRecord->update([
                    'end_date'  => $endDate
                ]);
            }else{
                $endDate = date('Y-m-d', strtotime(now()));
                $oldServiceRecord->update([
                    'end_date'  => $endDate
                ]);
            }

            $serviceRecord = ServiceRecord::create([
                'user_id'           =>  $request->user_id,
                'start_date'        =>  $request->date_of_effectivity, 
                'position_id'       =>  $employeeInfo->position_id,
                'salary'            =>  $annualSalary,
                'allowance'         =>  $allAllowances,
                'employee_type_id'  =>  $employeeInfo->employee_type,
                'division_id'       =>  $employeeInfo->division_id,
                'remark_id'         =>  $remark->id,  //* Change of Status
                'created_by'        =>  Auth::id(),
            ]);

            $this->logCustomMessage('update_employee_type', $employeeType, Auth::user()->name . ' updated the employee type of ' . $employeeInfo->user->name, $employeeType, 'Update employee type', new Activity());

            DB::commit();
            return response()->json([
                'text'  =>  "Record successfully created",
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [ 'There was a problem in creating a record.' ],
                'message'   => $e->getMessage(),
                'line'      => $e -> getLine()
            ], 500); 
        }
    }

    public function fetch_emp_type(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        $validate = Validator::make($request->all(), [
            'user_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'employee_type_histories,user_id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {
            $data = ETH::where('user_id', $request->user_id)->with('employeeType', function ($query){
                $query->select('id', 'title', 'abbreviation', 'status');
            })->select('id', 'user_id', 'employee_type_id', 'date_of_effectivity', 'status')->orderBy('id', 'desc')->paginate($paginate);

            return response()->json([
                'data'  => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the record'],
                'message'   => $e->getMessage()
            ], 500);
        }
        
    }

    public function updateEmpTypeHistory(Request $request) {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'employee_type_histories,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {
            $employeeTypeHistory = ETH::where('id', $request->id)->first();
            $currentDate = Carbon::parse(now())->format('Y-m-d');

            $employeeTypeHistory->update([
                'date_of_effectivity' => $request->date_of_effectivity,
                'status' => ($currentDate == $request->date_of_effectivity) ? 1 : 3,
                'updated_by' => Auth::id(),
                'updated_at' => now()
            ]);

            $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $employeeTypeHistory->user_id)->first();

            if ($employeeTypeHistory->status == 1) {
                $employeeMetaInfo->update([
                    'employee_type' => $employeeTypeHistory->employee_type_id,
                    'updated_at' => now(),
                    'updated_by' => Auth::id()
                ]);
            }

            $updated = ETH::where('id', $request->id)->first();

            $updated->old = collect($employeeTypeHistory);
            $updated->attributes = collect($updated);

            $this->logCustomMessage('update_employee_type', $updated, Auth::user()->name . ' updated the employee type of ' . $updated->user->name, $updated, 'Update employee type', new Activity());

            DB::commit();
            return response()->json([
                'text'  =>  "Record successfully updated",
            ]);

        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in updating the record']
            ], 500);
        }
    }

    public function generateAppointmentForm(Request $request) {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'employee_type_histories,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {
            $activeSector = Sector::where('status', 'active')->first();
            $sector = $activeSector->name;

            $employeeTypeHistory = ETH::where('id', $request->id)->first();
            
            $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $employeeTypeHistory->user_id)->first();

            // Salary
            if($sector == 'public'){
                $newSalary = SalaryGrade::whereHas('publicSalary', function($query) use($employeeTypeHistory){
                    $query->whereHas('userSalary', function($query) use($employeeTypeHistory){
                        $query->where([
                            ['user_id', $employeeTypeHistory->user_id]
                        ]);
                    });
                })->first();

                $salary = $newSalary->value;
            }
            else{
                $newSalary = PrivateSalary::whereHas('userSalary', function ($query) use($employeeTypeHistory){
                    $query->where('user_id', $employeeTypeHistory->user_id);
                })->first();

                $salary = $newSalary->salary;
            }

            $salary = str_replace(',', '', $salary);
            $numToWords = new NumberFormatter('en', NumberFormatter::SPELLOUT);

            $uuid = uniqid();
            $designation = ActualDesignation::where('user_id', $employeeTypeHistory->user_id)
                ->with('office','department', 'division','section','unit')
                ->first();

            $relationships = ['office', 'department', 'division', 'section', 'unit'];
            $actualDesignation = [];

            foreach ($relationships as $relationship) {
                if ($designation->$relationship) {
                    $actualDesignation[] = $designation->$relationship->name;
                }
            }

            $actualDesignation = implode(' / ', $actualDesignation);

            $data = [
                'current_date' => Carbon::now()->format('F j, Y'),
                'name' => strtoupper($employeeMetaInfo->user->name),
                'position'  => strtoupper($employeeMetaInfo->position->title),
                'item_code' => $employeeMetaInfo->itemCode->item_code ?? null,
                'employee_type' => strtoupper($employeeTypeHistory->employeeType->title),
                'salary_words' => str_replace('-', ' ', strtoupper($numToWords->format($salary))),
                'salary_number' => '( Php ' . number_format($salary, 2, '.', ',') . ' )',
                'department' => $actualDesignation,
                'SG' => $sector == 'public' ? $newSalary->salary_grade. '.' . $newSalary->step : '',
            ];

            $pdf = new Pdf(public_path('csc-form-template/Appointment_Form_acroform_v2.1.pdf'), config('pdftk.options'));

            if (!Storage::disk('employee_type_history')->exists($employeeTypeHistory->user_id)) {
                Storage::disk('employee_type_history')->makeDirectory($employeeTypeHistory->user_id);
            }

            $pdf->fillForm($data)->needAppearances()
                ->saveAs(storage_path('app/appointment_form/employee-type-history/' . $employeeTypeHistory->user_id . '/' .  $uuid . '.pdf'));

            $this->logCustomMessage('generate_appointment_form', $employeeTypeHistory, Auth::user()->name . ' generated the appointment form for employee ' . $employeeMetaInfo->user->name, $employeeTypeHistory, 'Generate appointment form', new Activity());

            return response()->download(Storage::disk('employee_type_history')
            ->path($employeeTypeHistory->user_id . '/' .  $uuid . '.pdf'))->deleteFileAfterSend(true);

        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in generating appointment form.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }
}
