<?php

namespace Suiterus\Adg\Controllers\Employee;


use Exception;
use App\Models\User;
use App\Enums\Status;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Unit;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Section;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Suiterus\Adg\Models\SM\Department;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Models\SM\Corporation;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\ActualDesignation;
use App\Enums\Log\ActualDesignationLogType;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;

class EmployeeController extends Controller
{

    use HasCustomLogs;

    private $orgStructureService;

    public function __construct(OrgStructureService $orgStructureService)
    {
        $this->orgStructureService = $orgStructureService;
    }

    public function fetchEmployeeInfo()
    {
        return Auth::user();
    }

    public function fetchEmployeePDS(Request $request)
    {
        return response()->json([
            'data' => User::whereId($request->user()->id)->with(['user_supervisor', 'salary' => function ($query) {
                $query->with(['publicSalary']);
            }, 'personalInformation' => function ($query) {
                $this->display($query);
            }, 'pdsChoices' => function ($query) {
                $this->display($query);
            }, 'otherInformation' => function ($query) {
                $this->display($query);
            }, 'permanentAddress' => function ($query) {
                $this->display($query);
            }, 'residentialAddress' => function ($query) {
                $this->display($query);
            }, 'spouse' => function ($query) {
                $this->display($query);
            }, 'children' => function ($query) {
                $this->display($query);
            }, 'father' => function ($query) {
                $this->display($query);
            }, 'maidenName' => function ($query) {
                $this->display($query);
            }, 'elementary' => function ($query) {
                $this->display($query);
            }, 'secondary' => function ($query) {
                $this->display($query);
            }, 'vocational' => function ($query) {
                $this->display($query);
            }, 'college' => function ($query) {
                $this->display($query);
            }, 'graduate' => function ($query) {
                $this->display($query);
            }, 'civilService' => function ($query) {
                $this->display($query);
            },  'workExperience' => function ($query) {
                $this->display($query);
            }, 'voluntaryWork' => function ($query) {
                $this->display($query);
            }, 'learningDevelopment' => function ($query) {
                $this->display($query);
            }, 'skillHobby' => function ($query) {
                $this->display($query);
            }, 'choice' => function ($query) {
                $this->display($query);
            }, 'reference' => function ($query) {
                $this->display($query);
            }, 'governmentIssuedId' => function ($query) {
                $this->display($query);
            }, 'fileAttachment' => function ($query) {
                $this->display($query);
            }])->first()
        ]);
    }

    public function display($query)
    {
        $query->with(['employeeExtraFieldColumn' => function ($query) {
            $query->with(['updatePDS' => function ($query) {
                $query->orderBy('id', 'desc');
            }]);
        }]);
    }

    public function fetch_employee_paginate(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return User::role(['Employee'])->where('name', 'LIKE', '%' . $req->employeeName . '%')->where('status', Status::ACTIVE)->whereHas('employeeMetaInfo', function ($query) use ($req) {
            $query->when($req->employee_id != null && isset($req->employee_id) && $req->employee_id != '', function ($query) use ($req) {
                if (count($req->employee_id) > 0) {
                    $query->whereIn('employee_id', $req->employee_id);
                } else {
                    $query->where('employee_id', $req->employee_id);
                }
            });
            $query->when($req->position != null && isset($req->position) && $req->position != '', function ($query) use ($req) {
                $query->where('position_id', $req->position);
            });
            $query->when($req->department != null && isset($req->department) && $req->department != '', function ($query) use ($req) {
                $query->where('department_id', $req->department);
            });
            $query->when($req->employeeType != null && isset($req->employeeType) && $req->employeeType != '', function ($query) use ($req) {
                $query->where('employee_type', $req->employeeType);
            });
        })->when($req->supervisor, function ($query) use ($req) {
            $query->whereHas('user_supervisor', function ($query) use ($req) {
                $query->when($req->supervisor != null && isset($req->supervisor) && $req->supervisor != '', function ($query) use ($req) {
                    $query->where('supervisor_id', $req->supervisor);
                });
            });
        })->with(['user_supervisor', 'personalInformation', 'pdsChoices', 'otherInformation', 'permanentAddress', 'residentialAddress', 'spouse', 'children', 'father', 'maidenName', 'elementary', 'secondary', 'vocational', 'college', 'graduate', 'civilService', 'workExperience', 'voluntaryWork', 'learningDevelopment', 'skillHobby', 'choice', 'reference', 'governmentIssuedId', 'fileAttachment'])->paginate($paginate);
    }


    public function fetch_inactive_employee()
    {
        return User::role(['Employee'])->where('status', Status::INACTIVE)
            ->without(['roles', 'storage', 'permissions', 'employeeMetaInfo'])->get();
    }

    public function file_attachment_download(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'path' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'error' => $valid->errors()
            ], 400);
        }

        return Storage::download($req->path);
    }

    public function fetchEmployeeServiceRecord(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return User::role(['Employee'])->where('name', 'LIKE', '%' . $req->employeeName . '%')->whereHas('employeeMetaInfo', function ($query) use ($req) {
            $query->when($req->employee_id != null && isset($req->employee_id) && $req->employee_id != '', function ($query) use ($req) {
                if (count($req->employee_id) > 0) {
                    $query->whereIn('employee_id', $req->employee_id);
                } else {
                    $query->where('employee_id', $req->employee_id);
                }
            });
            $query->when($req->position != null && isset($req->position) && $req->position != '', function ($query) use ($req) {
                $query->where('position_id', $req->position);
            });
            $query->when($req->department != null && isset($req->department) && $req->department != '', function ($query) use ($req) {
                $query->where('department_id', $req->department);
            });
            $query->when($req->divisionUnit != null && isset($req->divisionUnit) && $req->divisionUnit != '', function ($query) use ($req) {
                $query->where('employee_type', $req->divisionUnit);
            });
        })->when($req->supervisor, function ($query) use ($req) {
            $query->whereHas('user_supervisor', function ($query) use ($req) {
                $query->when($req->supervisor != null && isset($req->supervisor) && $req->supervisor != '', function ($query) use ($req) {
                    $query->where('supervisor_id', $req->supervisor);
                });
            });
        })->with(['personalInformation', 'serviceRecord'])->paginate($paginate);
    }

    public function updateCorporation()
    {

        DB::beginTransaction();
        try {
            $users = User::whereHas('employeeMetaInfo')->get();

            foreach ($users as $user) {
                if ($user->employeeMetaInfo->corp_id == null) {
                    $corporation = Corporation::where('name', 'National Kidney and Transplant Institute')->first();
                    EmployeeMetaInfo::where('user_id', $user->id)->update([
                        'corp_id'   =>  $corporation->id
                    ]);
                }
            }

            DB::commit();
            return response()->json([
                'text'  => 'Corporation ID successfully synced to employee_meta_info table'
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function fetch_employees(Request $req)
    {
        return User::role(['Employee'])->with([
            'personalInformation' => function ($query) {
                $query->with(['employeeExtraFieldColumn' => function ($query) {
                    $query->where('field_name', 'date_of_birth');
                }]);
            },
            'pdsChoices',
            'employeeMetaInfo' => function ($query) {
                $query->select('id', 'user_id' , 'employee_id', 'employee_type', 'date_hired')
                    ->without('corporation', 'branch', 'division', 'department', 'position', 'itemCode')
                    ->with('employeeType');
            }, 


        ])
            ->without(
                'currentRole',
                'roles',
                'permissions',
                'storage',
                'supervisor',
                'user_supervisor',
                'exitInterview',
                'userProfilePicture',
                'profileBasicInfo'
            )->get();
    }

    public function showEmployeePcr()
    {
        try {
            $user = User::without([
                'currentRole',
                'roles',
                'permissions',
                'storage',
                'employeeMetaInfo',
                'supervisor',
                'user_supervisor',
                'exitInterview',
                'userProfilePicture',
                'profileBasicInfo'
            ])->find(Auth::id());

            $model = $this->orgStructureService->getEmployeeCurrentOrg($user);

            $uniquePcrTypeIds = [];
            $uniquePcrTypes = [];

            $headEmployeePcrs = $this->orgStructureService->getHeadEmployeePcrs();

            $employeePermissions = $this->orgStructureService->getEmployeePcrPermissions();

            $employeePcrsBasedonOrg = $model ? (new OrgStructureService)->getPcrConfig($model) : [];

            $pcrs = collect([...$employeePcrsBasedonOrg, ...$headEmployeePcrs]);

            foreach ($pcrs as $pcr) {
                if (!in_array($pcr->pcr_form_type_id, $uniquePcrTypeIds)) {
                    $uniquePcrTypeIds[] = $pcr->pcr_form_type_id;

                    $pcr->permissions = $employeePermissions->filter(function ($permission) use ($pcr) {
                        return $permission->pcr_form_type_id == $pcr->pcr_form_type_id;
                    });

                    $uniquePcrTypes[] = $pcr;
                }
            }

            return $uniquePcrTypes;
        } catch (Exception $e) {
            return response()->json([
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function showEmployeeActualDesignation(Request $request)
    {
        try {
            $user = User::find($request->user_id);

            if (!$user->actualDesignation) {
                return null;
            };

            return User::find($request->user_id)->actualDesignation()->with(['office', 'department', 'division', 'section', 'unit'])->first();
        } catch (Exception $e) {
            return response()->json([
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function updateEmployeeActualDesignation(Request $request)
    {
        try {
            return DB::connection(env('ADG_DB_CONNECTION'))->transaction(function () use ($request) {
                $actualDesignation = ActualDesignation::where('user_id', $request->user_id)->first();

                if ($actualDesignation) {
                    $oldActualDesignation = clone $actualDesignation;
                    $actualDesignation->update([
                        'office_id' => null,
                        'department_id' => null,
                        'division_id' => null,
                        'section_id' => null,
                        'unit_id' => null,
                    ]);
                }

                $updateQuery = [];

                if ($request->unit_id) {
                    $unit = Unit::find($request->unit_id);
                    $updateQuery['unit_id'] = $request->unit_id;
                    $updateQuery['section_id'] = $unit->section ? $unit->section->id : null;
                    $updateQuery['division_id'] = $unit->section ? $unit->section->division->id : null;
                    $updateQuery['department_id'] = $unit->section ? $unit->section->division->department->id : null;
                    $updateQuery['office_id'] = $unit->section ? $unit->section->division->department->office->id : null;
                }

                if ($request->section_id) {
                    $section = Section::find($request->section_id);
                    $updateQuery['section_id'] = $request->section_id;
                    $updateQuery['division_id'] = $section->division->id;
                    $updateQuery['department_id'] = $section->division->department->id;
                    $updateQuery['office_id'] = $section->division->department->office->id;
                }

                if ($request->division_id) {
                    $division = Division::find($request->division_id);
                    $updateQuery['division_id'] = $request->division_id;
                    $updateQuery['department_id'] = $division->department ? $division->department->id : null;
                    $updateQuery['office_id'] = $division->department ? $division->department->office->id : null;
                }

                if ($request->department_id) {
                    $department = Department::find($request->department_id);
                    $updateQuery['department_id'] = $request->department_id;
                    $updateQuery['office_id'] = $department->office->id;
                }

                if ($request->office_id) {
                    $updateQuery['office_id'] = $request->office_id;
                }

                $updatedActualDesignation = ActualDesignation::updateOrCreate(
                    ['user_id' => $request->user_id],
                    $updateQuery
                );

                if ($actualDesignation) {
                    $actualDesignation->old = collect($oldActualDesignation);
                    $actualDesignation->attributes = collect($actualDesignation);

                    $this->logCustomMessage(
                        'update_actual_designation',
                        $actualDesignation,
                        Auth::user()->name . ' updated actual designation of ' . $actualDesignation->user->name,
                        $actualDesignation,
                        ActualDesignationLogType::UPDATE,
                        new Activity()
                    );
                } else {
                    $this->logCustomMessage(
                        'create_actual_designation',
                        $updatedActualDesignation,
                        Auth::user()->name . ' created actual designation of ' . $updatedActualDesignation->user->name,
                        $updatedActualDesignation,
                        ActualDesignationLogType::CREATE,
                        new Activity()
                    );
                }

                return $updatedActualDesignation;
            });
        } catch (Exception $e) {
            return response()->json([
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
