<?php

namespace Suiterus\Adg\Controllers\Employee;

use Exception;
use mikehaertl\pdftk\Pdf;
use App\Enums\Dms\DriveType;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Suiterus\Dms\Enums\File\AccessLevel;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Dms\Services\File\FileService;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\EMI\AppointmentForm;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Adg\Models\EMI\EmployeeTypeHistory;
use Suiterus\Dms\Models\Repositories\SectionAccess;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Suiterus\Adg\Services\EmployeeTypeHistory\DownloadAppointmentFormRequest;

class DownloadUploadAppointmentForm extends Controller
{

    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function uploadAppointmentForm(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'employee_type_histories,id',
            'upload_appointment_form' => 'required|max:50000|mimes:pdf',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'   => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $employeeTypeHistory = EmployeeTypeHistory::whereId($req->id)->first();
            $path = 'user/' . $employeeTypeHistory->user_id;
            $filePath = Storage::disk('employee_type_history')->put($path, $req->file('upload_appointment_form'));
            FileVault::disk('employee_type_history')->encrypt($filePath);

            $appointmentForm = AppointmentForm::create([
                'employee_type_history_id' =>   $req->id,
                'file_path'    =>   $filePath,
                'created_by'   =>   Auth::id(),
            ]);

            $employeeMetaInfo = EmployeeMetaInfo::where('user_id', $employeeTypeHistory->user_id)->first();

            $folderName = $employeeMetaInfo->employee_id . ' - ' . $employeeMetaInfo->user->name;
            $userFolder = Section::where("name", $folderName)->pluck('id')->first();

            $appointmentFormFolder = Section::where([
                ['name', 'Appointment Forms'], ['parent_id', $userFolder]
            ])->first();
                
            if (!$appointmentFormFolder) {
                $section = Section::create([
                    'name' =>  'Appointment Forms',
                    'parent_id' => $userFolder,
                    'type' => DriveType::PRIVATE,
                    "status" => 1,
                    "created_by" => Auth::id(),
                    "created_at" => now(),
                    "updated_at" => now()
                ]);

                $folderIds = Section::where('name', $section->name)->first();
                
                $folderAccessEmployee = [
                    "section_id" => $folderIds['id'],
                    'user_id' => $employeeMetaInfo->user_id,
                    'access_level' => 1,
                    "created_at" => now(),
                    "updated_at" => now()
                ];
                
                SectionAccess::insert($folderAccessEmployee);
            }

            $appointmentFormSection = Section::where([
                ['name', 'Appointment Forms'], ['parent_id', $userFolder]
            ])->first();

            $option = 'continue';

            $file = $this->fileService->uploadFile(
                $appointmentFormSection->id,
                $req->file('upload_appointment_form'),
                $appointmentFormSection->type,
                6, // Document Type = Forms
                1, // Privacy = Confidential
                $option
            );

            $this->fileService->giveFileAccess($file, [$employeeMetaInfo->user_id], [], AccessLevel::VIEWER);
        
            $this->logCustomMessage('upload_appointment_form', $appointmentForm, Auth::user()->name . ' uploaded an appointment form for user ' . $employeeTypeHistory->user->name, $appointmentForm, 'Upload appointment form', new Activity());

            DB::commit();
            return response()->json([
                'text' => 'Appointment form has been uploaded successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'msg'   =>  $e->getMessage(),
            ], 500);
        }
    }

    public function downloadAppointmentForm(DownloadAppointmentFormRequest $req)
    {
        try {

            $filePath = $req->path . '.enc';

            FileVault::disk('employee_type_history')->decryptCopy($filePath);
            $decryptedFilePath = Storage::disk('employee_type_history')->path($req->path);

            $pdf = new Pdf($decryptedFilePath, config('pdftk.options'));

            $result = $pdf->flatten()->saveAs($decryptedFilePath);

            if (!$result) {
                throw new FileException('There was a problem processing your appointment form');
            }

            return response()->download($decryptedFilePath)->deleteFileAfterSend(true);
        } catch (Exception $e) {
            Storage::disk('appointment_form')->delete($req->path);
            return response()->json([
                'errors'    => ['File not found'],
                'message'   => $e->getMessage(),
            ], 404);
        }
    }
    
    public function fetchEmployeeAppointmentForm()
    {
        return AppointmentForm::whereHas('employeeTypeHistory', function ($query) {
            $query->where('user_id', Auth::id());
        })->latest()->first();
    }

    public function appointmentFormHistory(Request $req)
    {
        return AppointmentForm::where('employee_type_history_id', $req->id)->get();
    }
}
