<?php

namespace Suiterus\Adg\Controllers\Biometrics;

use App\Enums\Log\BiometricsLogType;
use App\Exceptions\BiometricConnectException;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use ErrorException;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Timekeeping\BiometricDevice;
use Suiterus\Adg\Services\ZKTecoService;

/**
 * Synchronization of users in the system and the biometric devices
 */

class SyncController extends Controller
{
    use HasCustomLogs;
    private $db;
    private $zk;

    function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * Sync users from system to device based on the IP
     * This sets all of the details
     */ 
    public function syncUsersByDevice(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.biometric_devices,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $device = BiometricDevice::findOrFail($request->id);
            $zk = new ZKTecoService($device->host, $device->port);
            
            $users = User::whereHas('biometric_rfid_record')->get();

            $zk->checkDeviceConnection();
            foreach($users as $user) {
                $zk->reSetUser($user);
            }
            $zk->disconnect();

            $this->db->commit();
            return response()->json([
                'text'  => 'Users Synced'
            ]);

        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ]);
        }

    }

    /**
     * Sync users by from system to multiple devices based on IP
     */
    public function syncUsersMultipleDevices(Request $request) {

        $validate = Validator::make($request->all(), [
            'data.*.id'    => 'required|exists:adg_db.biometric_devices,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            foreach($request->data as $data) {
                $device = BiometricDevice::findOrFail($data['id']);
                $zk = new ZKTecoService($device->host, $device->port);
                
                $users = User::whereHas('biometric_rfid_record')->get();
    
                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                foreach($users as $user) {
                    $zk->reSetUser($user);
                }
                $zk->enable();
                $zk->disconnect();
            }

            $this->logCustomMessage(
                'sync_system_users_to_device',
                null,
                Auth::user()->name . ' Sync system users to device',
                null,
                BiometricsLogType::SYNC,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => 'Users Synced'
            ]);

        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ]);
        }
    }

    /**
     * Sync device IDs from device to system
     */
    public function syncUserId(Request $request) {

        $validate = Validator::make($request->all(), [
            'data.*.id'    => 'required|exists:adg_db.biometric_devices,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $device = BiometricDevice::find($request->id);
            $this->zk = new ZKTecoService($device->host, $device->port);

            $this->zk->checkDeviceConnection();
            $this->zk->syncUserFromDevice();
            $this->zk->disconnect();

            $this->db->commit();
            return response()->json([
                'text'  => 'Device Users Synced'    
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Sync multiple device IDs from device to system
     */
    public function syncMultipleUserId(Request $request) {

        $validate = Validator::make($request->all(), [
            'data.*.id'    => 'required|exists:adg_db.biometric_devices,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            foreach($request->data as $data) {

                $device = BiometricDevice::find($data['id']);
                $zk = new ZKTecoService($device->host, $device->port);

                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                $zk->syncUserFromDevice();
                $zk->enable();
                $zk->disconnect();
            }

            $this->logCustomMessage(
                'sync_device_data_to_system',
                null,
                Auth::user()->name . ' Sync device data to system',
                null,
                BiometricsLogType::SYNC,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => 'Device Users Synced'    
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

}