<?php

namespace Suiterus\Adg\Controllers\Biometrics;

use App\Exceptions\BiometricConnectException;
use App\Http\Controllers\Controller;
use ErrorException;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Suiterus\Adg\Models\Timekeeping\BiometricDevice;
use Suiterus\Adg\Services\ZKTecoService;
use Svg\Tag\Rect;

/**
 * Used for the connection of the system with the biometric devices
 */

class DeviceConnectionController extends Controller
{

    private $db;
    private $zk;

    function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * Test device connection
     */
    public function testConnection(Request $request) {

        $validate = Validator::make($request->all(), [
            'ip'        => 'required|ip',
            'port'      => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $this->zk = new ZKTecoService($request->ip, $request->port);
            $connection = $this->zk->checkDeviceConnection();

            return response()->json([
                'success'  => $connection
            ]);

        } catch(BiometricConnectException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Test device voice connection
     */
    public function testVoice(Request $request) {

        $validate = Validator::make($request->all(), [
            'ip'        => 'required|ip',
            'port'      => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $this->zk = new ZKTecoService($request->ip, $request->port);
            $connection = $this->zk->testVoice();

            return response()->json([
                'success'  => $connection
            ]);

        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage(),
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'exception' => get_class($e)
            ], 500);
        }

    }

    /**
     * Disconnect device
     */
    public function disconnect(Request $request) {

        $validate = Validator::make($request->all(), [
            'ip'        => 'required|ip',
            'port'      => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $this->zk = new ZKTecoService($request->ip, $request->port);
            if($this->zk->checkDeviceConnection()) {
                $this->zk->disconnect();
            }

            return response()->json([
                'success'  => true 
            ]);

        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Restart Device
     */
    public function restart(Request $request) {

        $validate = Validator::make($request->all(), [
            'ip'        => 'required|ip',
            'port'      => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $this->zk = new ZKTecoService($request->ip, $request->port);
            $this->zk->restart();

            return response()->json([
                'success'  => true 
            ]);

        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Sleep Device
     */
    public function sleep(Request $request) {

        $validate = Validator::make($request->all(), [
            'ip'        => 'required|ip',
            'port'      => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $this->zk = new ZKTecoService($request->ip, $request->port);
            $this->zk->sleep();

            return response()->json([
                'success'  => true 
            ]);

        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Shutdown Device
     */
    public function shutdown(Request $request) {

        $validate = Validator::make($request->all(), [
            'ip'        => 'required|ip',
            'port'      => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $this->zk = new ZKTecoService($request->ip, $request->port);
            $this->zk->shutdown();

            return response()->json([
                'success'  => true 
            ]);

        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
}
