<?php

namespace Suiterus\Adg\Controllers\Biometrics;

use App\Enums\Biometrics\EnabledStatus;
use App\Enums\Log\BiometricsLogType;
use App\Exceptions\BiometricConnectException;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use ErrorException;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use InvalidArgumentException;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Timekeeping\BiometricDevice;
use Suiterus\Adg\Models\Timekeeping\BiometricRfidUser;
use Suiterus\Adg\Services\ZKTecoService;

/**
 * For biometrics device data synchronization. 
 * This controller contains functions to get fingerprints and card numbers from the device.
 * Saved fingerprints, card numbers, and unique ID can be passed onto devices as well
 */

class BiometricSyncController extends Controller
{
    use HasCustomLogs;
    private $db;
    private $zk;

    function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * Fetch fingerprints from single device - based on IP
     */
    public function fetchFingerprints(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.biometric_devices,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $device = BiometricDevice::find($request->id);
            $this->zk = new ZKTecoService($device->host, $device->port);
            $this->zk->checkDeviceConnection();

            $users = User::whereHas('employeeMetaInfo', function($query) use ($device) {
                $query->where('department_id', $device->department_id)
                    ->where('division_id', $device->division_id);
            });

            foreach($users as $user) {
                $this->zk->getUserFingerprints($user);
            }

            $this->db->commit();
            return response()->json([
                'text'  => 'Fingerprints saved.'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Edit Device details - User ID and Name
     */
    public function editDeviceUser(Request $request) {

        $validate = Validator::make($request->all(), [
            'user_id'       => 'required|exists:mysql.users,id',
            'unique_id'     =>  'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $biometric_user = BiometricRfidUser::where('user_id', $request->user_id)->first();
            $old_record = $biometric_user;
            $biometric_user->update([
                'unique_id' => $request->unique_id,
            ]);
            
            if(isset($request->card_number) && $request->card_number !== null && $request->card_number !== "") {
                $biometric_user->card->card_number = $request->card_number;
                $biometric_user->card->save();
            }

            $biometric_user->save();
           
            $devices = BiometricDevice::where('status', EnabledStatus::ENABLED)->get();
            if($devices === null) {
                throw new BiometricConnectException('The device for the user does not exist.');
            }
            foreach($devices as $device){
                $zk = new ZKTecoService($device->host, $device->port);
                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                $zk->reSetUser($biometric_user->user);
                $zk->enable();
                $zk->disconnect();
            }

            $this->logCustomMessage(
                'update_rfid_user',
                $old_record,
                Auth::user()->name . ' Update RFID user',
                $biometric_user,
                BiometricsLogType::UPDATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.edit')  
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    } 


}