<?php

namespace Suiterus\Adg\Controllers\Biometrics;

use App\Enums\Log\BiometricsSetupLogType;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Timekeeping\BiometricDevice;
use Suiterus\Adg\Services\ZKTecoService;
use Svg\Tag\Rect;

/**
 * Controller used for the CRUD functionalities of biometric devices connected
 * to the system and in relation to the departments and divisions
 */

class BiometricDeviceController extends Controller
{
    use HasCustomLogs;
    private $db;

    function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * List biometric devices as paginated table
     */
    public function list(Request $request) {

        try {

            $paginate = isset($request->paginate) && $request->paginate !== null ? $request->paginate : env('DEFAULT_PAGECOUNT');

            $data = BiometricDevice::when(isset($request->search) && $request->search != '', function($query) use ($request) {
                    $keyword = '%' . $request->search . '%';
                    $query->where('name', 'LIKE', $keyword)
                        ->orWhere('description', 'LIKE', $keyword)
                        ->orWhere('host', 'LIKE', $keyword)
                        ->orWhere('port', 'LIKE', $keyword);
                })->paginate($paginate);

            return response()->json([
                'data'  => $data 
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Fetch all as raw data
     */
    public function index() {
        
        try {

            return response()->json([
                'data'  => BiometricDevice::all()
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Get single record
     */
    public function fetch(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.biometric_devices,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            return response()->json([
                'data'  => BiometricDevice::findOrFail($request->id)
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
    
    /**
     * Create new biometric device record
     */
    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'name'      => 'required',
            'host'      => [
                'required',
                'ip',
                Rule::unique('adg_db.biometric_devices')->where(function($query) use ($request) {
                    return $query->where('host', $request->host)
                        ->where('port', $request->port);
                })->withoutTrashed()  
            ], 
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $data = BiometricDevice::create([
                'name'          => $request->name,
                'description'   => $request->description,
                'host'          => $request->host,
                'port'          => $request->port,
                'status'        => $request->status,
                'created_by'    => $request->user()->id,
            ]); 

            $this->logCustomMessage(
                'create_biometric_device',
                $data,
                Auth::user()->name . ' Create biometric Device',
                $data,
                BiometricsSetupLogType::CREATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    /**
     * Update Biometric Device Record
     */
    public function update(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.biometric_devices,id',
            'name'      => 'required',
            'host'      => [
                'required',
                'ip',
                Rule::unique('adg_db.biometric_devices')->where(function($query) use ($request) {
                    return $query->where('host', $request->host)
                        ->where('port', $request->port)
                        ->where('id', '!=', $request->id);
                })  
            ],
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $record = BiometricDevice::findOrFail($request->id);
            $old_record = $record;
            $record->update([
                'name'          => $request->name,
                'description'   => $request->description,
                'host'          => $request->host,
                'port'          => $request->port,
                'status'        => $request->status,
                'updated_by'    => $request->user()->id,
            ]);

            $record->save();

            
            $this->logCustomMessage(
                'update_biometric_device',
                $old_record,
                Auth::user()->name . ' Update biometric Device',
                $record,
                BiometricsSetupLogType::CREATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.update')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
    
    /**
     * Delete Record
     */
    public function delete(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.biometric_devices,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $record = BiometricDevice::findOrFail($request->id);
            $record->delete();


            $this->logCustomMessage(
                'delete_biometric_device',
                $record,
                Auth::user()->name . ' Delete biometric Device',
                $record,
                BiometricsSetupLogType::DELETE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.delete')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
}