<?php

namespace Suiterus\Adg\Controllers\Approvals;

use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\Approvals\TrainingApproval;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use Exception;


use function PHPUnit\Framework\isEmpty;

class TrainingApprovalController extends Controller
{
    //FETCH ALL
    public function init_list()
    {
        $d = TrainingApproval::with(['user' => function($q){

            $q->select('id','name')->without(['storage','roles', 'permissions']);

        },'user.employeeMetaInfo' =>function($q){ 

            $q->select('id','user_id', 'employee_id')->without('employeeType','corporation','division','department','branch','position'); 
        
        }])->paginate(10);
       

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $d,
        ]);
    }
    
    //FILTER EMPLOYEE
    public function filter_employee(Request $req){
        try{
            $records = User::where('name', 'LIKE', '%'.$req->name.'%')->whereHas('training_approvals')->without(['storage','roles','permissions'])
            ->with(['training_approvals', 'employeeMetaInfo' => function($q){
                $q->select('id','user_id','employee_id')->without('employeeType','corporation','division','department','branch','position');
            }])->select('id','name');

            return response()->json([
                'data' => $records->orderBy('created_at', 'asc')->paginate(10)
            ]);
        
        } 
        catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }


    //FILTER RECORDS
    public function filter_request(Request $req){
        try{
            $records = TrainingApproval::with(['user' => function($q){ 
            
                $q->select('id','name')->without(['storage','roles','permissions']); 

            }, 'user.employeeMetaInfo' =>function($q){ 

                $q->select('id','user_id', 'employee_id')->without('employeeType','corporation','division','department','branch','position'); 
            
            } ])->when(count($req->filter_dates)>0, function($q) use ($req){
                //if filter_dates has FROM and TO
                $q->when(count($req->filter_dates)>1, function($q) use ($req){
                    $q->whereDate('date', '>=', $req->filter_dates[0])
                    ->whereDate('date', '<=', $req->filter_dates[1]);
                });
                //if filter_dates has only one date
                $q->when(count($req->filter_dates)==1, function($q) use ($req){
                    $q->whereDate('date', $req->filter_dates[0]);
                });
            })->when($req->title!=null, function($q) use ($req){
                $q->where('title', "LIKE", '%'.$req->title.'%');
            })->when($req->company!=null, function($q) use ($req){
                $q->where('company', "LIKE", '%'.$req->company.'%');
            })->when($req->type!=null, function($q) use ($req){
                $q->where('type', $req->type);
            })->when($req->status!=null, function($q) use ($req){
                $q->where('status', $req->status);
            })->whereHas("user", function ($q) use($req){
                $q->where('name', "LIKE", '%'.$req->employee_name.'%');
            });
            
            
            //Employees IDs filter
            if(count($req->employees) > 0) {
                $records = $records->whereIn('employee_id', $req->employees);
            }

            return response()->json([
                'data' => $records->orderBy('created_at', 'asc')->paginate(10)
            ]);
        
        } 
        catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }


    // DELETE
    public function delete(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.training_approvals,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }

        DB::beginTransaction();
        try  {

            $record = TrainingApproval::where('status', 2)->findOrFail($request->id);
            $record->delete();

            DB::commit();

            return response()->json([
                'text'  => $record->user->name . "'s request has been deleted."
            ]); 

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only declined requests can be deleted.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in deleting the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    
    // APPROVE
    public function approve(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.training_approvals,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }

        DB::beginTransaction();
        try {
            $record = TrainingApproval::where('status', 3)->findOrFail($request->id);

            $record->update([
                'status'       => 1,
                'updated_at'   => now(),
            ]); 

            $record->save();
            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s request has been approved."
            ]);

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be approved.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    // APPROVE ALL
    public function approve_all(Request $req) {
        DB::beginTransaction();
        try {
            $record = TrainingApproval::where('status', 3)->get();

            if(!$record->isEmpty()){
                TrainingApproval::where('status', 3)->update([
                    'status'       => 1,
                    'updated_at'   => now(),
                ]); 
    
                DB::commit();

                return response()->json([
                    'text'  => "All pending request has been approved."
                ]);
            }else{
                throw new ME();
            }            
            

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be approved.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    // DECLINE
    public function decline(Request $request) {
        
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.training_approvals,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }

        DB::beginTransaction();
        try {
            $record = TrainingApproval::where('status', 3)->findOrFail($request->id);

            $record->update([
                'status'       => 2,
                'updated_at'   => now(),
            ]); 

            $record->save();
            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s request has been declined."
            ]);

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be declined.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in declining the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
}