<?php

namespace Suiterus\Adg\Controllers\Approvals;

use App\Enums\ApprovedStatus;
use App\Enums\Log\OvertimeReportLogType;
use App\Http\Classes\NotificationMessage;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Suiterus\Adg\Models\Approvals\OvertimeAccomplishmentReport;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Controllers\Approvals\Services\COCPointService;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Approvals\CTO\COCPoints;

class OvertimeReportController extends Controller
{
    use HasCustomLogs;
    //initialize table
    public function init_list(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        $data = OvertimeAccomplishmentReport::with(['overtime.user' => function ($query) {
            $query->select('id', 'name')->with(['user_supervisor'])->without(['storage', 'roles', 'permissions']);
        }, 'overtime.user.employeeMetaInfo' => function ($query) {
            $query->select('id', 'user_id', 'employee_id', 'position_id', 'department_id')->without('employeeType', 'corporation', 'division', 'branch');
        }])->orderBy('updated_at', 'desc');

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $data->paginate($paginate),
        ]);
    }

    //fetch records
    public function fetch_report(Request $request)
    {
        $data = OvertimeAccomplishmentReport::where('id', $request->id)->with(['overtime.user' => function ($query) {
            $query->select('id', 'name')->without(['storage', 'roles', 'permissions', 'employeeMetaInfo', 'supervisor']);
        }])->first();

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $data
        ]);
    }

    public function filter_request(Request $request)
    {
        try {
            $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

            $records = OvertimeAccomplishmentReport::with(['overtime.user' => function ($query) {
                $query->select('id', 'name')->with(['user_supervisor'])->without(['storage', 'roles', 'permissions']);
            }, 'overtime.user.employeeMetaInfo' => function ($query) {
                $query->select('id', 'user_id', 'employee_id', 'position_id', 'department_id')->without('employeeType', 'corporation', 'division', 'branch');
            }])
                ->when(count($request->filter_dates) > 0, function ($query) use ($request) {
                    $query->whereHas('overtime', function ($query) use ($request) {
                        //if filter_dates has FROM and TO
                        $query->when(count($request->filter_dates) == 2, function ($query) use ($request) {
                            $query->whereDate('start_date', '>=', $request->filter_dates[0])
                                ->whereDate('end_date', '<=', $request->filter_dates[1]);
                        });
                        //if filter_dates has only one date
                        $query->when(count($request->filter_dates) == 1, function ($query) use ($request) {
                            $query->whereDate('start_date', $request->filter_dates[0])
                                ->orWhere('end_date', $request->filter_dates[0]);
                        });
                    });
                })
                ->when($request->department != null, function ($query) use ($request) {
                    $query->whereHas('overtime.user.employeeMetaInfo', function ($query) use ($request) {
                        $query->where('department_id',  $request->department);
                    });
                })
                ->when($request->position != null, function ($query) use ($request) {
                    $query->whereHas('overtime.user.employeeMetaInfo', function ($query) use ($request) {
                        $query->where('position_id',  $request->position);
                    });
                })
                ->when($request->supervisor != null, function ($query) use ($request) {
                    $query->whereHas('overtime.user.user_supervisor.supervisor', function ($query) use ($request) {
                        $query->where('id', $request->supervisor);
                    });
                })
                ->when($request->status != null, function ($query) use ($request) {
                    $query->where('status', $request->status);
                })
                ->whereHas('overtime.user', function ($query) use ($request) {
                    $query->where('name', 'LIKE', '%' . $request->employee_name . '%');
                })
                ->when(count($request->employees) > 0, function ($query) use ($request) {
                    $query->whereHas('overtime.user', function ($query) use ($request) {
                        $query->whereIn('user_id', $request->employees);
                    });
                });

            return response()->json([
                'data' => $records->orderBy('updated_at', 'desc')->paginate($paginate)
            ]);
        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage(),
                'line'      => $me->getLine()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function approve(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.overtime_accomplishment_reports,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $record = OvertimeAccomplishmentReport::where('status', ApprovedStatus::PENDING)->findOrFail($request->id);

            $record->update([
                'status'       => ApprovedStatus::APPROVED,
                'updated_by'   => Auth::id() ?? null
            ]);
            $record->save();

            $report = OvertimeAccomplishmentReport::whereId($request->id)->with('overtime')->first();
            $mh_rendered = floatval($report->mh_rendered);

            $coc = COCPoints::where('user_id', $report->overtime->user_id)->first();
            $coc_points = $coc->points + $mh_rendered;

            COCPoints::where('user_id', $coc->user_id)->update([
                'points'    =>  $coc_points
            ]);

            $increase = new COCPointService;
            $increase->recordIncrease($coc->user_id, $coc_points, $mh_rendered);
            
            $this->logCustomMessage(
                'approve_overtime_application',
                $record,
                Auth::user()->name . ' Approve overtime report',
                $record,
                OvertimeReportLogType::APPROVE,
                new Activity()
            );

            NotificationMessage::notifySenderAndReceiver('Approved Overtime Accomplishment', $report->overtime->user_id, Auth::id());

            DB::commit();
            return response()->json([
                'text'  => "Overtime Accomplishment Report has been approved."
            ]);
        } catch (ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be approved.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    // DECLINE
    public function disapprove(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.overtime_accomplishment_reports,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $record = OvertimeAccomplishmentReport::where('status', ApprovedStatus::PENDING)->with('overtime')->findOrFail($request->id);

            $record->update([
                'status'       => ApprovedStatus::DECLINED,
                'remarks'      => $request->remarks,
                'updated_at'   => now(),
            ]);

            $record->save();

            $this->logCustomMessage(
                'disapprove_overtime_application',
                $record,
                Auth::user()->name . ' Disapprove overtime report',
                $record,
                OvertimeReportLogType::DISAPPROVE,
                new Activity()
            );

            NotificationMessage::notifySenderAndReceiver('Disapproved Overtime Accomplishment', $record->overtime->user_id, Auth::id());

            DB::commit();
            return response()->json([
                'text'  => "Overtime Accomplishment Report has been declined."
            ]);
        } catch (ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be declined.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in declining the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function file_download(Request $request)
    {
        try {
            $valid = Validator::make($request->all(), [
                'path'  =>  'required'
            ]);

            if ($valid->fails()) {
                return response()->json([
                    'error' => $valid->errors()
                ], 400);
            }

            $this->logCustomMessage(
                'download_overtime_application',
                null,
                Auth::user()->name . ' Download overtime report '. basename($request->path),
                null,
                OvertimeReportLogType::DOWNLOAD,
                new Activity()
            );

            return Storage::download($request->path);
        } catch (Exception $e) {

            return response()->json([
                'errors'    => ['File not found'],
                'message'   => $e->getMessage(),
            ], 404);
        }
    }
}
