<?php

namespace Suiterus\Adg\Controllers\Approvals;

use App\Enums\Log\OvertimeLogType;
use App\Http\Classes\NotificationMessage;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\Approvals\UserOvertime;
use Suiterus\Adg\Models\SM\Department;
use Suiterus\Hrjp\Models\Position;
use Suiterus\Adg\Models\UserSupervisors\Supervisors;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Controllers\Approvals\Services\COCPointService;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Approvals\CTO\COCPoints;
use Suiterus\Adg\Models\Timekeeping\Attendance;

class OvertimeController extends Controller
{
    use HasCustomLogs;
    //INITIALIZE TABLE
    public function init_list(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');

        $data = UserOvertime::with(['user' => function($query){

            $query->select('id','name')->with(['user_supervisor'])->without(['storage','roles', 'permissions']);

        },'user.employeeMetaInfo' =>function($query){ 

            $query->select('id','user_id', 'employee_id', 'position_id', 'department_id')->without('employeeType','corporation','division','branch'); 
        
        }])->orderBy('updated_at', 'desc');
        
        return response()->json([
            'text' => 'fetch successful.',
            'data' => $data->paginate($paginate),
        ]);
    }


    public function init_list_no_paginate()
    {
        return UserOvertime::with(['user' => function($query){
            $query->select('id','name')->with(['user_supervisor'])->without(['storage','roles', 'permissions']);
        },'user.employeeMetaInfo' =>function($query){ 
            $query->select('id','user_id', 'employee_id', 'position_id', 'department_id')->without('employeeType','corporation','division','branch'); 
        }])->orderBy('updated_at', 'desc')->get();
    
    }
    


    //FETCH RECORDS
    public function fetch_user_overtime(Request $request)
    {
        $data = UserOvertime::where('id', $request->id)->with(['user' => function($query){
            $query->select('id','name')->without(['storage','roles', 'permissions', 'employeeMetaInfo','supervisor']);
        }])->get();

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $data
        ]);
    }

    //FILTER RECORDS
    public function filter_request(Request $req){
        try{
            $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');

            $records = UserOvertime::with(['user' => function($query){ 
                
                $query->select('id','name')->with(['user_supervisor'])->without(['storage','roles','permissions']); 
            
            }, 'user.employeeMetaInfo' =>function($query){ 
                
                $query->select('id','user_id', 'employee_id', 'position_id', 'department_id')->without('employeeType','corporation','division','branch'); 
            
            }])
            ->when(count($req->filter_dates)>0, function($query) use ($req){
                //if filter_dates has FROM and TO
                $query->when(count($req->filter_dates)==2, function($query) use ($req){
                    $query->whereDate('start_date', '>=', $req->filter_dates[0])
                    ->whereDate('end_date', '<=', $req->filter_dates[1]);
                });
                //if filter_dates has only one date
                $query->when(count($req->filter_dates)==1, function($query) use ($req){
                    $query->whereDate('start_date', $req->filter_dates[0])
                    ->orWhere('end_date', $req->filter_dates[0]);
                });
            })
            ->when($req->department != null, function($query) use ($req){
                $query->whereHas('user.employeeMetaInfo', function ($query) use($req){
                    $query->where('department_id',  $req->department);
                });
            })
            ->when($req->position != null, function($query) use ($req){
                $query->whereHas('user.employeeMetaInfo', function ($query) use($req){
                    $query->where('position_id',  $req->position);
                });
            })
            ->when($req->supervisor != null, function($query) use ($req){
                $query->whereHas('user.user_supervisor.supervisor', function ($query) use($req){
                    $query->where('id', $req->supervisor);
                });

            })
            ->when($req->status != null, function($query) use ($req){
                $query->where('status', $req->status);

            })
            ->whereHas("user", function($query) use($req){
                $query->where('name', 'LIKE', '%'.$req->employee_name.'%');
            });
            
            //Employees IDs filter
            if(count($req->employees) > 0) {
                $records = $records->whereIn('user_id', $req->employees);
            }

            return response()->json([
                'data' => $records->orderBy('created_at', 'asc')->paginate($paginate)
            ]);
        
        } 
        catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }


    // DELETE
    public function delete(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.user_overtimes,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }

        DB::beginTransaction();
        try  {

            $record = UserOvertime::where('status', 3)->findOrFail($request->id);
            $record->delete();

            $this->logCustomMessage(
                'delete_overtime_application',
                $record,
                Auth::user()->name . ' Delete overtime application',
                $record,
                OvertimeLogType::DELETE,
                new Activity()
            );

            DB::commit();

            return response()->json([
                'text'  => $record->user->name . "'s request has been deleted."
            ]); 

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only declined requests can be deleted.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in deleting the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }


    // APPROVE
    public function approve(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.user_overtimes,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }

        DB::beginTransaction();
        try {
            $record = UserOvertime::where('status', 1)->findOrFail($request->id);

            if (!$this->checkIfOvertimeApplicable($record->start_date, $record->end_date, $record->user_id)) {
                return response()->json([
                    'text'  => "User has more than 15 minutes late on selected date/s"
                ], 400);
            }

            $record->update([
                'status'       => 2,
                'updated_by'   => Auth::id()
            ]); 

            $record->save();

            $this->logCustomMessage(
                'approve_overtime_application',
                $record,
                Auth::user()->name . ' Approve overtime application',
                $record,
                OvertimeLogType::APPROVE,
                new Activity()
            );

            NotificationMessage::notifySenderAndReceiver('Approved Overtime Application', $record->user_id, Auth::id());

            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s request has been approved."
            ]);

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be approved.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    public function checkIfOvertimeApplicable($startDate, $endDate, $userId): bool
    {
        $startDate = Carbon::parse($startDate);
        $endDate = Carbon::parse($endDate);

        while ($startDate <= $endDate)
        {
            $attendance = Attendance::where('user_id', $userId)->whereDate('time_in', $startDate)->first();
            if ($attendance && $attendance->late_hours !== null) {
                $lateTime = Carbon::createFromFormat('H:i', $attendance->late_hours);
                $maxLate = Carbon::createFromFormat('H:i', "00:15");

                if ($lateTime->gte($maxLate))  {
                    return false;
                }
            }
            $startDate->addDay();
        }
        return true;
    }
    
    // DECLINE
    public function decline(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.user_overtimes,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }
        
        DB::beginTransaction();
        try {
            $record = UserOvertime::where('status', 1)->findOrFail($request->id);

            $record->update([
                'status'       => 3,
                'updated_at'   => now(),
            ]); 

            $record->save();
            
            $this->logCustomMessage(
                'disapprove_overtime_application',
                $record,
                Auth::user()->name . ' Dissaprove overtime application',
                $record,
                OvertimeLogType::DISAPPROVE,
                new Activity()
            );

            NotificationMessage::notifySenderAndReceiver('Disapproved Overtime Application', $record->user_id, Auth::id());

            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s request has been declined."
            ]);

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be declined.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in declining the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }


    // APPROVE ALL
    public function approve_all() {
        DB::beginTransaction();
        try {
            $record = UserOvertime::where('status', 1)->get();

            if(!$record->isEmpty()){
                UserOvertime::where('status', 1)->update([
                    'status'       => 2,
                    'updated_at'   => now(),
                ]); 

                DB::commit();
                return response()->json([
                    'text'  => "All pending request has been approved."
                ]);
            }else{
                throw new ME();
            }            

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be approved.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

}