<?php

namespace Suiterus\Adg\Controllers\Approvals\Employee;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use Carbon\CarbonInterval;
use App\Enums\OvertimeType;
use Illuminate\Http\Request;
use App\Enums\OvertimeStatus;
use App\Enums\OvertimeMaximumDays;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth; 
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Models\Approvals\UserOvertime;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;
use Suiterus\Adg\Models\Approvals\OvertimeAccomplishmentReport;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadRequiredHours;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeRequiredHours;

class UserOvertimeController extends Controller
{
    public function create_user_overtime(Request $req) {
        $valid = Validator::make($req->all(),[

            'start_date' => 'required|date',
            'end_date' => 'required|date',
            'purpose'   =>  'required'

        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        DB::beginTransaction();
        try {

            if (!$this->isOverTimeApplicable($req->start_date, $req->end_date, $req->est_mh, $req->est_qty, $req->overtime_type)) {
                return response()->json([
                    'errors' => ['Overtime input does not match attendance or is overdue for filing'],
                ], 400);
            }

            $mh = is_numeric($req->est_mh) ? CarbonInterval::createFromFormat('H', $req->est_mh)->forHumans() : $req->est_mh;

            $orgStructureService = new OrgStructureService();

            $overtimeOrganization = $this->getOvertimeOrganization();

            if (!$overtimeOrganization) {
                return response()->json([
                    'errors' => ['Employee organization not found'],
                ], 400);
            }
    
            $overtime = $orgStructureService->getMinimumOvertimeConfig($overtimeOrganization);
    
            if ($overtime->first()) {
                $minimumOvertime = Carbon::parse($overtime->first()->minimum_overtime)->format('H:i');
                if (strpos($req->est_mh, 'hour') !== false) {
                    $estimatedMH = CarbonInterval::hours((int) $req->est_mh);
                } else {
                    $estimatedMH = CarbonInterval::minutes((int) $req->est_mh);
                }
    
                $estimatedMHFormatted = $estimatedMH->format('%H:%I');
    
                if ($estimatedMHFormatted < $minimumOvertime) {
                    return response()->json([
                        'errors' => ['The minimum overtime hours must be at least ' . CarbonInterval::createFromFormat('H:i', $minimumOvertime)->forHumans()],
                    ], 400);
                }
            }
    
            UserOvertime::create([

                'user_id' => Auth::id(),
                'start_date' => $req->start_date,
                'start_time' => $req->start_time,
                'end_date' => $req->end_date,
                'end_time' => $req->end_time,
                'purpose'   => $req->purpose,
                'est_mh_needed' => $mh,
                'est_quantity'  =>  $req->est_qty,
                'overtime_justification' => $req->overtime_justification,
                'status' => $req->status ? $req->status : OvertimeStatus::FOR_REVIEW,
                'created_by' => Auth::id(),
                'overtime_type' =>$req->overtime_type

            ]);

            DB::commit();
            return response()->json([
                'text'  => 'Request Overtime has been created successfully.',
                'data'  => $req->all()
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : OT-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    private function isOverTimeApplicable($start_date, $end_date, $overtime, $quantity = 0, $overtime_type)
    {
        if ($overtime_type == OvertimeType::TIMETABLE) {
            if (Carbon::parse($start_date)->addDays(OvertimeMaximumDays::MAX)->lt(Carbon::now())) {
                return false;
            }

            $attendances = Attendance::where('user_id', Auth::id())
                ->whereDate('time_in', '>=', date($start_date))
                ->whereDate('time_out', '<=', date($end_date))
                ->whereNotNull('time_out')
                ->latest('id')
                ->get();
            $ot = CarbonInterval::fromString($overtime)->divide($quantity);

            foreach ($attendances as $attendance) {
                //skip the loop if there is no schedule detected
                if (!$attendance->timetable_id && !$attendance->shift_id)
                    continue;

                $attendance_ot = CarbonInterval::createFromFormat('H:i', $attendance->over_time);

                if ($attendance_ot->lt($ot)) {
                    return false;
                }
            }

        } elseif ($overtime_type == OvertimeType::ROSTER) {
            $roster_req_hours = $this->getRosterRequiredHours($start_date, $end_date);

            if (!$roster_req_hours) {
                return false;
            }

            if (Carbon::parse($end_date)->addDays(OvertimeMaximumDays::MAX)->lt(Carbon::now())) {
                return false;
            }

            $attendances = Attendance::where('user_id', Auth::id())
                ->whereDate('time_in', '>=', date($start_date))
                ->whereDate('time_out', '<=', date($end_date))
                ->whereNotNull('time_out')
                ->whereHas('shift')
                ->latest('id')
                ->get();

            $total_hours = 0;
            foreach ($attendances as $attendance) {
                if ($attendance->total) {
                    $time = $attendance->total;
                    $parts = explode(':', $time); // Split the time into hours and minutes
                    $hours = (int) $parts[0];
                    $total_hours += $hours;
                }
            }
            if (($overtime + $roster_req_hours) > $total_hours || $roster_req_hours >= $total_hours) {
                return false;
            }
        }

        return true;
    }

    private function getOvertimeOrganization()
    {
        $user = ActualDesignation::where('user_id', Auth::id())->first();

        $overtimeOrganization = $user->unit;
        $overtimeOrganization = $overtimeOrganization ?? $user->section;
        $overtimeOrganization = $overtimeOrganization ?? $user->division;
        $overtimeOrganization = $overtimeOrganization ?? $user->department;
        $overtimeOrganization = $overtimeOrganization ?? $user->office;
        
        return $overtimeOrganization;
    }

    private function getRosterRequiredHours($start_date, $end_date)
    {
        $employee_req_hours = RosterEmployeeRequiredHours::whereHas('partition', function ($query) use ($start_date, $end_date) {
            $query->where('start_date', $start_date)
                ->where('end_date', $end_date)
                ->whereHas('roster', function ($query) {
                    $query->where('status', Status::ACTIVE);
                });
        })->whereHas('rosterEmployee', function ($query) {
            $query->where('user_id', Auth::id());
        })->first();

        $head_employee_req_hours = RosterHeadRequiredHours::whereHas('partition', function ($query) use ($start_date, $end_date) {
            $query->where('start_date', $start_date)
                ->where('end_date', $end_date)
                ->whereHas('roster', function ($query) {
                    $query->where('status', Status::ACTIVE)
                        ->where('head_nurse_id', Auth::id());
                });
        })->first();

        return $employee_req_hours ? $employee_req_hours->required_hours : ($head_employee_req_hours ? $head_employee_req_hours->required_hours : null);
    }

    public function fetch_user_overtime(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');

        return UserOvertime::when($req->est_mh_needed != '' && $req->est_mh_needed != null, function($query) use($req){
            return $query->where('est_mh_needed', 'LIKE', '%' . $req->est_mh_needed . '%');

        })->when(isset($req->filter_dates) && count($req->filter_dates) > 0, function($query) use ($req) {
            $query->when(count($req->filter_dates) == 1, function($query) use($req){
                $query->whereDate('start_date', $req->filter_dates[0])
                ->orWhereDate('end_date', $req->filter_dates[0]);  
            })
            ->when(count($req->filter_dates) == 2, function($query) use ($req) {
                $query->where(function($query) use($req) {
                    $query->whereDate('start_date', '>=', $req->filter_dates[0])
                        ->whereDate('end_date', '<=', $req->filter_dates[1]);
                });
            });

        })->when($req->status != '' && $req->status != null, function($query) use($req){
            return $query->where('status', $req->status);

        })->whereHas('user', function($query){
            $query->when(Auth::id() != '' && Auth::id() != null, function($query){
                return $query->where('id', Auth::id());
            });
        })->with(['report'])->orderBy('created_at', 'DESC')->paginate($paginate);
    }

    //update accomplishment report
    public function create_report(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.user_overtimes,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }


        DB::beginTransaction();
        try {

            $overtime = UserOvertime::findOrFail($request->id);
            
            $path = 'overtime_accomplishment_reports/'. Auth::id();
            $filePath = Storage::disk('local')->put($path, $request->file('upload_report'));

            OvertimeAccomplishmentReport::create([
                'overtime_id'   =>  $overtime->id,
                'mh_rendered'   =>  $request->mh_rendered,
                'actual_output' =>  $filePath,
                'actual_overtime_work'  =>  $request->overtime_work
            ]);
            DB::commit();
            return response()->json([
                'text' => 'Accomplishment Report has been submitted successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'msg' => $e->getMessage(),
                'line'  =>  $e->getLine()
            ], 500);
        }
    }

        //Fetch Accomplishment Report
        public function fetch_report(Request $req)
        {
            $overtime = OvertimeAccomplishmentReport::where('overtime_id', $req->overtime_id)->with(['overtime'])->first();
    
            return response()->json([
                'text' => 'fetch successful.',
                'data' => $overtime
            ]);
        }

        public function file_download(Request $req)
        {
            try {
                $valid = Validator::make($req->all(), [
                    'path'  =>  'required'
                ]);

                if ($valid->fails()) {
                    return response()->json([
                        'error' => $valid->errors()
                    ], 400);
                }
                return Storage::download($req->path);
            } catch (Exception $e) {

                return response()->json([
                    'errors'    => ['File not found'],
                    'message'   => $e->getMessage(),
                ], 404);
            }
        }
    
        //statement of overtime
        public function fetch_statement_overtime(Request $req)
        {
            # code...
        }

}
