<?php

namespace Suiterus\Adg\Controllers\Approvals\Employee;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Enums\Log\CTOLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Approvals\CTO\COCPoints;
use Suiterus\Adg\Models\Approvals\CTO\CTOApplication;
use Suiterus\Adg\Controllers\Approvals\Services\COCPointService;

class CTOApplicationController extends Controller
{
    use HasCustomLogs;
    public function create_cto_application(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'user_id'   =>    'required|exists:mysql.users,id',
            'points'    => 'required',
            'reason'    =>  'required|string|nullable',
            'start_date'    =>  'required|date',
            'end_date'  =>  'required|date'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $coc = COCPoints::where('user_id', Auth::id())->first();

            $data = CTOApplication::create([
                'user_id' => Auth::id(),
                'coc_points_id' => $coc->id,
                'points' => $request->points,
                'reason' => $request->reason,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'status' => $request->status ? $request->status : 0,
                'created_by' => Auth::id()
            ]);

            $this->logCustomMessage(
                'create_cto_application',
                $data,
                Auth::user()->name . ' Create CTO application',
                $data,
                CTOLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Request CTO Application has been created successfully.',
                'data'  => $request->all()
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : OT-comp-0x01'],
                'message'   =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function cancel_cto_application(Request $request)
    {
        $request->merge([
            'id'  =>  $request->cto_application_id
        ]);
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.cto_applications,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $record = CTOApplication::findOrFail($request->id);

            if (!(Auth::user()->hasModules('CTO Applications') || Auth::id() == $record->user_id)) {
                return response()->json([
                    'errors'    => ['You are not allowed to cancel this CTO application.'],
                ], 403);
            }

            $record->update([
                'status'    =>  3, //cancelled by employee
                'status_remarks' => $request->status_remarks,
                'updated_by'    =>  Auth::id()
            ]);

            $record->save();

            $this->logCustomMessage(
                'cancel_cto_application',
                $record,
                Auth::user()->name . ' cancel CTO application',
                $record,
                CTOLogType::CANCEL,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s request has been cancelled."
            ]);
        } catch (ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be approved.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function fetch_applications(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        return CTOApplication::whereHas('user', function ($query) {
            $query->whereId(Auth::id());
        })->paginate($paginate);
    }

    public function fetch_points(Request $request)
    {
        return COCPoints::whereHas('user', function ($query) {
            $query->whereId(Auth::id());
        })->first();
    }
}
