<?php

namespace Suiterus\Adg\Controllers\Approvals;

use App\Enums\Log\CTOLogType;
use App\Http\Classes\NotificationMessage;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Controllers\Approvals\Services\COCPointService;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Approvals\CTO\COCPoints;
use Suiterus\Adg\Models\Approvals\CTO\CTOApplication;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterLegend;

class CTOController extends Controller
{   
    use HasCustomLogs;
    // initialize list
    public function fetch_cto_applications(Request $request){
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        try {

            $cto = CTOApplication::when( count($request->employees) > 0, function($query) use ($request) {
                    $query->whereIn('user_id', $request->employees);
                })->when($request->position != null, function($query) use ($request) {
                    $query->whereHas('user.employeeMetaInfo', function($query) use ($request) {
                        $query->where('position_id', $request->position);
                    });
                })->when($request->department != null, function($query) use ($request) {
                    $query->whereHas('user.employeeMetaInfo', function($query) use ($request) {
                        $query->where('department_id', $request->department);
                    });
                })->when(isset($request->filter_dates) && count($request->filter_dates) > 0, function($query) use ($request) {
                    $query->when(count($request->filter_dates) == 1, function($query) use ($request){
                        $query->whereDate('start_date', '<=', $request->filter_dates[0])
                        ->whereDate('end_date', '>=', $request->filter_dates[0]);  
                    })->when(count($request->filter_dates) == 2, function($query) use ($request) {
                        $query->where(function($query) use ($request) {
                            $query->whereDate('start_date', '<=', $request->filter_dates[0])
                                ->whereDate('end_date', '>=', $request->filter_dates[1]);
                        })->orWhere(function($query) use ($request) {
                            $query->whereDate('start_date', '<=', $request->filter_dates[1])
                                ->whereDate('end_date', '>=', $request->filter_dates[0]);
                        });
                    });
                })->when($request->status != null, function($query) use ($request) {
                    $query->where('status', $request->status);
                })->whereHas('user', function($query) use ($request) {
                    $keyword = '%' . $request->employee_name . '%';
                    $query->where('name', 'LIKE', $keyword)->when(isset($request->supervisor) && $request->supervisor != 0, function($query) use ($request) {
                        $query->whereHas('user_supervisor.supervisor.user', function($query) use ($request) {
                            $query->whereId($request->supervisor);
                        });
                    });
                })->with(['user'  => function($query) {
                    $query->select('id', 'name')->without(['roles', 'permissions','storage' ])
                        ->with(['user_supervisor' => function($query) {
                            $query->with(['supervisor' => function($query) {
                                $query->with(['user' => function($query) {
                                    $query->without(['roles', 'permissions', 'supervisor', 'storage', 'employeeMetaInfo']);
                                }]);
                            }]);
                        }]);
                }])->orderBy('updated_at', 'desc')->paginate($paginate);

            return response()->json([
                'data'  => $cto
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records'],
                'message'   => $e->getMessage()
            ],500);
        }

    }

    // approve
    public function approve(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.cto_applications,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }

        DB::beginTransaction();
        try {
            $record = CTOApplication::where('status', 0)->findOrFail($request->id);
            $record->update([
                'status'    =>  1,
                'remarks' => 'OR',
                'updated_by'    =>  Auth::id()
            ]);

            $cto = CTOApplication::findOrFail($request->id);

            $cocService = new COCPointService;

            $application = CTOApplication::whereId($request->id)->first();
            $coc_record = COCPoints::where('user_id', $application->user_id)->first();

            $coc_points = $application->points;
            $cocService->deductPoints($record, $coc_points);

            $record->save();
            $this->update_cto_roster_table($record);

            $this->logCustomMessage(
                'approve_cto_application',
                $application,
                Auth::user()->name . ' Approve CTO application',
                $application,
                CTOLogType::APPROVE,
                new Activity()
            );

            NotificationMessage::notifySenderAndReceiver('Approved CTO', $record->user_id, Auth::id());

            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s request has been approved."
            ]);

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be approved.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in approving the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    // disapprove
    public function disapprove(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.cto_applications,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);    
        }
        
        DB::beginTransaction();
        try {
            $record = CTOApplication::where('status', 0)->findOrFail($request->id);

            $record->update([
                'status'    =>  2,
                'remarks'   =>  $request->remarks,
                'updated_by'    =>  Auth::id()
            ]); 

            $record->save();

            $this->logCustomMessage(
                'disapprove_cto_application',
                $record,
                Auth::user()->name . ' Disapprove CTO application',
                $record,
                CTOLogType::DISAPPROVE,
                new Activity()
            );

            NotificationMessage::notifySenderAndReceiver('Disapproved CTO', $record->user_id, Auth::id());
            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s request has been disapproved."
            ]);

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['Only pending records can be disapprove.'],
                'message'   => $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in disapproving the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function fetch_application(Request $request) {
        
        return CTOApplication::whereId($request->id)->first();

    }

    public function update_cto_roster_table($cto) {

        $roster_head_shift = RosterHeadEmployeeShift::whereHas('roster', function ($query) use ($cto) {
            $query->where('head_nurse_id',$cto->user_id);
        })->whereHas('rosterDay', function ($query) use ($cto) {
            $query->whereBetween('date', [$cto->start_date, $cto->end_date]);
        })->get();
    
        $roster_emp_shift = RosterDayEmployeeShift::whereHas('rosterEmployeePerGroup', function ($query) use ($cto) {
                $query->where('user_id',$cto->user_id);
            })->whereHas('rosterDay', function ($query) use ($cto) {
                $query->whereBetween('date', [$cto->start_date, $cto->end_date]);
            })->get();
        
        if ($roster_emp_shift->isNotEmpty()) {
            $roster_emp_shift->toQuery()->update([
                'remarks' => $cto->remarks, 
                'roster_legend_id' => null
            ]);
        }
        if ($roster_head_shift->isNotEmpty()) {
            $roster_head_shift->toQuery()->update([
                'remarks' => $cto->remarks, 
                'roster_legend_id' => null
            ]);
        }
    }

}
