<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use Validator;
use DB;
use Exception;
use Suiterus\Adg\Models\Approvals\CTO\COCPointsEarning;
use Suiterus\Adg\Models\Approvals\CTO\COCPointsDeduction;


class COCPointsController extends Controller
{
    public function fetch_cto_users(Request $request) {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');
        
        try {

            $users = User::whereHas('employeeMetaInfo', function($query) use ($request) {
                    $query->when(isset($request->position) && $request->position != null, function($query) use ($request) {
                        $query->whereHas('position', function($query) use ($request) {
                            $query->whereId($request->position);
                        });
                    })
                    ->when(isset($request->department) && $request->department != null, function($query) use ($request) {
                        $query->whereHas('department', function($query) use ($request) {
                            $query->whereId($request->department);
                        });
                    });
                })
                ->when(isset($request->employees) && count($request->employees) > 0, function($query) use ($request) {
                    $query->whereIn('id', $request->employees);
                })
                ->when(isset($request->supervisor) && $request->supervisor != null, function($query) use ($request) {
                    $query->whereHas('user_supervisor.supervisor.user', function($query) use ($request) {
                        $query->whereId($request->supervisor);
                    });
                })
                ->where('name' , 'LIKE', '%' . $request->employee_name . '%')
                ->with(['user_supervisor' => function($query) {
                    $query->with(['supervisor' => function($query) {
                        $query->with(['user' => function($query) {
                            $query->without(['roles', 'permissions', 'supervisor', 'storage', 'employeeMetaInfo']);
                        }]);
                    }]);
                }])->without(['storage', 'permissions', 'roles', 'supervisor'])->paginate($paginate);

            return response()->json([
                'data'  => $users
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the users'],
                'message'   => $e->getMessage()
            ]);
        }
    }

    // Fetch Leave Earnings and Deductions
    public function fetch_points(Request $request) {

        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            $user = User::find($request->id);

            $points = $user->coc_points()->where('user_id', $user->id)->first();

            $fields = ['period', 'points', 'remarks', 'resulting_points', 'created_at','deleted_at'];
            $earning_fields = array_merge($fields, [DB::raw('"coc_points_earnings" as source')]);
            $deduction_fields = array_merge($fields, [DB::raw('"coc_points_deductions" as source')]);
            
            $earnings = COCPointsEarning::select($earning_fields)
                ->where('coc_points_id', $points->id)
                ->where('remarks', 'Points Earned')
                ->whereYear('period', '=', $request->year);

            $leaves = $user->leaves()->pluck('id')->toArray();

            $points = COCPointsDeduction::select($deduction_fields)
            ->where('coc_points_id', $points->id)
                ->where('remarks', 'Points Deducted')->whereYear('period', '=', $request->year)
                ->union($earnings)->orderBy('created_at')->paginate($paginate);
            
            return response()->json([
                'data'  => $points
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ],500);
        }
    }

    public function fetch_remaining_points(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:mysql.users,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $user = User::whereId($request->id)->without([
                'permissions',
                'storage',
                'roles'
            ])->first();

            $coc_points = $user->coc_points()->first();

            return response()->json([
                'data'  => $coc_points
            ]);
            
            
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the data'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }
}
