function flatHooks(configHooks, hooks = {}, parentName) {
  for (const key in configHooks) {
    const subHook = configHooks[key];
    const name = parentName ? `${parentName}:${key}` : key;
    if (typeof subHook === "object" && subHook !== null) {
      flatHooks(subHook, hooks, name);
    } else if (typeof subHook === "function") {
      hooks[name] = subHook;
    }
  }
  return hooks;
}
function mergeHooks(...hooks) {
  const finalHooks = {};
  for (let _hook of hooks) {
    _hook = flatHooks(_hook);
    for (const key in _hook) {
      if (finalHooks[key]) {
        finalHooks[key].push(_hook[key]);
      } else {
        finalHooks[key] = [_hook[key]];
      }
    }
  }
  for (const key in finalHooks) {
    if (finalHooks[key].length > 1) {
      const arr = finalHooks[key];
      finalHooks[key] = (...args) => serial(arr, (fn) => fn(...args));
    } else {
      finalHooks[key] = finalHooks[key][0];
    }
  }
  return finalHooks;
}
function serial(tasks, fn) {
  return tasks.reduce((promise, task) => promise.then(() => fn(task)), Promise.resolve(null));
}

class Hookable {
  constructor(logger = console) {
    this._logger = logger;
    this._hooks = {};
    this._deprecatedHooks = {};
    this.hook = this.hook.bind(this);
    this.callHook = this.callHook.bind(this);
  }
  hook(name, fn) {
    if (!name || typeof fn !== "function") {
      return () => {
      };
    }
    const originalName = name;
    let deprecatedHook;
    while (this._deprecatedHooks[name]) {
      deprecatedHook = this._deprecatedHooks[name];
      if (typeof deprecatedHook === "string") {
        deprecatedHook = {to: deprecatedHook};
      }
      name = deprecatedHook.to;
    }
    if (deprecatedHook && this._logger) {
      if (!deprecatedHook.message) {
        this._logger.warn(`${originalName} hook has been deprecated` + (deprecatedHook.to ? `, please use ${deprecatedHook.to}` : ""));
      } else {
        this._logger.warn(deprecatedHook.message);
      }
    }
    this._hooks[name] = this._hooks[name] || [];
    this._hooks[name].push(fn);
    return () => {
      if (fn) {
        this.removeHook(name, fn);
        fn = null;
      }
    };
  }
  hookOnce(name, fn) {
    let _unreg;
    let _fn = (...args) => {
      _unreg();
      _unreg = null;
      _fn = null;
      return fn(...args);
    };
    _unreg = this.hook(name, _fn);
    return _unreg;
  }
  removeHook(name, fn) {
    if (this._hooks[name]) {
      const idx = this._hooks[name].indexOf(fn);
      if (idx !== -1) {
        this._hooks[name].splice(idx, 1);
      }
      if (this._hooks[name].length === 0) {
        delete this._hooks[name];
      }
    }
  }
  deprecateHook(name, deprecated) {
    this._deprecatedHooks[name] = deprecated;
  }
  deprecateHooks(deprecatedHooks) {
    Object.assign(this._deprecatedHooks, deprecatedHooks);
  }
  addHooks(configHooks) {
    const hooks = flatHooks(configHooks);
    const removeFns = Object.keys(hooks).map((key) => this.hook(key, hooks[key]));
    return () => {
      removeFns.splice(0, removeFns.length).forEach((unreg) => unreg());
    };
  }
  removeHooks(configHooks) {
    const hooks = flatHooks(configHooks);
    for (const key in hooks) {
      this.removeHook(key, hooks[key]);
    }
  }
  async callHook(name, ...args) {
    if (!this._hooks[name]) {
      return;
    }
    try {
      await serial(this._hooks[name], (fn) => fn(...args));
    } catch (err) {
      if (name !== "error") {
        await this.callHook("error", err);
      }
      if (this._logger) {
        if (this._logger.fatal) {
          this._logger.fatal(err);
        } else {
          this._logger.error(err);
        }
      }
    }
  }
}
Hookable.mergeHooks = mergeHooks;
Hookable.prototype.mergeHooks = mergeHooks;

export default Hookable;
